package ru.yandex.intranet.d.dao.transfers;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.values.PrimitiveValue;
import com.yandex.ydb.table.values.Value;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.dao.AbstractDao;
import ru.yandex.intranet.d.datasource.impl.YdbQuerySource;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.WithTenant;
import ru.yandex.intranet.d.model.transfers.TransferRequestByServiceModel;
import ru.yandex.intranet.d.model.transfers.TransferRequestStatus;

/**
 * Transfer request by service DAO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class TransferRequestByServiceDao
        extends AbstractDao<TransferRequestByServiceModel, TransferRequestByServiceModel.Identity> {

    public TransferRequestByServiceDao(YdbQuerySource ydbQuerySource) {
        super(ydbQuerySource);
    }

    public Mono<Void> deleteOneRetryable(YdbTxSession session,
                                         WithTenant<TransferRequestByServiceModel.Identity> idWithTenant) {
        String query = ydbQuerySource.getQuery(queryKeyPrefix() + ".deleteOne");
        Params params = getIdentityWithTenantParams(idWithTenant.getIdentity(), idWithTenant.getTenantId());
        return session.executeDataQueryRetryable(query, params).then();
    }

    public Mono<Void> deleteManyRetryable(YdbTxSession session,
                                          List<WithTenant<TransferRequestByServiceModel.Identity>> ids) {
        if (ids.isEmpty()) {
            return Mono.empty();
        }
        String query = ydbQuerySource.getQuery(queryKeyPrefix() + ".deleteMany");
        Params params = Params.of("$ids", toWithTenantsListValue(ids));
        return session.executeDataQueryRetryable(query, params).then();
    }

    @Override
    protected WithTenant<TransferRequestByServiceModel.Identity> getIdentityWithTenant(
            TransferRequestByServiceModel model) {
        return new WithTenant<>(model.getTenantId(), model.getIdentity());
    }

    @Override
    protected Params getIdentityParams(TransferRequestByServiceModel.Identity id) {
        return Params.create()
                .put("$service_id", PrimitiveValue.int64(id.getServiceId()))
                .put("$status", PrimitiveValue.utf8(id.getStatus().name()))
                .put("$created_at", PrimitiveValue.timestamp(id.getCreatedAt()))
                .put("$transfer_request_id", PrimitiveValue.utf8(id.getTransferRequestId()));
    }

    @Override
    protected Map<String, Value> prepareFieldValues(TransferRequestByServiceModel model) {
        HashMap<String, Value> fields = new HashMap<>();
        fields.put(Fields.TENANT_ID.field(), PrimitiveValue.utf8(model.getTenantId().getId()));
        fields.put(Fields.SERVICE_ID.field(), PrimitiveValue.int64(model.getServiceId()));
        fields.put(Fields.STATUS.field(), PrimitiveValue.utf8(model.getStatus().name()));
        fields.put(Fields.CREATED_AT.field(), PrimitiveValue.timestamp(model.getCreatedAt()));
        fields.put(Fields.TRANSFER_REQUEST_ID.field(), PrimitiveValue.utf8(model.getTransferRequestId()));
        return fields;
    }

    @Override
    protected TransferRequestByServiceModel readOneRow(ResultSetReader reader, Map<String, TenantId> tenantIdCache) {
        return TransferRequestByServiceModel.builder()
                .tenantId(tenantIdCache.computeIfAbsent(reader
                        .getColumn(Fields.TENANT_ID.field()).getUtf8(), TenantId::new))
                .serviceId(reader.getColumn(Fields.SERVICE_ID.field()).getInt64())
                .status(TransferRequestStatus.valueOf(reader.getColumn(Fields.STATUS.field()).getUtf8()))
                .createdAt(reader.getColumn(Fields.CREATED_AT.field()).getTimestamp())
                .transferRequestId(reader.getColumn(Fields.TRANSFER_REQUEST_ID.field()).getUtf8())
                .build();
    }

    @Override
    protected String queryKeyPrefix() {
        return "yql.queries.transferRequestByService";
    }

    public enum Fields {

        TENANT_ID,
        SERVICE_ID,
        STATUS,
        CREATED_AT,
        TRANSFER_REQUEST_ID;

        public String field() {
            return name().toLowerCase();
        }

    }

}
