package ru.yandex.intranet.d.dao.transfers;

import java.time.Instant;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.fasterxml.jackson.core.type.TypeReference;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.values.PrimitiveValue;
import com.yandex.ydb.table.values.Value;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.dao.AbstractDao;
import ru.yandex.intranet.d.dao.JsonFieldHelper;
import ru.yandex.intranet.d.datasource.Ydb;
import ru.yandex.intranet.d.datasource.impl.YdbQuerySource;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.WithTenant;
import ru.yandex.intranet.d.model.transfers.LoanMeta;
import ru.yandex.intranet.d.model.transfers.TransferApplicationDetails;
import ru.yandex.intranet.d.model.transfers.TransferParameters;
import ru.yandex.intranet.d.model.transfers.TransferRequestEventType;
import ru.yandex.intranet.d.model.transfers.TransferRequestHistoryFields;
import ru.yandex.intranet.d.model.transfers.TransferRequestHistoryModel;
import ru.yandex.intranet.d.model.transfers.TransferResponsible;
import ru.yandex.intranet.d.model.transfers.TransferVotes;
import ru.yandex.intranet.d.util.ObjectMapperHolder;

/**
 * Transfer request history DAO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class TransferRequestHistoryDao
        extends AbstractDao<TransferRequestHistoryModel, TransferRequestHistoryModel.Identity> {

    private final JsonFieldHelper<TransferRequestHistoryFields> requestFieldsHelper;
    private final JsonFieldHelper<TransferParameters> transferParametersHelper;
    private final JsonFieldHelper<TransferResponsible> transferResponsibleHelper;
    private final JsonFieldHelper<TransferVotes> transferVotesHelper;
    private final JsonFieldHelper<TransferApplicationDetails> transferApplicationDetailsHelper;
    private final JsonFieldHelper<LoanMeta> loanMetaHelper;

    public TransferRequestHistoryDao(YdbQuerySource ydbQuerySource,
                                     @Qualifier("ydbJsonObjectMapper") ObjectMapperHolder objectMapper) {
        super(ydbQuerySource);
        this.requestFieldsHelper = new JsonFieldHelper<>(objectMapper,
                new TypeReference<TransferRequestHistoryFields>() { });
        this.transferParametersHelper = new JsonFieldHelper<>(objectMapper,
                new TypeReference<TransferParameters>() { });
        this.transferResponsibleHelper = new JsonFieldHelper<>(objectMapper,
                new TypeReference<TransferResponsible>() { });
        this.transferVotesHelper = new JsonFieldHelper<>(objectMapper,
                new TypeReference<TransferVotes>() { });
        this.transferApplicationDetailsHelper = new JsonFieldHelper<>(objectMapper,
                new TypeReference<TransferApplicationDetails>() { });
        this.loanMetaHelper = new JsonFieldHelper<>(objectMapper,
                new TypeReference<LoanMeta>() { });
    }

    public Mono<List<TransferRequestHistoryModel>> findFirstPage(YdbTxSession session, String transferRequestId,
                                                                 TenantId tenantId,
                                                                 long limit) {

        String query = ydbQuerySource.getQuery(queryKeyPrefix() + ".getFirstPage");
        Params params = Params.of("$tenant_id", PrimitiveValue.utf8(tenantId.getId()),
                "$transfer_request_id", PrimitiveValue.utf8(transferRequestId),
                "$limit", PrimitiveValue.uint64(limit));
        return session.executeDataQueryRetryable(query, params).map(this::toModels);
    }

    public Mono<List<TransferRequestHistoryModel>> findNextPage(YdbTxSession session, String transferRequestId,
                                                                TenantId tenantId, Instant fromTimestamp,
                                                                String fromId, long limit) {
        String query = ydbQuerySource.getQuery(queryKeyPrefix() + ".getNextPage");
        Params params = Params.of("$tenant_id", PrimitiveValue.utf8(tenantId.getId()),
                "$transfer_request_id", PrimitiveValue.utf8(transferRequestId),
                "$limit", PrimitiveValue.uint64(limit),
                "$from_event_timestamp", PrimitiveValue.timestamp(fromTimestamp),
                "$from_id", PrimitiveValue.utf8(fromId));
        return session.executeDataQueryRetryable(query, params).map(this::toModels);
    }

    @Override
    protected WithTenant<TransferRequestHistoryModel.Identity> getIdentityWithTenant(
            TransferRequestHistoryModel model) {
        return new WithTenant<>(model.getTenantId(), model.getIdentity());
    }

    @Override
    protected Params getIdentityParams(TransferRequestHistoryModel.Identity id) {
        return Params.create()
                .put("$id", PrimitiveValue.utf8(id.getId()))
                .put("$transfer_request_id", PrimitiveValue.utf8(id.getTransferRequestId()))
                .put("$event_timestamp", PrimitiveValue.timestamp(id.getTimestamp()));
    }

    @Override
    protected Map<String, Value> prepareFieldValues(TransferRequestHistoryModel model) {
        HashMap<String, Value> fields = new HashMap<>();
        fields.put(Fields.ID.field(), PrimitiveValue.utf8(model.getId()));
        fields.put(Fields.TENANT_ID.field(), PrimitiveValue.utf8(model.getTenantId().getId()));
        fields.put(Fields.TRANSFER_REQUEST_ID.field(), PrimitiveValue.utf8(model.getTransferRequestId()));
        fields.put(Fields.EVENT_TYPE.field(), PrimitiveValue.utf8(model.getType().name()));
        fields.put(Fields.EVENT_TIMESTAMP.field(), PrimitiveValue.timestamp(model.getTimestamp()));
        fields.put(Fields.AUTHOR_ID.field(), Ydb.nullableUtf8(model.getAuthorId().orElse(null)));
        fields.put(Fields.OLD_FIELDS.field(), requestFieldsHelper.writeOptional(model.getOldFields().orElse(null)));
        fields.put(Fields.NEW_FIELDS.field(), requestFieldsHelper.writeOptional(model.getNewFields().orElse(null)));
        fields.put(Fields.OLD_TRANSFER.field(), transferParametersHelper
                .writeOptional(model.getOldParameters().orElse(null)));
        fields.put(Fields.OLD_RESPONSIBLE.field(), transferResponsibleHelper
                .writeOptional(model.getOldResponsible().orElse(null)));
        fields.put(Fields.OLD_VOTES.field(), transferVotesHelper.writeOptional(model.getOldVotes().orElse(null)));
        fields.put(Fields.OLD_APPLICATION_DETAILS.field(), transferApplicationDetailsHelper
                .writeOptional(model.getOldApplicationDetails().orElse(null)));
        fields.put(Fields.NEW_TRANSFER.field(), transferParametersHelper
                .writeOptional(model.getNewParameters().orElse(null)));
        fields.put(Fields.NEW_RESPONSIBLE.field(), transferResponsibleHelper
                .writeOptional(model.getNewResponsible().orElse(null)));
        fields.put(Fields.NEW_VOTES.field(), transferVotesHelper.writeOptional(model.getNewVotes().orElse(null)));
        fields.put(Fields.NEW_APPLICATION_DETAILS.field(), transferApplicationDetailsHelper
                .writeOptional(model.getNewApplicationDetails().orElse(null)));
        fields.put(Fields.ORDER.field(), PrimitiveValue.int64(model.getOrder()));
        fields.put(Fields.OLD_LOAN_META.field(), loanMetaHelper.writeOptional(model.getOldLoanMeta().orElse(null)));
        fields.put(Fields.NEW_LOAN_META.field(), loanMetaHelper.writeOptional(model.getNewLoanMeta().orElse(null)));
        return fields;
    }

    @Override
    protected TransferRequestHistoryModel readOneRow(ResultSetReader reader, Map<String, TenantId> tenantIdCache) {
        return TransferRequestHistoryModel.builder()
                .id(reader.getColumn(Fields.ID.field()).getUtf8())
                .tenantId(tenantIdCache.computeIfAbsent(reader
                        .getColumn(Fields.TENANT_ID.field()).getUtf8(), TenantId::new))
                .transferRequestId(reader.getColumn(Fields.TRANSFER_REQUEST_ID.field()).getUtf8())
                .type(TransferRequestEventType.valueOf(reader.getColumn(Fields.EVENT_TYPE.field()).getUtf8()))
                .timestamp(reader.getColumn(Fields.EVENT_TIMESTAMP.field()).getTimestamp())
                .authorId(Ydb.utf8OrNull(reader.getColumn(Fields.AUTHOR_ID.field())))
                .oldFields(requestFieldsHelper.read(reader.getColumn(Fields.OLD_FIELDS.field())))
                .newFields(requestFieldsHelper.read(reader.getColumn(Fields.NEW_FIELDS.field())))
                .oldParameters(transferParametersHelper.read(reader.getColumn(Fields.OLD_TRANSFER.field())))
                .oldResponsible(transferResponsibleHelper.read(reader.getColumn(Fields.OLD_RESPONSIBLE.field())))
                .oldVotes(transferVotesHelper.read(reader.getColumn(Fields.OLD_VOTES.field())))
                .oldApplicationDetails(transferApplicationDetailsHelper.read(reader
                        .getColumn(Fields.OLD_APPLICATION_DETAILS.field())))
                .newParameters(transferParametersHelper.read(reader.getColumn(Fields.NEW_TRANSFER.field())))
                .newResponsible(transferResponsibleHelper.read(reader.getColumn(Fields.NEW_RESPONSIBLE.field())))
                .newVotes(transferVotesHelper.read(reader.getColumn(Fields.NEW_VOTES.field())))
                .newApplicationDetails(transferApplicationDetailsHelper.read(reader
                        .getColumn(Fields.NEW_APPLICATION_DETAILS.field())))
                .order(reader.getColumn(Fields.ORDER.field()).getInt64())
                .oldLoanMeta(loanMetaHelper.read(reader.getColumn(Fields.OLD_LOAN_META.field())))
                .newLoanMeta(loanMetaHelper.read(reader.getColumn(Fields.NEW_LOAN_META.field())))
                .build();
    }

    @Override
    protected String queryKeyPrefix() {
        return "yql.queries.transferRequestHistory";
    }

    public enum Fields {

        ID,
        TENANT_ID,
        TRANSFER_REQUEST_ID,
        EVENT_TYPE,
        EVENT_TIMESTAMP,
        AUTHOR_ID,
        OLD_FIELDS,
        NEW_FIELDS,
        OLD_TRANSFER,
        OLD_RESPONSIBLE,
        OLD_VOTES,
        OLD_APPLICATION_DETAILS,
        NEW_TRANSFER,
        NEW_RESPONSIBLE,
        NEW_VOTES,
        NEW_APPLICATION_DETAILS,
        ORDER,
        OLD_LOAN_META,
        NEW_LOAN_META;

        public String field() {
            return name().toLowerCase();
        }

    }

}
