package ru.yandex.intranet.d.dao.transfers;

import java.time.Instant;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import com.fasterxml.jackson.core.type.TypeReference;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.values.ListValue;
import com.yandex.ydb.table.values.PrimitiveValue;
import com.yandex.ydb.table.values.TupleValue;
import com.yandex.ydb.table.values.Value;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.dao.AbstractDao;
import ru.yandex.intranet.d.dao.JsonFieldHelper;
import ru.yandex.intranet.d.datasource.Ydb;
import ru.yandex.intranet.d.datasource.impl.YdbQuerySource;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.WithTenant;
import ru.yandex.intranet.d.model.transfers.LoanMeta;
import ru.yandex.intranet.d.model.transfers.TransferApplicationDetails;
import ru.yandex.intranet.d.model.transfers.TransferNotified;
import ru.yandex.intranet.d.model.transfers.TransferParameters;
import ru.yandex.intranet.d.model.transfers.TransferRequestModel;
import ru.yandex.intranet.d.model.transfers.TransferRequestStatus;
import ru.yandex.intranet.d.model.transfers.TransferRequestSubtype;
import ru.yandex.intranet.d.model.transfers.TransferRequestType;
import ru.yandex.intranet.d.model.transfers.TransferResponsible;
import ru.yandex.intranet.d.model.transfers.TransferVotes;
import ru.yandex.intranet.d.util.ObjectMapperHolder;
import ru.yandex.intranet.d.util.result.Result;

/**
 * Transfer requests DAO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class TransferRequestsDao extends AbstractDao<TransferRequestModel, String> {

    private final JsonFieldHelper<TransferParameters> transferParametersHelper;
    private final JsonFieldHelper<TransferResponsible> transferResponsibleHelper;
    private final JsonFieldHelper<TransferVotes> transferVotesHelper;
    private final JsonFieldHelper<TransferApplicationDetails> transferApplicationDetailsHelper;
    private final JsonFieldHelper<TransferNotified> transferNotifiedHelper;
    private final JsonFieldHelper<LoanMeta> loanMetaHelper;

    public TransferRequestsDao(YdbQuerySource ydbQuerySource,
                               @Qualifier("ydbJsonObjectMapper") ObjectMapperHolder objectMapper) {
        super(ydbQuerySource);
        this.transferParametersHelper = new JsonFieldHelper<>(objectMapper,
                new TypeReference<>() { });
        this.transferResponsibleHelper = new JsonFieldHelper<>(objectMapper,
                new TypeReference<>() { });
        this.transferVotesHelper = new JsonFieldHelper<>(objectMapper,
                new TypeReference<>() { });
        this.transferApplicationDetailsHelper = new JsonFieldHelper<>(objectMapper,
                new TypeReference<>() { });
        transferNotifiedHelper = new JsonFieldHelper<>(objectMapper,
                new TypeReference<>() { });
        this.loanMetaHelper = new JsonFieldHelper<>(objectMapper,
                new TypeReference<>() { });
    }

    public Mono<List<TransferRequestModel>> findByResponsibleFirstPage(YdbTxSession session, String responsibleId,
                                                                       Set<TransferRequestStatus> statuses,
                                                                       TenantId tenantId,
                                                                       long limit) {
        Set<TransferRequestStatus> statusesToFilter = statuses.isEmpty()
                ? EnumSet.allOf(TransferRequestStatus.class) : statuses;
        Map<String, Value<?>> paramsMap = new HashMap<>();
        String query;
        if (statusesToFilter.size() != 1) {
            query = ydbQuerySource.getQuery(queryKeyPrefix() + ".getByResponsibleFirstPage");
            ListValue statusesParam = ListValue.of(statusesToFilter.stream().map(v -> PrimitiveValue.utf8(v.name()))
                    .toArray(PrimitiveValue[]::new));
            paramsMap.put("$statuses", statusesParam);
        } else {
            query = ydbQuerySource.getQuery(queryKeyPrefix() + ".getByResponsibleSingleStatusFirstPage");
            paramsMap.put("$status", PrimitiveValue.utf8(statusesToFilter.iterator().next().name()));
        }
        paramsMap.put("$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
        paramsMap.put("$responsible_id", PrimitiveValue.utf8(responsibleId));
        paramsMap.put("$limit", PrimitiveValue.uint64(limit));
        Params params = Params.copyOf(paramsMap);
        return session.executeDataQueryRetryable(query, params).map(this::toModels);
    }

    public Mono<List<TransferRequestModel>> findByResponsibleNextPage(YdbTxSession session, String responsibleId,
                                                                      Set<TransferRequestStatus> statuses,
                                                                      TenantId tenantId,
                                                                      Instant fromCreatedAt,
                                                                      String fromId,
                                                                      long limit) {
        Set<TransferRequestStatus> statusesToFilter = statuses.isEmpty()
                ? EnumSet.allOf(TransferRequestStatus.class) : statuses;
        Map<String, Value<?>> paramsMap = new HashMap<>();
        String query;
        if (statusesToFilter.size() != 1) {
            query = ydbQuerySource.getQuery(queryKeyPrefix() + ".getByResponsibleNextPage");
            ListValue statusesParam = ListValue.of(statusesToFilter.stream().map(v -> PrimitiveValue.utf8(v.name()))
                    .toArray(PrimitiveValue[]::new));
            paramsMap.put("$statuses", statusesParam);
        } else {
            query = ydbQuerySource.getQuery(queryKeyPrefix() + ".getByResponsibleSingleStatusNextPage");
            paramsMap.put("$status", PrimitiveValue.utf8(statusesToFilter.iterator().next().name()));
        }
        paramsMap.put("$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
        paramsMap.put("$responsible_id", PrimitiveValue.utf8(responsibleId));
        paramsMap.put("$limit", PrimitiveValue.uint64(limit));
        paramsMap.put("$from_created_at", PrimitiveValue.timestamp(fromCreatedAt));
        paramsMap.put("$from_id", PrimitiveValue.utf8(fromId));
        Params params = Params.copyOf(paramsMap);
        return session.executeDataQueryRetryable(query, params).map(this::toModels);
    }

    public Mono<List<TransferRequestModel>> findByFolderFirstPage(YdbTxSession session, String folderId,
                                                                  Set<TransferRequestStatus> statuses,
                                                                  TenantId tenantId,
                                                                  long limit) {
        Set<TransferRequestStatus> statusesToFilter = statuses.isEmpty()
                ? EnumSet.allOf(TransferRequestStatus.class) : statuses;
        Map<String, Value<?>> paramsMap = new HashMap<>();
        String query;
        if (statusesToFilter.size() != 1) {
            query = ydbQuerySource.getQuery(queryKeyPrefix() + ".getByFolderFirstPage");
            ListValue statusesParam = ListValue.of(statusesToFilter.stream().map(v -> PrimitiveValue.utf8(v.name()))
                    .toArray(PrimitiveValue[]::new));
            paramsMap.put("$statuses", statusesParam);
        } else {
            query = ydbQuerySource.getQuery(queryKeyPrefix() + ".getByFolderSingleStatusFirstPage");
            paramsMap.put("$status", PrimitiveValue.utf8(statusesToFilter.iterator().next().name()));
        }
        paramsMap.put("$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
        paramsMap.put("$folder_id", PrimitiveValue.utf8(folderId));
        paramsMap.put("$limit", PrimitiveValue.uint64(limit));
        Params params = Params.copyOf(paramsMap);
        return session.executeDataQueryRetryable(query, params).map(this::toModels);
    }

    public Mono<List<TransferRequestModel>> findByFolderNextPage(YdbTxSession session, String folderId,
                                                                 Set<TransferRequestStatus> statuses,
                                                                 TenantId tenantId,
                                                                 Instant fromCreatedAt,
                                                                 String fromId,
                                                                 long limit) {
        Set<TransferRequestStatus> statusesToFilter = statuses.isEmpty()
                ? EnumSet.allOf(TransferRequestStatus.class) : statuses;
        Map<String, Value<?>> paramsMap = new HashMap<>();
        String query;
        if (statusesToFilter.size() != 1) {
            query = ydbQuerySource.getQuery(queryKeyPrefix() + ".getByFolderNextPage");
            ListValue statusesParam = ListValue.of(statusesToFilter.stream().map(v -> PrimitiveValue.utf8(v.name()))
                    .toArray(PrimitiveValue[]::new));
            paramsMap.put("$statuses", statusesParam);
        } else {
            query = ydbQuerySource.getQuery(queryKeyPrefix() + ".getByFolderSingleStatusNextPage");
            paramsMap.put("$status", PrimitiveValue.utf8(statusesToFilter.iterator().next().name()));
        }
        paramsMap.put("$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
        paramsMap.put("$folder_id", PrimitiveValue.utf8(folderId));
        paramsMap.put("$limit", PrimitiveValue.uint64(limit));
        paramsMap.put("$from_created_at", PrimitiveValue.timestamp(fromCreatedAt));
        paramsMap.put("$from_id", PrimitiveValue.utf8(fromId));
        Params params = Params.copyOf(paramsMap);
        return session.executeDataQueryRetryable(query, params).map(this::toModels);
    }

    public Mono<List<TransferRequestModel>> findByServiceFirstPage(YdbTxSession session, long serviceId,
                                                                   Set<TransferRequestStatus> statuses,
                                                                   TenantId tenantId,
                                                                   long limit) {
        Set<TransferRequestStatus> statusesToFilter = statuses.isEmpty()
                ? EnumSet.allOf(TransferRequestStatus.class) : statuses;
        Map<String, Value<?>> paramsMap = new HashMap<>();
        String query;
        if (statusesToFilter.size() != 1) {
            query = ydbQuerySource.getQuery(queryKeyPrefix() + ".getByServiceFirstPage");
            ListValue statusesParam = ListValue.of(statusesToFilter.stream().map(v -> PrimitiveValue.utf8(v.name()))
                    .toArray(PrimitiveValue[]::new));
            paramsMap.put("$statuses", statusesParam);
        } else {
            query = ydbQuerySource.getQuery(queryKeyPrefix() + ".getByServiceSingleStatusFirstPage");
            paramsMap.put("$status", PrimitiveValue.utf8(statusesToFilter.iterator().next().name()));
        }
        paramsMap.put("$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
        paramsMap.put("$service_id", PrimitiveValue.int64(serviceId));
        paramsMap.put("$limit", PrimitiveValue.uint64(limit));
        Params params = Params.copyOf(paramsMap);
        return session.executeDataQueryRetryable(query, params).map(this::toModels);
    }

    public Mono<List<TransferRequestModel>> findByServiceNextPage(YdbTxSession session, long serviceId,
                                                                  Set<TransferRequestStatus> statuses,
                                                                  TenantId tenantId,
                                                                  Instant fromCreatedAt,
                                                                  String fromId,
                                                                  long limit) {
        Set<TransferRequestStatus> statusesToFilter = statuses.isEmpty()
                ? EnumSet.allOf(TransferRequestStatus.class) : statuses;
        Map<String, Value<?>> paramsMap = new HashMap<>();
        String query;
        if (statusesToFilter.size() != 1) {
            query = ydbQuerySource.getQuery(queryKeyPrefix() + ".getByServiceNextPage");
            ListValue statusesParam = ListValue.of(statusesToFilter.stream().map(v -> PrimitiveValue.utf8(v.name()))
                    .toArray(PrimitiveValue[]::new));
            paramsMap.put("$statuses", statusesParam);
        } else {
            query = ydbQuerySource.getQuery(queryKeyPrefix() + ".getByServiceSingleStatusNextPage");
            paramsMap.put("$status", PrimitiveValue.utf8(statusesToFilter.iterator().next().name()));
        }
        paramsMap.put("$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
        paramsMap.put("$service_id", PrimitiveValue.int64(serviceId));
        paramsMap.put("$limit", PrimitiveValue.uint64(limit));
        paramsMap.put("$from_created_at", PrimitiveValue.timestamp(fromCreatedAt));
        paramsMap.put("$from_id", PrimitiveValue.utf8(fromId));
        Params params = Params.copyOf(paramsMap);
        return session.executeDataQueryRetryable(query, params).map(this::toModels);
    }

    public Mono<List<TransferRequestModel>> getByServicesAndDate(YdbTxSession session, List<Long> serviceIds,
                                                                 TenantId tenantId,
                                                                 Instant dateFrom, Instant dateTo) {
        Map<String, Value<?>> paramsMap = new HashMap<>();
        String query = ydbQuerySource.getQuery(queryKeyPrefix() + ".getByServicesAndDate");
        ListValue idsParam = ListValue.of(serviceIds.stream()
                .map(id -> TupleValue.of(PrimitiveValue.int64(id),
                        PrimitiveValue.utf8(tenantId.getId()))).toArray(TupleValue[]::new));
        paramsMap.put("$ids", idsParam);
        paramsMap.put("$date_from", PrimitiveValue.timestamp(dateFrom));
        paramsMap.put("$date_to", PrimitiveValue.timestamp(dateTo));
        Params params = Params.copyOf(paramsMap);
        return session.executeDataQueryRetryable(query, params).map(this::toModels);
    }

    public Mono<Result<Void>> updateTrackerIssueKeyRetryable(YdbTxSession session, String transferId, TenantId tenantId,
                                                             String trackerIssueKey) {
        String query = ydbQuerySource.getQuery(queryKeyPrefix() + ".updateTrackerIssuesKey");
        Params params = Params.of("$tenant_id", PrimitiveValue.utf8(tenantId.getId()),
                "$id", PrimitiveValue.utf8(transferId),
                "$tracker_issue_key", PrimitiveValue.utf8(trackerIssueKey));
        return session.executeDataQueryRetryable(query, params).thenReturn(Result.success(null));
    }

    @Override
    protected WithTenant<String> getIdentityWithTenant(TransferRequestModel model) {
        return new WithTenant<>(model.getTenantId(), model.getId());
    }

    @Override
    protected Params getIdentityParams(String id) {
        return Params.create()
                .put("$id", PrimitiveValue.utf8(id));
    }

    @Override
    protected Map<String, Value> prepareFieldValues(TransferRequestModel model) {
        HashMap<String, Value> fields = new HashMap<>();
        fields.put(Fields.ID.field(), PrimitiveValue.utf8(model.getId()));
        fields.put(Fields.TENANT_ID.field(), PrimitiveValue.utf8(model.getTenantId().getId()));
        fields.put(Fields.VERSION.field(), PrimitiveValue.int64(model.getVersion()));
        fields.put(Fields.SUMMARY.field(), Ydb.nullableUtf8(model.getSummary().orElse(null)));
        fields.put(Fields.DESCRIPTION.field(), Ydb.nullableUtf8(model.getDescription().orElse(null)));
        fields.put(Fields.TRACKER_ISSUE_KEY.field(), Ydb.nullableUtf8(model.getTrackerIssueKey().orElse(null)));
        fields.put(Fields.REQUEST_TYPE.field(), PrimitiveValue.utf8(model.getType().name()));
        fields.put(Fields.REQUEST_SUBTYPE.field(),
                Ydb.nullableUtf8(model.getSubtype().map(Enum::name).orElse(null)));
        fields.put(Fields.STATUS.field(), PrimitiveValue.utf8(model.getStatus().name()));
        fields.put(Fields.CREATED_BY.field(), PrimitiveValue.utf8(model.getCreatedBy()));
        fields.put(Fields.UPDATED_BY.field(), Ydb.nullableUtf8(model.getUpdatedBy().orElse(null)));
        fields.put(Fields.CREATED_AT.field(), PrimitiveValue.timestamp(model.getCreatedAt()));
        fields.put(Fields.UPDATED_AT.field(), Ydb.nullableTimestamp(model.getUpdatedAt().orElse(null)));
        fields.put(Fields.APPLIED_AT.field(), Ydb.nullableTimestamp(model.getAppliedAt().orElse(null)));
        fields.put(Fields.TRANSFER.field(), transferParametersHelper.write(model.getParameters()));
        fields.put(Fields.RESPONSIBLE.field(), transferResponsibleHelper.write(model.getResponsible()));
        fields.put(Fields.VOTES.field(), transferVotesHelper.write(model.getVotes()));
        fields.put(Fields.APPLICATION_DETAILS.field(), transferApplicationDetailsHelper
                .writeOptional(model.getApplicationDetails().orElse(null)));
        fields.put(Fields.NEXT_HISTORY_ORDER.field(), PrimitiveValue.int64(model.getNextHistoryOrder()));
        fields.put(Fields.NOTIFIED.field(), transferNotifiedHelper
                .writeOptional(model.getTransferNotified().orElse(null)));
        fields.put(Fields.LOAN_META.field(), loanMetaHelper.writeOptional(model.getLoanMeta().orElse(null)));
        return fields;
    }

    @Override
    protected TransferRequestModel readOneRow(ResultSetReader reader, Map<String, TenantId> tenantIdCache) {
        return TransferRequestModel.builder()
                .id(reader.getColumn(Fields.ID.field()).getUtf8())
                .tenantId(tenantIdCache.computeIfAbsent(reader
                        .getColumn(Fields.TENANT_ID.field()).getUtf8(), TenantId::new))
                .version(reader.getColumn(Fields.VERSION.field()).getInt64())
                .summary(Ydb.utf8OrNull(reader.getColumn(Fields.SUMMARY.field())))
                .description(Ydb.utf8OrNull(reader.getColumn(Fields.DESCRIPTION.field())))
                .trackerIssueKey(Ydb.utf8OrNull(reader.getColumn(Fields.TRACKER_ISSUE_KEY.field())))
                .type(TransferRequestType.valueOf(reader.getColumn(Fields.REQUEST_TYPE.field()).getUtf8()))
                .subtype(Optional.ofNullable(Ydb.utf8OrNull(reader.getColumn(Fields.REQUEST_SUBTYPE.field())))
                        .map(TransferRequestSubtype::valueOf).orElse(null))
                .status(TransferRequestStatus.valueOf(reader.getColumn(Fields.STATUS.field()).getUtf8()))
                .createdBy(reader.getColumn(Fields.CREATED_BY.field()).getUtf8())
                .updatedBy(Ydb.utf8OrNull(reader.getColumn(Fields.UPDATED_BY.field())))
                .createdAt(reader.getColumn(Fields.CREATED_AT.field()).getTimestamp())
                .updatedAt(Ydb.timestampOrNull(reader.getColumn(Fields.UPDATED_AT.field())))
                .appliedAt(Ydb.timestampOrNull(reader.getColumn(Fields.APPLIED_AT.field())))
                .parameters(transferParametersHelper.read(reader.getColumn(Fields.TRANSFER.field())))
                .responsible(transferResponsibleHelper.read(reader.getColumn(Fields.RESPONSIBLE.field())))
                .votes(transferVotesHelper.read(reader.getColumn(Fields.VOTES.field())))
                .applicationDetails(transferApplicationDetailsHelper.read(reader
                        .getColumn(Fields.APPLICATION_DETAILS.field())))
                .nextHistoryOrder(reader.getColumn(Fields.NEXT_HISTORY_ORDER.field()).getInt64())
                .transferNotified(transferNotifiedHelper.read(reader.getColumn(Fields.NOTIFIED.field())))
                .loanMeta(loanMetaHelper.read(reader.getColumn(Fields.LOAN_META.field())))
                .build();
    }

    @Override
    protected String queryKeyPrefix() {
        return "yql.queries.transferRequests";
    }

    public enum Fields {

        ID,
        TENANT_ID,
        VERSION,
        SUMMARY,
        DESCRIPTION,
        TRACKER_ISSUE_KEY,
        REQUEST_TYPE,
        REQUEST_SUBTYPE,
        STATUS,
        CREATED_BY,
        UPDATED_BY,
        CREATED_AT,
        UPDATED_AT,
        APPLIED_AT,
        TRANSFER,
        RESPONSIBLE,
        VOTES,
        APPLICATION_DETAILS,
        NEXT_HISTORY_ORDER,
        NOTIFIED,
        LOAN_META;

        public String field() {
            return name().toLowerCase();
        }

    }

}
