package ru.yandex.intranet.d.dao.unique

import com.fasterxml.jackson.core.type.TypeReference
import com.yandex.ydb.table.query.Params
import com.yandex.ydb.table.result.ResultSetReader
import com.yandex.ydb.table.values.PrimitiveValue
import kotlinx.coroutines.reactor.awaitSingle
import kotlinx.coroutines.reactor.awaitSingleOrNull
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.stereotype.Component
import ru.yandex.intranet.d.dao.DaoReader
import ru.yandex.intranet.d.dao.JsonFieldHelper
import ru.yandex.intranet.d.datasource.impl.YdbQuerySource
import ru.yandex.intranet.d.datasource.model.YdbTxSession
import ru.yandex.intranet.d.model.TenantId
import ru.yandex.intranet.d.model.uniques.RequestUniqueEndpoint
import ru.yandex.intranet.d.model.uniques.RequestUniqueIdentity
import ru.yandex.intranet.d.model.uniques.RequestUniqueMetadata
import ru.yandex.intranet.d.model.uniques.RequestUniqueModel
import ru.yandex.intranet.d.model.uniques.RequestUniqueSubject
import ru.yandex.intranet.d.util.ObjectMapperHolder

/**
 * Request unique DAO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
class RequestUniqueDao(private val ydbQuerySource: YdbQuerySource,
                       @Qualifier("ydbJsonObjectMapper") private val objectMapper: ObjectMapperHolder) {
    private val metadataFieldHelper: JsonFieldHelper<RequestUniqueMetadata> = JsonFieldHelper(objectMapper,
        object : TypeReference<RequestUniqueMetadata>() {})

    suspend fun getById(session: YdbTxSession, id: RequestUniqueIdentity): RequestUniqueModel? {
        val query = ydbQuerySource.getQuery("yql.queries.requestsUnique.getById")
        val params = toIdentityParams(id)
        return DaoReader.toModel(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toModel)
    }

    suspend fun upsertRetryable(session: YdbTxSession, model: RequestUniqueModel): Void? {
        val query = ydbQuerySource.getQuery("yql.queries.requestsUnique.upsertOne")
        val params = toUpsertParams(model)
        return session.executeDataQueryRetryable(query, params).then().awaitSingleOrNull()
    }

    private fun toModel(reader: ResultSetReader): RequestUniqueModel {
        return RequestUniqueModel(tenantId = TenantId(reader.getColumn("tenant_id").utf8),
            unique = reader.getColumn("unique").utf8,
            subject = RequestUniqueSubject.deserialize(reader.getColumn("subject").utf8),
            endpoint = RequestUniqueEndpoint.valueOf(reader.getColumn("endpoint").utf8),
            createdAt = reader.getColumn("created_at").timestamp,
            metadata = metadataFieldHelper.read(reader.getColumn("metadata"))!!
        )
    }

    private fun toIdentityParams(id: RequestUniqueIdentity): Params {
        return Params.of("\$tenant_id", PrimitiveValue.utf8(id.tenantId.id),
            "\$unique", PrimitiveValue.utf8(id.unique),
            "\$subject", PrimitiveValue.utf8(id.subject.serialize()),
            "\$endpoint", PrimitiveValue.utf8(id.endpoint.name))
    }

    private fun toUpsertParams(model: RequestUniqueModel): Params {
        return Params.of("\$tenant_id", PrimitiveValue.utf8(model.tenantId.id),
            "\$unique", PrimitiveValue.utf8(model.unique),
            "\$subject", PrimitiveValue.utf8(model.subject.serialize()),
            "\$endpoint", PrimitiveValue.utf8(model.endpoint.name),
            "\$created_at", PrimitiveValue.timestamp(model.createdAt),
            "\$metadata", metadataFieldHelper.write(model.metadata))
    }

}
