package ru.yandex.intranet.d.dao.units;

import java.io.UncheckedIOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectReader;
import com.fasterxml.jackson.databind.ObjectWriter;
import com.yandex.ydb.table.query.DataQueryResult;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.values.ListValue;
import com.yandex.ydb.table.values.PrimitiveValue;
import com.yandex.ydb.table.values.StructValue;
import com.yandex.ydb.table.values.TupleValue;
import com.yandex.ydb.table.values.Value;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.datasource.impl.YdbQuerySource;
import ru.yandex.intranet.d.datasource.model.YdbReadTableSettings;
import ru.yandex.intranet.d.datasource.model.YdbSession;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.units.UnitModel;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.util.ObjectMapperHolder;

/**
 * Units ensembles DAO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class UnitsEnsemblesDao {

    private final YdbQuerySource ydbQuerySource;
    private final ObjectReader unitsReader;
    private final ObjectWriter unitsWriter;

    public UnitsEnsemblesDao(YdbQuerySource ydbQuerySource,
                             @Qualifier("ydbJsonObjectMapper") ObjectMapperHolder objectMapper) {
        this.ydbQuerySource = ydbQuerySource;
        this.unitsReader = objectMapper.getObjectMapper().readerFor(new TypeReference<Set<UnitModel>>() { });
        this.unitsWriter = objectMapper.getObjectMapper().writerFor(new TypeReference<Set<UnitModel>>() { });
    }

    public Mono<Optional<UnitsEnsembleModel>> getById(YdbTxSession session, String id, TenantId tenantId) {
        String query = ydbQuerySource.getQuery("yql.queries.units_ensembles.getOneById");
        Params params = Params.of("$id", PrimitiveValue.utf8(id),
                "$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
        return session.executeDataQueryRetryable(query, params).map(this::toUnitsEnsemble);
    }

    public Mono<Tuple2<Optional<UnitsEnsembleModel>, String>> getByIdStartTx(YdbTxSession session, String id,
                                                                             TenantId tenantId) {
        String query = ydbQuerySource.getQuery("yql.queries.units_ensembles.getOneById");
        Params params = Params.of("$id", PrimitiveValue.utf8(id),
                "$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
        return session.executeDataQueryRetryable(query, params).map(r -> Tuples.of(toUnitsEnsemble(r), r.getTxId()));
    }

    public Mono<List<UnitsEnsembleModel>> getByIds(YdbTxSession session, List<Tuple2<String, TenantId>> ids) {
        if (ids.isEmpty()) {
            return Mono.just(List.of());
        }
        String query = ydbQuerySource.getQuery("yql.queries.units_ensembles.getByIds");
        ListValue idsParam = ListValue.of(ids.stream().map(id -> TupleValue.of(PrimitiveValue.utf8(id.getT1()),
                PrimitiveValue.utf8(id.getT2().getId()))).toArray(TupleValue[]::new));
        Params params = Params.of("$ids", idsParam);
        return session.executeDataQueryRetryable(query, params).map(this::toUnitsEnsembles);
    }

    public Mono<Void> upsertUnitsEnsembleRetryable(YdbTxSession session, UnitsEnsembleModel unitsEnsemble) {
        String query = ydbQuerySource.getQuery("yql.queries.units_ensembles.upsertOneUnitsEnsemble");
        Map<String, Value> fields = prepareUnitsEnsembleFields(unitsEnsemble);
        Params params = Params.of("$units_ensemble", StructValue.of(fields));
        return session.executeDataQueryRetryable(query, params).then();
    }

    public Mono<Void> upsertUnitsEnsemblesRetryable(YdbTxSession session, List<UnitsEnsembleModel> unitsEnsembles) {
        if (unitsEnsembles.isEmpty()) {
            return Mono.empty();
        }
        String query = ydbQuerySource.getQuery("yql.queries.units_ensembles.upsertManyUnitsEnsembles");
        Params params = Params.of("$units_ensembles", ListValue.of(unitsEnsembles.stream().map(unitsEnsemble -> {
            Map<String, Value> fields = prepareUnitsEnsembleFields(unitsEnsemble);
            return StructValue.of(fields);
        }).toArray(StructValue[]::new)));
        return session.executeDataQueryRetryable(query, params).then();
    }

    public Mono<Void> updateUnitsEnsembleRetryable(YdbTxSession session, UnitsEnsembleModel unitsEnsemble) {
        String query = ydbQuerySource.getQuery("yql.queries.units_ensembles.updateOneUnitsEnsemble");
        Map<String, Value> fields = prepareUnitsEnsembleFields(unitsEnsemble);
        Params params = Params.of("$units_ensemble", StructValue.of(fields));
        return session.executeDataQueryRetryable(query, params).then();
    }

    public Mono<Void> updateUnitsEnsemblesRetryable(YdbTxSession session, List<UnitsEnsembleModel> unitsEnsembles) {
        if (unitsEnsembles.isEmpty()) {
            return Mono.empty();
        }
        String query = ydbQuerySource.getQuery("yql.queries.units_ensembles.updateManyUnitsEnsembles");
        Params params = Params.of("$units_ensembles", ListValue.of(unitsEnsembles.stream().map(unitsEnsemble -> {
            Map<String, Value> fields = prepareUnitsEnsembleFields(unitsEnsemble);
            return StructValue.of(fields);
        }).toArray(StructValue[]::new)));
        return session.executeDataQueryRetryable(query, params).then();
    }

    public Mono<List<UnitsEnsembleModel>> getByTenant(YdbTxSession session, TenantId tenantId,
                                                 String unitsEnsembleIdFrom, int limit, boolean withDeleted) {
        String query;
        Params params;
        ListValue deletedParam = withDeleted
                ? ListValue.of(PrimitiveValue.bool(true), PrimitiveValue.bool(false))
                : ListValue.of(PrimitiveValue.bool(false));
        if (unitsEnsembleIdFrom != null) {
            query = ydbQuerySource.getQuery("yql.queries.units_ensembles.getNextPage");
            params = Params.of("$from_id", PrimitiveValue.utf8(unitsEnsembleIdFrom),
                    "$limit", PrimitiveValue.uint64(limit),
                    "$deleted", deletedParam,
                    "$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
        } else {
            query = ydbQuerySource.getQuery("yql.queries.units_ensembles.getFirstPage");
            params = Params.of("$limit", PrimitiveValue.uint64(limit),
                    "$deleted", deletedParam,
                    "$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
        }
        return session.executeDataQueryRetryable(query, params).map(this::toUnitsEnsembles);
    }

    public Mono<Boolean> existsByKey(YdbTxSession session, TenantId tenantId, String key, boolean withDeleted) {
        ListValue deletedParam = withDeleted
                ? ListValue.of(PrimitiveValue.bool(true), PrimitiveValue.bool(false))
                : ListValue.of(PrimitiveValue.bool(false));
        String query = ydbQuerySource.getQuery("yql.queries.units_ensembles.existsByKey");
        Params params = Params.of("$key", PrimitiveValue.utf8(key),
                "$tenant_id", PrimitiveValue.utf8(tenantId.getId()),
                "$deleted", deletedParam);
        return session.executeDataQueryRetryable(query, params).map(this::toExists);
    }

    private boolean toExists(DataQueryResult result) {
        if (result.isEmpty() || result.getResultSetCount() > 1) {
            throw new IllegalStateException("Exactly one result set is required");
        }
        ResultSetReader reader = result.getResultSet(0);
        if (!reader.next() || reader.getRowCount() > 1) {
            throw new IllegalStateException("Exactly one result is required");
        }
        return reader.getColumn("units_ensemble_exists").getBool();
    }

    private Optional<UnitsEnsembleModel> toUnitsEnsemble(DataQueryResult result) {
        if (result.isEmpty()) {
            return Optional.empty();
        }
        if (result.getResultSetCount() > 1) {
            throw new IllegalStateException("Too many result sets");
        }
        ResultSetReader reader = result.getResultSet(0);
        if (!reader.next()) {
            return Optional.empty();
        }
        if (reader.getRowCount() > 1) {
            throw new IllegalStateException("Non unique units ensemble");
        }
        UnitsEnsembleModel unitsEnsemble = readOneUnitsEnsemble(reader, new HashMap<>());
        return Optional.of(unitsEnsemble);
    }

    private List<UnitsEnsembleModel> toUnitsEnsembles(DataQueryResult result) {
        if (result.isEmpty()) {
            return List.of();
        }
        if (result.getResultSetCount() > 1) {
            throw new IllegalStateException("Too many result sets");
        }
        ResultSetReader reader = result.getResultSet(0);
        List<UnitsEnsembleModel> unitsEnsembles = new ArrayList<>();
        Map<String, TenantId> tenantIdCache = new HashMap<>();
        while (reader.next()) {
            unitsEnsembles.add(readOneUnitsEnsemble(reader, tenantIdCache));
        }
        return unitsEnsembles;
    }

    private UnitsEnsembleModel readOneUnitsEnsemble(ResultSetReader reader, Map<String, TenantId> tenantIdCache) {
        String id = reader.getColumn("id").getUtf8();
        String tenantIdString = reader.getColumn("tenant_id").getUtf8();
        TenantId tenantId = tenantIdCache.computeIfAbsent(tenantIdString, TenantId::new);
        long version = reader.getColumn("version").getInt64();
        String nameEn = reader.getColumn("name_en").getUtf8();
        String nameRu = reader.getColumn("name_ru").getUtf8();
        String descriptionEn = reader.getColumn("description_en").getUtf8();
        String descriptionRu = reader.getColumn("description_ru").getUtf8();
        boolean fractionsAllowed = reader.getColumn("fractions_allowed").getBool();
        boolean deleted = reader.getColumn("deleted").getBool();
        Set<UnitModel> units = readUnits(reader.getColumn("units").getJsonDocument());
        String key = reader.getColumn("key").getUtf8();
        return new UnitsEnsembleModel(id, tenantId, version, nameEn, nameRu, descriptionEn, descriptionRu,
                fractionsAllowed, deleted, units, key);
    }

    private Set<UnitModel> readUnits(String json) {
        try {
            return unitsReader.readValue(json);
        } catch (JsonProcessingException e) {
            throw new UncheckedIOException(e);
        }
    }

    private String writeUnits(Set<UnitModel> units) {
        try {
            return unitsWriter.writeValueAsString(units);
        } catch (JsonProcessingException e) {
            throw new UncheckedIOException(e);
        }
    }

    private Map<String, Value> prepareUnitsEnsembleFields(UnitsEnsembleModel unitsEnsemble) {
        Map<String, Value> fields = new HashMap<>();
        fields.put("id", PrimitiveValue.utf8(unitsEnsemble.getId()));
        fields.put("tenant_id", PrimitiveValue.utf8(unitsEnsemble.getTenantId().getId()));
        fields.put("version", PrimitiveValue.int64(unitsEnsemble.getVersion()));
        fields.put("name_en", PrimitiveValue.utf8(unitsEnsemble.getNameEn()));
        fields.put("name_ru", PrimitiveValue.utf8(unitsEnsemble.getNameRu()));
        fields.put("description_en", PrimitiveValue.utf8(unitsEnsemble.getDescriptionEn()));
        fields.put("description_ru", PrimitiveValue.utf8(unitsEnsemble.getDescriptionRu()));
        fields.put("fractions_allowed", PrimitiveValue.bool(unitsEnsemble.isFractionsAllowed()));
        fields.put("deleted", PrimitiveValue.bool(unitsEnsemble.isDeleted()));
        fields.put("units", PrimitiveValue.jsonDocument(writeUnits(unitsEnsemble.getUnits())));
        fields.put("key", PrimitiveValue.utf8(unitsEnsemble.getKey()));
        return fields;
    }

    public Flux<UnitsEnsembleModel> getAll(YdbSession session) {
        YdbReadTableSettings.Builder settings = YdbReadTableSettings.builder();
        settings.ordered(true);
        settings.addColumn("id");
        settings.addColumn("tenant_id");
        settings.addColumn("version");
        settings.addColumn("name_en");
        settings.addColumn("name_ru");
        settings.addColumn("description_en");
        settings.addColumn("description_ru");
        settings.addColumn("fractions_allowed");
        settings.addColumn("deleted");
        settings.addColumn("units");
        settings.addColumn("key");
        return session.readTable(
                ydbQuerySource.preprocessTableName("units_ensembles"), settings.build()
        ).flatMapIterable((ResultSetReader reader) -> {
            List<UnitsEnsembleModel> page = new ArrayList<>();
            Map<String, TenantId> tenantIdCache = new HashMap<>();
            tenantIdCache.put(Tenants.DEFAULT_TENANT_ID.getId(), Tenants.DEFAULT_TENANT_ID);
            while (reader.next()) {
                page.add(readOneUnitsEnsemble(reader, tenantIdCache));
            }
            return page;
        });
    }
}
