package ru.yandex.intranet.d.dao.usage

import com.fasterxml.jackson.core.type.TypeReference
import com.yandex.ydb.table.query.Params
import com.yandex.ydb.table.result.ResultSetReader
import com.yandex.ydb.table.values.ListValue
import com.yandex.ydb.table.values.PrimitiveValue
import com.yandex.ydb.table.values.StructValue
import kotlinx.coroutines.reactor.awaitSingle
import kotlinx.coroutines.reactor.awaitSingleOrNull
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.stereotype.Component
import ru.yandex.intranet.d.dao.DaoPagination
import ru.yandex.intranet.d.dao.DaoReader
import ru.yandex.intranet.d.dao.JsonFieldHelper
import ru.yandex.intranet.d.dao.WithTx
import ru.yandex.intranet.d.datasource.impl.YdbQuerySource
import ru.yandex.intranet.d.datasource.model.YdbTxSession
import ru.yandex.intranet.d.model.TenantId
import ru.yandex.intranet.d.model.usage.AccountUsageKey
import ru.yandex.intranet.d.model.usage.AccountUsageKeyWithEpoch
import ru.yandex.intranet.d.model.usage.AccountUsageKeyWithEpochPage
import ru.yandex.intranet.d.model.usage.AccountUsageModel
import ru.yandex.intranet.d.model.usage.UsageAmount
import ru.yandex.intranet.d.util.ObjectMapperHolder

/**
 * Account usage DAO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
class AccountUsageDao(private val ydbQuerySource: YdbQuerySource,
                      @Qualifier("ydbJsonObjectMapper") private val objectMapper: ObjectMapperHolder) {

    private val ownUsageFieldHelper: JsonFieldHelper<UsageAmount> = JsonFieldHelper(objectMapper,
        object : TypeReference<UsageAmount>() {})

    suspend fun getById(session: YdbTxSession, id: AccountUsageKey): AccountUsageModel? {
        val query = ydbQuerySource.getQuery("yql.queries.accountUsage.getById")
        val params = toKeyParams(id)
        return DaoReader.toModel(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toModel)
    }

    suspend fun getByIds(session: YdbTxSession,
                         ids: Collection<AccountUsageKey>): List<AccountUsageModel> {
        if (ids.isEmpty()) {
            return listOf()
        }
        return ids.distinct().chunked(1000).map {
            val query = ydbQuerySource.getQuery("yql.queries.accountUsage.getByIds")
            val params = toKeyListParams(it)
            DaoReader.toModels(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toModel)
        }.flatten()
    }

    suspend fun upsertOneRetryable(session: YdbTxSession, value: AccountUsageModel) {
        val query = ydbQuerySource.getQuery("yql.queries.accountUsage.upsertOne")
        val params = toUpsertOneParams(value)
        session.executeDataQueryRetryable(query, params).awaitSingleOrNull()
    }

    suspend fun upsertManyRetryable(session: YdbTxSession, values: Collection<AccountUsageModel>) {
        if (values.isEmpty()) {
            return
        }
        val query = ydbQuerySource.getQuery("yql.queries.accountUsage.upsertMany")
        val params = toUpsertManyParams(values)
        session.executeDataQueryRetryable(query, params).awaitSingleOrNull()
    }

    suspend fun getByAccount(session: YdbTxSession, tenantId: TenantId,
                             accountId: String, perPage: Long): List<AccountUsageModel> {
        return DaoPagination.getAllPages(session,
            { ydbQuerySource.getQuery("yql.queries.accountUsage.getByAccountFirstPage") },
            { ydbQuerySource.getQuery("yql.queries.accountUsage.getByAccountNextPage") },
            { limit -> toGetByAccountFirstPageParams(tenantId, accountId, limit) },
            { limit, lastOnPreviousPage -> toGetByAccountNextPageParams(tenantId, accountId, limit,
                lastOnPreviousPage.key.resourceId)},
            this::toModel,
            perPage
        )
    }

    suspend fun getByAccounts(session: YdbTxSession, tenantId: TenantId,
                              accountIds: Collection<String>, perPage: Long): List<AccountUsageModel> {
        if (accountIds.isEmpty()) {
            return listOf()
        }
        val sortedAccountIds = accountIds.distinct().sorted()
        return DaoPagination.getAllPages(session,
            { ydbQuerySource.getQuery("yql.queries.accountUsage.getByAccountsFirstPage") },
            { ydbQuerySource.getQuery("yql.queries.accountUsage.getByAccountsNextPage") },
            { ydbQuerySource.getQuery("yql.queries.accountUsage.getByAccountsLastPage") },
            { limit -> toGetByAccountsFirstPageParams(tenantId, sortedAccountIds, limit) },
            { limit, lastOnPreviousPage -> toGetByAccountsNextPageParams(tenantId,
                accountIdsTail(sortedAccountIds, lastOnPreviousPage.key.accountId), limit,
                lastOnPreviousPage.key.accountId, lastOnPreviousPage.key.resourceId)},
            { limit, lastOnPreviousPage -> toGetByAccountsLastPageParams(tenantId, limit,
                lastOnPreviousPage.key.accountId, lastOnPreviousPage.key.resourceId)},
            { lastOnPreviousPage -> accountIdsTail(sortedAccountIds, lastOnPreviousPage.key.accountId).isEmpty() },
            this::toModel,
            perPage
        )
    }

    suspend fun getKeysForOlderEpochsFirstPage(session: YdbTxSession,
                                               tenantId: TenantId,
                                               resourceId: String,
                                               currentEpoch: Long,
                                               limit: Long): WithTx<AccountUsageKeyWithEpochPage> {
        val query = ydbQuerySource.getQuery("yql.queries.accountUsage.getKeysForOlderEpochsFirstPage")
        val params = toGetKeysForOlderEpochsFirstPageParams(tenantId, resourceId, currentEpoch, limit)
        val page = DaoReader.toModelsWithTx(session.executeDataQueryRetryable(query, params).awaitSingle(),
            this::toKeyWithEpoch)
        return WithTx(AccountUsageKeyWithEpochPage(keys = page.value,
            nextFrom = if (page.value.size >= limit) { page.value.last() } else { null }), page.txId)
    }

    suspend fun getKeysForOlderEpochsNextPage(
        session: YdbTxSession, from: AccountUsageKeyWithEpoch, limit: Long): WithTx<AccountUsageKeyWithEpochPage> {
        val query = ydbQuerySource.getQuery("yql.queries.accountUsage.getKeysForOlderEpochsNextPage")
        val params = toGetKeysForOlderEpochsNextPageParams(from, limit)
        val page = DaoReader.toModelsWithTx(session.executeDataQueryRetryable(query, params).awaitSingle(),
            this::toKeyWithEpoch)
        return WithTx(AccountUsageKeyWithEpochPage(keys = page.value,
            nextFrom = if (page.value.size >= limit) { page.value.last() } else { null }), page.txId)
    }

    suspend fun deleteByIdRetryable(session: YdbTxSession, id: AccountUsageKey) {
        val query = ydbQuerySource.getQuery("yql.queries.accountUsage.deleteById")
        val params = toKeyParams(id)
        session.executeDataQueryRetryable(query, params).awaitSingleOrNull()
    }

    suspend fun deleteByIdsRetryable(session: YdbTxSession, ids: Collection<AccountUsageKey>) {
        if (ids.isEmpty()) {
            return
        }
        val query = ydbQuerySource.getQuery("yql.queries.accountUsage.deleteByIds")
        val params = toKeyListParams(ids)
        session.executeDataQueryRetryable(query, params).awaitSingleOrNull()
    }

    suspend fun getKeysForOlderEpochsMultiResourceFirstPage(session: YdbTxSession,
                                                            tenantId: TenantId,
                                                            resourceIds: Collection<String>,
                                                            currentEpoch: Long,
                                                            limit: Long): WithTx<AccountUsageKeyWithEpochPage> {
        val query = ydbQuerySource
            .getQuery("yql.queries.accountUsage.getKeysForOlderEpochsMultiResourceFirstPage")
        val params = toGetKeysForOlderEpochsMultiResourceFirstPageParams(tenantId, resourceIds, currentEpoch, limit)
        val page = DaoReader.toModelsWithTx(session.executeDataQueryRetryable(query, params).awaitSingle(),
            this::toKeyWithEpoch)
        return WithTx(AccountUsageKeyWithEpochPage(keys = page.value,
            nextFrom = if (page.value.size >= limit) { page.value.last() } else { null }), page.txId)
    }

    suspend fun getKeysForOlderEpochsMultiResourceNextPage(
        session: YdbTxSession,
        from: AccountUsageKeyWithEpoch,
        resourceIds: Collection<String>,
        currentEpoch: Long,
        limit: Long
    ): WithTx<AccountUsageKeyWithEpochPage> {
        val sortedResourceIds = resourceIds.sortedDescending()
        val resourceIdsTail = sortedResourceIds.filter { it < from.key.resourceId }
        val (query, params) = if (resourceIdsTail.isNotEmpty()) {
            val nextQuery = ydbQuerySource
                .getQuery("yql.queries.accountUsage.getKeysForOlderEpochsMultiResourceNextPage")
            val nextParams = toGetKeysForOlderEpochsMultiResourceNextPageParams(from, resourceIdsTail,
                currentEpoch, limit)
            Pair(nextQuery, nextParams)
        } else {
            val lastQuery = ydbQuerySource
                .getQuery("yql.queries.accountUsage.getKeysForOlderEpochsMultiResourceLastPage")
            val lastParams = toGetKeysForOlderEpochsMultiResourceLastPageParams(from, limit)
            Pair(lastQuery, lastParams)
        }
        val page = DaoReader.toModelsWithTx(session.executeDataQueryRetryable(query, params).awaitSingle(),
            this::toKeyWithEpoch)
        return WithTx(AccountUsageKeyWithEpochPage(keys = page.value,
            nextFrom = if (page.value.size >= limit) { page.value.last() } else { null }), page.txId)
    }

    private fun toModel(reader: ResultSetReader) = AccountUsageModel(
        key = toKey(reader),
        lastUpdate = reader.getColumn("last_update").timestamp,
        epoch = reader.getColumn("epoch").int64,
        ownUsage = ownUsageFieldHelper.read(reader.getColumn("own_usage"))
    )

    private fun toKey(reader: ResultSetReader) = AccountUsageKey(
        tenantId = TenantId(reader.getColumn("tenant_id").utf8),
        accountId = reader.getColumn("account_id").utf8,
        resourceId = reader.getColumn("resource_id").utf8
    )

    private fun toKeyWithEpoch(reader: ResultSetReader) = AccountUsageKeyWithEpoch(
        key = toKey(reader),
        epoch = reader.getColumn("epoch").int64
    )

    private fun toKeyParams(id: AccountUsageKey) = Params.of(
        "\$id", toKeyStruct(id)
    )

    private fun toKeyListParams(ids: Collection<AccountUsageKey>) = Params.of(
        "\$ids", ListValue.of(*ids.map { toKeyStruct(it) }.toTypedArray())
    )

    private fun toKeyStruct(id: AccountUsageKey) = StructValue.of(mapOf(
        "tenant_id" to PrimitiveValue.utf8(id.tenantId.id),
        "account_id" to PrimitiveValue.utf8(id.accountId),
        "resource_id" to PrimitiveValue.utf8(id.resourceId)
    ))

    private fun toUpsertOneParams(value: AccountUsageModel) = Params.of(
        "\$value", toUpsertStruct(value)
    )

    private fun toUpsertManyParams(values: Collection<AccountUsageModel>) = Params.of(
        "\$values", ListValue.of(*values.map { toUpsertStruct(it) }.toTypedArray())
    )

    private fun toUpsertStruct(value: AccountUsageModel) = StructValue.of(
        mapOf(
            "tenant_id" to PrimitiveValue.utf8(value.key.tenantId.id),
            "account_id" to PrimitiveValue.utf8(value.key.accountId),
            "resource_id" to PrimitiveValue.utf8(value.key.resourceId),
            "last_update" to PrimitiveValue.timestamp(value.lastUpdate),
            "epoch" to PrimitiveValue.int64(value.epoch),
            "own_usage" to ownUsageFieldHelper.writeOptional(value.ownUsage)
        )
    )

    private fun toGetByAccountFirstPageParams(tenantId: TenantId, accountId: String, limit: Long) = Params
        .of("\$tenant_id", PrimitiveValue.utf8(tenantId.id),
            "\$account_id", PrimitiveValue.utf8(accountId),
            "\$limit", PrimitiveValue.uint64(limit))

    private fun toGetByAccountNextPageParams(tenantId: TenantId, accountId: String, limit: Long,
                                             fromResourceId: String) = Params
        .of("\$tenant_id", PrimitiveValue.utf8(tenantId.id),
            "\$account_id", PrimitiveValue.utf8(accountId),
            "\$limit", PrimitiveValue.uint64(limit),
            "\$from_resource_id", PrimitiveValue.utf8(fromResourceId))

    private fun toGetByAccountsFirstPageParams(tenantId: TenantId, accountIds: Collection<String>, limit: Long) = Params
        .of("\$tenant_id", PrimitiveValue.utf8(tenantId.id),
            "\$account_ids", ListValue.of(*accountIds.map { PrimitiveValue.utf8(it) }.toTypedArray()),
            "\$limit", PrimitiveValue.uint64(limit))

    private fun toGetByAccountsNextPageParams(tenantId: TenantId, accountIds: Collection<String>, limit: Long,
                                              fromAccountId: String, fromResourceId: String) = Params
        .of("\$tenant_id", PrimitiveValue.utf8(tenantId.id),
            "\$account_ids", ListValue.of(*accountIds.map { PrimitiveValue.utf8(it) }.toTypedArray()),
            "\$limit", PrimitiveValue.uint64(limit),
            "\$from_account_id", PrimitiveValue.utf8(fromAccountId),
            "\$from_resource_id", PrimitiveValue.utf8(fromResourceId))

    private fun toGetByAccountsLastPageParams(tenantId: TenantId, limit: Long, fromAccountId: String,
                                              fromResourceId: String) = Params
        .of("\$tenant_id", PrimitiveValue.utf8(tenantId.id),
            "\$limit", PrimitiveValue.uint64(limit),
            "\$from_account_id", PrimitiveValue.utf8(fromAccountId),
            "\$from_resource_id", PrimitiveValue.utf8(fromResourceId))

    private fun accountIdsTail(sortedAccountIds: List<String>, currentAccountId: String) = sortedAccountIds
        .filter { it > currentAccountId }

    private fun toGetKeysForOlderEpochsFirstPageParams(tenantId: TenantId, resourceId: String,
                                                       currentEpoch: Long, limit: Long) = Params
        .of("\$tenant_id", PrimitiveValue.utf8(tenantId.id),
            "\$resource_id", PrimitiveValue.utf8(resourceId),
            "\$current_epoch", PrimitiveValue.int64(currentEpoch),
            "\$limit", PrimitiveValue.uint64(limit)
        )

    private fun toGetKeysForOlderEpochsNextPageParams(from: AccountUsageKeyWithEpoch, limit: Long) = Params
        .of("\$tenant_id", PrimitiveValue.utf8(from.key.tenantId.id),
            "\$resource_id", PrimitiveValue.utf8(from.key.resourceId),
            "\$from_epoch", PrimitiveValue.int64(from.epoch),
            "\$from_account_id", PrimitiveValue.utf8(from.key.accountId),
            "\$limit", PrimitiveValue.uint64(limit)
        )

    private fun toGetKeysForOlderEpochsMultiResourceFirstPageParams(tenantId: TenantId, resourceIds: Collection<String>,
                                                                    currentEpoch: Long, limit: Long) = Params
        .of("\$tenant_id", PrimitiveValue.utf8(tenantId.id),
            "\$resource_ids", ListValue.of(*resourceIds.map { PrimitiveValue.utf8(it) }.toTypedArray()),
            "\$current_epoch", PrimitiveValue.int64(currentEpoch),
            "\$limit", PrimitiveValue.uint64(limit)
        )

    private fun toGetKeysForOlderEpochsMultiResourceNextPageParams(from: AccountUsageKeyWithEpoch,
                                                                   resourceIds: Collection<String>, currentEpoch: Long,
                                                                   limit: Long) = Params
        .copyOf(mapOf("\$tenant_id" to PrimitiveValue.utf8(from.key.tenantId.id),
            "\$resource_ids" to ListValue.of(*resourceIds.map { PrimitiveValue.utf8(it) }.toTypedArray()),
            "\$from_resource_id" to PrimitiveValue.utf8(from.key.resourceId),
            "\$current_epoch" to PrimitiveValue.int64(currentEpoch),
            "\$from_epoch" to PrimitiveValue.int64(from.epoch),
            "\$from_account_id" to PrimitiveValue.utf8(from.key.accountId),
            "\$limit" to PrimitiveValue.uint64(limit)
        ))

    private fun toGetKeysForOlderEpochsMultiResourceLastPageParams(from: AccountUsageKeyWithEpoch,
                                                                   limit: Long) = Params
        .of("\$tenant_id", PrimitiveValue.utf8(from.key.tenantId.id),
            "\$from_resource_id", PrimitiveValue.utf8(from.key.resourceId),
            "\$from_epoch", PrimitiveValue.int64(from.epoch),
            "\$from_account_id", PrimitiveValue.utf8(from.key.accountId),
            "\$limit", PrimitiveValue.uint64(limit)
        )

}
