package ru.yandex.intranet.d.dao.usage

import com.fasterxml.jackson.core.type.TypeReference
import com.yandex.ydb.table.query.Params
import com.yandex.ydb.table.result.ResultSetReader
import com.yandex.ydb.table.values.ListValue
import com.yandex.ydb.table.values.PrimitiveValue
import com.yandex.ydb.table.values.StructValue
import kotlinx.coroutines.reactor.awaitSingle
import kotlinx.coroutines.reactor.awaitSingleOrNull
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.stereotype.Component
import ru.yandex.intranet.d.dao.DaoPagination
import ru.yandex.intranet.d.dao.DaoReader
import ru.yandex.intranet.d.dao.JsonFieldHelper
import ru.yandex.intranet.d.dao.WithTx
import ru.yandex.intranet.d.datasource.impl.YdbQuerySource
import ru.yandex.intranet.d.datasource.model.YdbTxSession
import ru.yandex.intranet.d.model.TenantId
import ru.yandex.intranet.d.model.usage.FolderUsageKey
import ru.yandex.intranet.d.model.usage.FolderUsageKeyWithEpoch
import ru.yandex.intranet.d.model.usage.FolderUsageKeyWithEpochPage
import ru.yandex.intranet.d.model.usage.FolderUsageModel
import ru.yandex.intranet.d.model.usage.UsageAmount
import ru.yandex.intranet.d.util.ObjectMapperHolder

/**
 * Folder usage DAO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
class FolderUsageDao(private val ydbQuerySource: YdbQuerySource,
                     @Qualifier("ydbJsonObjectMapper") private val objectMapper: ObjectMapperHolder) {

    private val ownUsageFieldHelper: JsonFieldHelper<UsageAmount> = JsonFieldHelper(objectMapper,
        object : TypeReference<UsageAmount>() {})

    suspend fun getById(session: YdbTxSession, id: FolderUsageKey): FolderUsageModel? {
        val query = ydbQuerySource.getQuery("yql.queries.folderUsage.getById")
        val params = toKeyParams(id)
        return DaoReader.toModel(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toModel)
    }

    suspend fun getByIds(session: YdbTxSession,
                         ids: Collection<FolderUsageKey>): List<FolderUsageModel> {
        if (ids.isEmpty()) {
            return listOf()
        }
        return ids.distinct().chunked(1000).map {
            val query = ydbQuerySource.getQuery("yql.queries.folderUsage.getByIds")
            val params = toKeyListParams(it)
            DaoReader.toModels(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toModel)
        }.flatten()
    }

    suspend fun upsertOneRetryable(session: YdbTxSession, value: FolderUsageModel) {
        val query = ydbQuerySource.getQuery("yql.queries.folderUsage.upsertOne")
        val params = toUpsertOneParams(value)
        session.executeDataQueryRetryable(query, params).awaitSingleOrNull()
    }

    suspend fun upsertManyRetryable(session: YdbTxSession, values: Collection<FolderUsageModel>) {
        if (values.isEmpty()) {
            return
        }
        val query = ydbQuerySource.getQuery("yql.queries.folderUsage.upsertMany")
        val params = toUpsertManyParams(values)
        session.executeDataQueryRetryable(query, params).awaitSingleOrNull()
    }

    suspend fun getByFolder(session: YdbTxSession, tenantId: TenantId,
                            folderId: String, perPage: Long): List<FolderUsageModel> {
        return DaoPagination.getAllPages(session,
            { ydbQuerySource.getQuery("yql.queries.folderUsage.getByFolderFirstPage") },
            { ydbQuerySource.getQuery("yql.queries.folderUsage.getByFolderNextPage") },
            { limit -> toGetByFolderFirstPageParams(tenantId, folderId, limit) },
            { limit, lastOnPreviousPage -> toGetByFolderNextPageParams(tenantId, folderId, limit,
                lastOnPreviousPage.key.resourceId)},
            this::toModel,
            perPage
        )
    }

    suspend fun getByFolders(session: YdbTxSession, tenantId: TenantId,
                             folderIds: Collection<String>, perPage: Long): List<FolderUsageModel> {
        if (folderIds.isEmpty()) {
            return listOf()
        }
        val sortedFolderIds = folderIds.distinct().sorted()
        return DaoPagination.getAllPages(session,
            { ydbQuerySource.getQuery("yql.queries.folderUsage.getByFoldersFirstPage") },
            { ydbQuerySource.getQuery("yql.queries.folderUsage.getByFoldersNextPage") },
            { ydbQuerySource.getQuery("yql.queries.folderUsage.getByFoldersLastPage") },
            { limit -> toGetByFoldersFirstPageParams(tenantId, sortedFolderIds, limit) },
            { limit, lastOnPreviousPage -> toGetByFoldersNextPageParams(tenantId,
                folderIdsTail(sortedFolderIds, lastOnPreviousPage.key.folderId), limit,
                lastOnPreviousPage.key.folderId, lastOnPreviousPage.key.resourceId)},
            { limit, lastOnPreviousPage -> toGetByFoldersLastPageParams(tenantId, limit,
                lastOnPreviousPage.key.folderId, lastOnPreviousPage.key.resourceId)},
            { lastOnPreviousPage -> folderIdsTail(sortedFolderIds, lastOnPreviousPage.key.folderId).isEmpty() },
            this::toModel,
            perPage
        )
    }

    suspend fun getKeysForOlderEpochsFirstPage(session: YdbTxSession,
                                               tenantId: TenantId,
                                               resourceId: String,
                                               currentEpoch: Long,
                                               limit: Long): WithTx<FolderUsageKeyWithEpochPage> {
        val query = ydbQuerySource.getQuery("yql.queries.folderUsage.getKeysForOlderEpochsFirstPage")
        val params = toGetKeysForOlderEpochsFirstPageParams(tenantId, resourceId, currentEpoch, limit)
        val page = DaoReader.toModelsWithTx(session.executeDataQueryRetryable(query, params).awaitSingle(),
            this::toKeyWithEpoch)
        return WithTx(FolderUsageKeyWithEpochPage(keys = page.value,
            nextFrom = if (page.value.size >= limit) { page.value.last() } else { null }), page.txId)
    }

    suspend fun getKeysForOlderEpochsNextPage(
        session: YdbTxSession, from: FolderUsageKeyWithEpoch, limit: Long): WithTx<FolderUsageKeyWithEpochPage> {
        val query = ydbQuerySource.getQuery("yql.queries.folderUsage.getKeysForOlderEpochsNextPage")
        val params = toGetKeysForOlderEpochsNextPageParams(from, limit)
        val page = DaoReader.toModelsWithTx(session.executeDataQueryRetryable(query, params).awaitSingle(),
            this::toKeyWithEpoch)
        return WithTx(FolderUsageKeyWithEpochPage(keys = page.value,
            nextFrom = if (page.value.size >= limit) { page.value.last() } else { null }), page.txId)
    }

    suspend fun deleteByIdRetryable(session: YdbTxSession, id: FolderUsageKey) {
        val query = ydbQuerySource.getQuery("yql.queries.folderUsage.deleteById")
        val params = toKeyParams(id)
        session.executeDataQueryRetryable(query, params).awaitSingleOrNull()
    }

    suspend fun deleteByIdsRetryable(session: YdbTxSession, ids: Collection<FolderUsageKey>) {
        if (ids.isEmpty()) {
            return
        }
        val query = ydbQuerySource.getQuery("yql.queries.folderUsage.deleteByIds")
        val params = toKeyListParams(ids)
        session.executeDataQueryRetryable(query, params).awaitSingleOrNull()
    }

    suspend fun getKeysForOlderEpochsMultiResourceFirstPage(session: YdbTxSession,
                                                            tenantId: TenantId,
                                                            resourceIds: Collection<String>,
                                                            currentEpoch: Long,
                                                            limit: Long): WithTx<FolderUsageKeyWithEpochPage> {
        val query = ydbQuerySource
            .getQuery("yql.queries.folderUsage.getKeysForOlderEpochsMultiResourceFirstPage")
        val params = toGetKeysForOlderEpochsMultiResourceFirstPageParams(tenantId, resourceIds, currentEpoch, limit)
        val page = DaoReader.toModelsWithTx(session.executeDataQueryRetryable(query, params).awaitSingle(),
            this::toKeyWithEpoch)
        return WithTx(FolderUsageKeyWithEpochPage(keys = page.value,
            nextFrom = if (page.value.size >= limit) { page.value.last() } else { null }), page.txId)
    }

    suspend fun getKeysForOlderEpochsMultiResourceNextPage(
        session: YdbTxSession,
        from: FolderUsageKeyWithEpoch,
        resourceIds: Collection<String>,
        currentEpoch: Long,
        limit: Long
    ): WithTx<FolderUsageKeyWithEpochPage> {
        val sortedResourceIds = resourceIds.sortedDescending()
        val resourceIdsTail = sortedResourceIds.filter { it < from.key.resourceId }
        val (query, params) = if (resourceIdsTail.isNotEmpty()) {
            val nextQuery = ydbQuerySource
                .getQuery("yql.queries.folderUsage.getKeysForOlderEpochsMultiResourceNextPage")
            val nextParams = toGetKeysForOlderEpochsMultiResourceNextPageParams(from, resourceIdsTail,
                currentEpoch, limit)
            Pair(nextQuery, nextParams)
        } else {
            val lastQuery = ydbQuerySource
                .getQuery("yql.queries.folderUsage.getKeysForOlderEpochsMultiResourceLastPage")
            val lastParams = toGetKeysForOlderEpochsMultiResourceLastPageParams(from, limit)
            Pair(lastQuery, lastParams)
        }
        val page = DaoReader.toModelsWithTx(session.executeDataQueryRetryable(query, params).awaitSingle(),
            this::toKeyWithEpoch)
        return WithTx(FolderUsageKeyWithEpochPage(keys = page.value,
            nextFrom = if (page.value.size >= limit) { page.value.last() } else { null }), page.txId)
    }

    private fun toModel(reader: ResultSetReader) = FolderUsageModel(
        key = toKey(reader),
        lastUpdate = reader.getColumn("last_update").timestamp,
        epoch = reader.getColumn("epoch").int64,
        ownUsage = ownUsageFieldHelper.read(reader.getColumn("own_usage"))
    )

    private fun toKey(reader: ResultSetReader) = FolderUsageKey(
        tenantId = TenantId(reader.getColumn("tenant_id").utf8),
        folderId = reader.getColumn("folder_id").utf8,
        resourceId = reader.getColumn("resource_id").utf8
    )

    private fun toKeyWithEpoch(reader: ResultSetReader) = FolderUsageKeyWithEpoch(
        key = toKey(reader),
        epoch = reader.getColumn("epoch").int64
    )

    private fun toKeyParams(id: FolderUsageKey) = Params.of(
        "\$id", toKeyStruct(id)
    )

    private fun toKeyListParams(ids: Collection<FolderUsageKey>) = Params.of(
        "\$ids", ListValue.of(*ids.map { toKeyStruct(it) }.toTypedArray())
    )

    private fun toKeyStruct(id: FolderUsageKey) = StructValue.of(mapOf(
        "tenant_id" to PrimitiveValue.utf8(id.tenantId.id),
        "folder_id" to PrimitiveValue.utf8(id.folderId),
        "resource_id" to PrimitiveValue.utf8(id.resourceId)
    ))

    private fun toUpsertOneParams(value: FolderUsageModel) = Params.of(
        "\$value", toUpsertStruct(value)
    )

    private fun toUpsertManyParams(values: Collection<FolderUsageModel>) = Params.of(
        "\$values", ListValue.of(*values.map { toUpsertStruct(it) }.toTypedArray())
    )

    private fun toUpsertStruct(value: FolderUsageModel) = StructValue.of(
        mapOf(
            "tenant_id" to PrimitiveValue.utf8(value.key.tenantId.id),
            "folder_id" to PrimitiveValue.utf8(value.key.folderId),
            "resource_id" to PrimitiveValue.utf8(value.key.resourceId),
            "last_update" to PrimitiveValue.timestamp(value.lastUpdate),
            "epoch" to PrimitiveValue.int64(value.epoch),
            "own_usage" to ownUsageFieldHelper.writeOptional(value.ownUsage)
        )
    )

    private fun toGetByFolderFirstPageParams(tenantId: TenantId, folderId: String, limit: Long) = Params
        .of("\$tenant_id", PrimitiveValue.utf8(tenantId.id),
            "\$folder_id", PrimitiveValue.utf8(folderId),
            "\$limit", PrimitiveValue.uint64(limit))

    private fun toGetByFolderNextPageParams(tenantId: TenantId, folderId: String, limit: Long,
                                            fromResourceId: String) = Params
        .of("\$tenant_id", PrimitiveValue.utf8(tenantId.id),
            "\$folder_id", PrimitiveValue.utf8(folderId),
            "\$limit", PrimitiveValue.uint64(limit),
            "\$from_resource_id", PrimitiveValue.utf8(fromResourceId))

    private fun toGetByFoldersFirstPageParams(tenantId: TenantId, folderIds: Collection<String>, limit: Long) = Params
        .of("\$tenant_id", PrimitiveValue.utf8(tenantId.id),
            "\$folder_ids", ListValue.of(*folderIds.map { PrimitiveValue.utf8(it) }.toTypedArray()),
            "\$limit", PrimitiveValue.uint64(limit))

    private fun toGetByFoldersNextPageParams(tenantId: TenantId, folderIds: Collection<String>, limit: Long,
                                             fromFolderId: String, fromResourceId: String) = Params
        .of("\$tenant_id", PrimitiveValue.utf8(tenantId.id),
            "\$folder_ids", ListValue.of(*folderIds.map { PrimitiveValue.utf8(it) }.toTypedArray()),
            "\$limit", PrimitiveValue.uint64(limit),
            "\$from_folder_id", PrimitiveValue.utf8(fromFolderId),
            "\$from_resource_id", PrimitiveValue.utf8(fromResourceId))

    private fun toGetByFoldersLastPageParams(tenantId: TenantId, limit: Long, fromFolderId: String,
                                             fromResourceId: String) = Params
        .of("\$tenant_id", PrimitiveValue.utf8(tenantId.id),
            "\$limit", PrimitiveValue.uint64(limit),
            "\$from_folder_id", PrimitiveValue.utf8(fromFolderId),
            "\$from_resource_id", PrimitiveValue.utf8(fromResourceId))

    private fun folderIdsTail(sortedFolderIds: List<String>, currentFolderId: String) = sortedFolderIds
        .filter { it > currentFolderId }

    private fun toGetKeysForOlderEpochsFirstPageParams(tenantId: TenantId, resourceId: String,
                                                       currentEpoch: Long, limit: Long) = Params
        .of("\$tenant_id", PrimitiveValue.utf8(tenantId.id),
            "\$resource_id", PrimitiveValue.utf8(resourceId),
            "\$current_epoch", PrimitiveValue.int64(currentEpoch),
            "\$limit", PrimitiveValue.uint64(limit)
        )

    private fun toGetKeysForOlderEpochsNextPageParams(from: FolderUsageKeyWithEpoch, limit: Long) = Params
        .of("\$tenant_id", PrimitiveValue.utf8(from.key.tenantId.id),
            "\$resource_id", PrimitiveValue.utf8(from.key.resourceId),
            "\$from_epoch", PrimitiveValue.int64(from.epoch),
            "\$from_folder_id", PrimitiveValue.utf8(from.key.folderId),
            "\$limit", PrimitiveValue.uint64(limit)
        )

    private fun toGetKeysForOlderEpochsMultiResourceFirstPageParams(tenantId: TenantId, resourceIds: Collection<String>,
                                                                    currentEpoch: Long, limit: Long) = Params
        .of("\$tenant_id", PrimitiveValue.utf8(tenantId.id),
            "\$resource_ids", ListValue.of(*resourceIds.map { PrimitiveValue.utf8(it) }.toTypedArray()),
            "\$current_epoch", PrimitiveValue.int64(currentEpoch),
            "\$limit", PrimitiveValue.uint64(limit)
        )

    private fun toGetKeysForOlderEpochsMultiResourceNextPageParams(from: FolderUsageKeyWithEpoch,
                                                                   resourceIds: Collection<String>, currentEpoch: Long,
                                                                   limit: Long) = Params
        .copyOf(mapOf("\$tenant_id" to PrimitiveValue.utf8(from.key.tenantId.id),
            "\$resource_ids" to ListValue.of(*resourceIds.map { PrimitiveValue.utf8(it) }.toTypedArray()),
            "\$from_resource_id" to PrimitiveValue.utf8(from.key.resourceId),
            "\$current_epoch" to PrimitiveValue.int64(currentEpoch),
            "\$from_epoch" to PrimitiveValue.int64(from.epoch),
            "\$from_folder_id" to PrimitiveValue.utf8(from.key.folderId),
            "\$limit" to PrimitiveValue.uint64(limit)
        ))

    private fun toGetKeysForOlderEpochsMultiResourceLastPageParams(from: FolderUsageKeyWithEpoch,
                                                                   limit: Long) = Params
        .of("\$tenant_id", PrimitiveValue.utf8(from.key.tenantId.id),
            "\$from_resource_id", PrimitiveValue.utf8(from.key.resourceId),
            "\$from_epoch", PrimitiveValue.int64(from.epoch),
            "\$from_folder_id", PrimitiveValue.utf8(from.key.folderId),
            "\$limit", PrimitiveValue.uint64(limit)
        )

}
