package ru.yandex.intranet.d.dao.usage

import com.fasterxml.jackson.core.type.TypeReference
import com.yandex.ydb.table.query.Params
import com.yandex.ydb.table.result.ResultSetReader
import com.yandex.ydb.table.values.ListValue
import com.yandex.ydb.table.values.PrimitiveValue
import com.yandex.ydb.table.values.StructValue
import kotlinx.coroutines.reactor.awaitSingle
import kotlinx.coroutines.reactor.awaitSingleOrNull
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.stereotype.Component
import ru.yandex.intranet.d.dao.DaoPagination
import ru.yandex.intranet.d.dao.DaoReader
import ru.yandex.intranet.d.dao.JsonFieldHelper
import ru.yandex.intranet.d.dao.WithTx
import ru.yandex.intranet.d.datasource.impl.YdbQuerySource
import ru.yandex.intranet.d.datasource.model.YdbExecuteScanQuerySettings
import ru.yandex.intranet.d.datasource.model.YdbSession
import ru.yandex.intranet.d.datasource.model.YdbTxSession
import ru.yandex.intranet.d.model.TenantId
import ru.yandex.intranet.d.model.usage.ServiceUsageAmounts
import ru.yandex.intranet.d.model.usage.ServiceUsageKey
import ru.yandex.intranet.d.model.usage.ServiceUsageKeyWithEpoch
import ru.yandex.intranet.d.model.usage.ServiceUsageKeyWithEpochPage
import ru.yandex.intranet.d.model.usage.ServiceUsageModel
import ru.yandex.intranet.d.util.ObjectMapperHolder
import java.time.Duration

/**
 * Service usage DAO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
class ServiceUsageDao(private val ydbQuerySource: YdbQuerySource,
                      @Qualifier("ydbJsonObjectMapper") private val objectMapper: ObjectMapperHolder) {

    private val usageFieldHelper: JsonFieldHelper<ServiceUsageAmounts> = JsonFieldHelper(objectMapper,
        object : TypeReference<ServiceUsageAmounts>() {})

    suspend fun getById(session: YdbTxSession, id: ServiceUsageKey): ServiceUsageModel? {
        val query = ydbQuerySource.getQuery("yql.queries.serviceUsage.getById")
        val params = toKeyParams(id)
        return DaoReader.toModel(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toModel)
    }

    suspend fun getByIds(session: YdbTxSession,
                         ids: Collection<ServiceUsageKey>): List<ServiceUsageModel> {
        if (ids.isEmpty()) {
            return listOf()
        }
        return ids.distinct().chunked(1000).map {
            val query = ydbQuerySource.getQuery("yql.queries.serviceUsage.getByIds")
            val params = toKeyListParams(it)
            DaoReader.toModels(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toModel)
        }.flatten()
    }

    suspend fun upsertOneRetryable(session: YdbTxSession, value: ServiceUsageModel) {
        val query = ydbQuerySource.getQuery("yql.queries.serviceUsage.upsertOne")
        val params = toUpsertOneParams(value)
        session.executeDataQueryRetryable(query, params).awaitSingleOrNull()
    }

    suspend fun upsertManyRetryable(session: YdbTxSession, values: Collection<ServiceUsageModel>) {
        if (values.isEmpty()) {
            return
        }
        val query = ydbQuerySource.getQuery("yql.queries.serviceUsage.upsertMany")
        val params = toUpsertManyParams(values)
        session.executeDataQueryRetryable(query, params).awaitSingleOrNull()
    }

    suspend fun getByService(session: YdbTxSession, tenantId: TenantId,
                             serviceId: Long, perPage: Long): List<ServiceUsageModel> {
        return DaoPagination.getAllPages(session,
            { ydbQuerySource.getQuery("yql.queries.serviceUsage.getByServiceFirstPage") },
            { ydbQuerySource.getQuery("yql.queries.serviceUsage.getByServiceNextPage") },
            { limit -> toGetByServiceFirstPageParams(tenantId, serviceId, limit) },
            { limit, lastOnPreviousPage -> toGetByServiceNextPageParams(tenantId, serviceId, limit,
                lastOnPreviousPage.key.providerId, lastOnPreviousPage.key.resourceId)},
            this::toModel,
            perPage
        )
    }

    suspend fun getByServiceAndProvider(session: YdbTxSession, tenantId: TenantId,
                                        serviceId: Long, providerId: String, perPage: Long): List<ServiceUsageModel> {
        return DaoPagination.getAllPages(session,
            { ydbQuerySource.getQuery("yql.queries.serviceUsage.getByServiceAndProviderFirstPage") },
            { ydbQuerySource.getQuery("yql.queries.serviceUsage.getByServiceAndProviderNextPage") },
            { limit -> toGetByServiceAndProviderFirstPageParams(tenantId, serviceId, providerId, limit) },
            { limit, lastOnPreviousPage -> toGetByServiceAndProviderNextPageParams(tenantId, serviceId, providerId,
                limit, lastOnPreviousPage.key.resourceId)},
            this::toModel,
            perPage
        )
    }

    suspend fun getByServices(session: YdbTxSession, tenantId: TenantId,
                              serviceIds: Collection<Long>, perPage: Long): List<ServiceUsageModel> {
        if (serviceIds.isEmpty()) {
            return listOf()
        }
        val sortedServiceIds = serviceIds.distinct().sorted()
        return DaoPagination.getAllPages(session,
            { ydbQuerySource.getQuery("yql.queries.serviceUsage.getByServicesFirstPage") },
            { ydbQuerySource.getQuery("yql.queries.serviceUsage.getByServicesNextPage") },
            { ydbQuerySource.getQuery("yql.queries.serviceUsage.getByServicesLastPage") },
            { limit -> toGetByServicesFirstPageParams(tenantId, sortedServiceIds, limit) },
            { limit, lastOnPreviousPage -> toGetByServicesNextPageParams(tenantId,
                serviceIdsTail(sortedServiceIds, lastOnPreviousPage.key.serviceId), limit,
                lastOnPreviousPage.key.serviceId, lastOnPreviousPage.key.providerId,
                lastOnPreviousPage.key.resourceId)},
            { limit, lastOnPreviousPage -> toGetByServicesLastPageParams(tenantId, limit,
                lastOnPreviousPage.key.serviceId, lastOnPreviousPage.key.providerId,
                lastOnPreviousPage.key.resourceId)},
            { lastOnPreviousPage -> serviceIdsTail(sortedServiceIds, lastOnPreviousPage.key.serviceId).isEmpty() },
            this::toModel,
            perPage
        )
    }

    suspend fun getKeysForOlderEpochsFirstPage(session: YdbTxSession,
                                               tenantId: TenantId,
                                               providerId: String,
                                               resourceId: String,
                                               currentEpoch: Long,
                                               limit: Long): WithTx<ServiceUsageKeyWithEpochPage> {
        val query = ydbQuerySource.getQuery("yql.queries.serviceUsage.getKeysForOlderEpochsFirstPage")
        val params = toGetKeysForOlderEpochsFirstPageParams(tenantId, providerId, resourceId, currentEpoch, limit)
        val page = DaoReader.toModelsWithTx(session.executeDataQueryRetryable(query, params).awaitSingle(),
            this::toKeyWithEpoch)
        return WithTx(ServiceUsageKeyWithEpochPage(keys = page.value,
            nextFrom = if (page.value.size >= limit) { page.value.last() } else { null }), page.txId)
    }

    suspend fun getKeysForOlderEpochsNextPage(
        session: YdbTxSession, from: ServiceUsageKeyWithEpoch, limit: Long): WithTx<ServiceUsageKeyWithEpochPage> {
        val query = ydbQuerySource.getQuery("yql.queries.serviceUsage.getKeysForOlderEpochsNextPage")
        val params = toGetKeysForOlderEpochsNextPageParams(from, limit)
        val page = DaoReader.toModelsWithTx(session.executeDataQueryRetryable(query, params).awaitSingle(),
            this::toKeyWithEpoch)
        return WithTx(ServiceUsageKeyWithEpochPage(keys = page.value,
            nextFrom = if (page.value.size >= limit) { page.value.last() } else { null }), page.txId)
    }

    suspend fun deleteByIdRetryable(session: YdbTxSession, id: ServiceUsageKey) {
        val query = ydbQuerySource.getQuery("yql.queries.serviceUsage.deleteById")
        val params = toKeyParams(id)
        session.executeDataQueryRetryable(query, params).awaitSingleOrNull()
    }

    suspend fun deleteByIdsRetryable(session: YdbTxSession, ids: Collection<ServiceUsageKey>) {
        if (ids.isEmpty()) {
            return
        }
        val query = ydbQuerySource.getQuery("yql.queries.serviceUsage.deleteByIds")
        val params = toKeyListParams(ids)
        session.executeDataQueryRetryable(query, params).awaitSingleOrNull()
    }

    suspend fun getKeysForOlderEpochsMultiResourceFirstPage(session: YdbTxSession,
                                                            tenantId: TenantId,
                                                            providerId: String,
                                                            resourceIds: Collection<String>,
                                                            currentEpoch: Long,
                                                            limit: Long): WithTx<ServiceUsageKeyWithEpochPage> {
        val query = ydbQuerySource
            .getQuery("yql.queries.serviceUsage.getKeysForOlderEpochsMultiResourceFirstPage")
        val params = toGetKeysForOlderEpochsMultiResourceFirstPageParams(tenantId, providerId, resourceIds,
            currentEpoch, limit)
        val page = DaoReader.toModelsWithTx(session.executeDataQueryRetryable(query, params).awaitSingle(),
            this::toKeyWithEpoch)
        return WithTx(ServiceUsageKeyWithEpochPage(keys = page.value,
            nextFrom = if (page.value.size >= limit) { page.value.last() } else { null }), page.txId)
    }

    suspend fun getKeysForOlderEpochsMultiResourceNextPage(
        session: YdbTxSession,
        from: ServiceUsageKeyWithEpoch,
        resourceIds: Collection<String>,
        currentEpoch: Long,
        limit: Long
    ): WithTx<ServiceUsageKeyWithEpochPage> {
        val sortedResourceIds = resourceIds.sortedDescending()
        val resourceIdsTail = sortedResourceIds.filter { it < from.key.resourceId }
        val (query, params) = if (resourceIdsTail.isNotEmpty()) {
            val nextQuery = ydbQuerySource
                .getQuery("yql.queries.serviceUsage.getKeysForOlderEpochsMultiResourceNextPage")
            val nextParams = toGetKeysForOlderEpochsMultiResourceNextPageParams(from, resourceIdsTail,
                currentEpoch, limit)
            Pair(nextQuery, nextParams)
        } else {
            val lastQuery = ydbQuerySource
                .getQuery("yql.queries.serviceUsage.getKeysForOlderEpochsMultiResourceLastPage")
            val lastParams = toGetKeysForOlderEpochsMultiResourceLastPageParams(from, limit)
            Pair(lastQuery, lastParams)
        }
        val page = DaoReader.toModelsWithTx(session.executeDataQueryRetryable(query, params).awaitSingle(),
            this::toKeyWithEpoch)
        return WithTx(ServiceUsageKeyWithEpochPage(keys = page.value,
            nextFrom = if (page.value.size >= limit) { page.value.last() } else { null }), page.txId)
    }

    suspend fun getAllByProviderResources(session: YdbTxSession, tenantId: TenantId,
                                          providerId: String, resourceIds: Collection<String>,
                                          perPage: Long): List<ServiceUsageModel> {
        if (resourceIds.isEmpty()) {
            return listOf()
        }
        val sortedResourceIds = resourceIds.distinct().sorted()
        return DaoPagination.getAllPages(session,
            { ydbQuerySource.getQuery("yql.queries.serviceUsage.getServiceUsageAggregationFirstPage") },
            { ydbQuerySource.getQuery("yql.queries.serviceUsage.getServiceUsageAggregationNextPage") },
            { ydbQuerySource.getQuery("yql.queries.serviceUsage.getServiceUsageAggregationLastPage") },
            { limit -> toGetByProviderResourcesFirstPageParams(tenantId, providerId, sortedResourceIds, limit) },
            { limit, lastOnPreviousPage -> toGetByProviderResourcesNextPageParams(tenantId,
                providerId, resourceIdsTail(sortedResourceIds, lastOnPreviousPage.key.resourceId), limit,
                lastOnPreviousPage.key.resourceId, lastOnPreviousPage.key.serviceId)},
            { limit, lastOnPreviousPage -> toGetByProviderResourcesLastPageParams(tenantId, providerId, limit,
                lastOnPreviousPage.key.resourceId, lastOnPreviousPage.key.serviceId)},
            { lastOnPreviousPage -> resourceIdsTail(sortedResourceIds, lastOnPreviousPage.key.resourceId).isEmpty() },
            this::toModel,
            perPage
        )
    }

    suspend fun getAllByProvider(session: YdbTxSession, tenantId: TenantId, providerId: String, perPage: Long): List<ServiceUsageModel> {
        return DaoPagination.getAllPages(session,
            { ydbQuerySource.getQuery("yql.queries.serviceUsage.getProviderServiceUsageAggregationFirstPage") },
            { ydbQuerySource.getQuery("yql.queries.serviceUsage.getProviderServiceUsageAggregationNextPage") },
            { limit -> toGetByProviderFirstPageParams(tenantId, providerId, limit) },
            { limit, lastOnPreviousPage -> toGetByProviderNextPageParams(tenantId, providerId, limit,
                lastOnPreviousPage.key.resourceId, lastOnPreviousPage.key.serviceId)},
            this::toModel,
            perPage
        )
    }

    suspend fun scanByProviderResources(session: YdbSession, tenantId: TenantId,
                                        providerId: String, resourceIds: Collection<String>,
                                        timeout: Duration): List<ServiceUsageModel> {
        if (resourceIds.isEmpty()) {
            return listOf()
        }
        val query = ydbQuerySource.getQuery("yql.queries.serviceUsage.scanServiceUsageAggregationSubset")
        val params = toScanByProviderResourcesParams(tenantId, providerId, resourceIds)
        val settings = YdbExecuteScanQuerySettings.builder().timeout(timeout).build()
        return session.executeScanQuery(query, params, settings)
            .flatMapIterable { reader -> DaoReader.toModels(reader, this::toModel) }.collectList().awaitSingle()
    }

    suspend fun scanByProvider(session: YdbSession, tenantId: TenantId, providerId: String,
                               timeout: Duration): List<ServiceUsageModel> {
        val query = ydbQuerySource.getQuery("yql.queries.serviceUsage.scanProviderServiceUsageAggregationSubset")
        val params = toScanByProviderParams(tenantId, providerId)
        val settings = YdbExecuteScanQuerySettings.builder().timeout(timeout).build()
        return session.executeScanQuery(query, params, settings)
            .flatMapIterable { reader -> DaoReader.toModels(reader, this::toModel) }.collectList().awaitSingle()
    }

    private fun toModel(reader: ResultSetReader) = ServiceUsageModel(
        key = toKey(reader),
        lastUpdate = reader.getColumn("last_update").timestamp,
        epoch = reader.getColumn("epoch").int64,
        usage = usageFieldHelper.read(reader.getColumn("usage"))!!
    )

    private fun toKey(reader: ResultSetReader) = ServiceUsageKey(
        tenantId = TenantId(reader.getColumn("tenant_id").utf8),
        serviceId = reader.getColumn("service_id").int64,
        providerId = reader.getColumn("provider_id").utf8,
        resourceId = reader.getColumn("resource_id").utf8
    )

    private fun toKeyWithEpoch(reader: ResultSetReader) = ServiceUsageKeyWithEpoch(
        key = toKey(reader),
        epoch = reader.getColumn("epoch").int64
    )

    private fun toKeyParams(id: ServiceUsageKey) = Params.of(
        "\$id", toKeyStruct(id)
    )

    private fun toKeyListParams(ids: Collection<ServiceUsageKey>) = Params.of(
        "\$ids", ListValue.of(*ids.map { toKeyStruct(it) }.toTypedArray())
    )

    private fun toKeyStruct(id: ServiceUsageKey) = StructValue.of(mapOf(
        "tenant_id" to PrimitiveValue.utf8(id.tenantId.id),
        "service_id" to PrimitiveValue.int64(id.serviceId),
        "provider_id" to PrimitiveValue.utf8(id.providerId),
        "resource_id" to PrimitiveValue.utf8(id.resourceId)
    ))

    private fun toUpsertOneParams(value: ServiceUsageModel) = Params.of(
        "\$value", toUpsertStruct(value)
    )

    private fun toUpsertManyParams(values: Collection<ServiceUsageModel>) = Params.of(
        "\$values", ListValue.of(*values.map { toUpsertStruct(it) }.toTypedArray())
    )

    private fun toUpsertStruct(value: ServiceUsageModel) = StructValue.of(
        mapOf(
            "tenant_id" to PrimitiveValue.utf8(value.key.tenantId.id),
            "service_id" to PrimitiveValue.int64(value.key.serviceId),
            "provider_id" to PrimitiveValue.utf8(value.key.providerId),
            "resource_id" to PrimitiveValue.utf8(value.key.resourceId),
            "last_update" to PrimitiveValue.timestamp(value.lastUpdate),
            "epoch" to PrimitiveValue.int64(value.epoch),
            "usage" to usageFieldHelper.write(value.usage)
        )
    )

    private fun toGetByServiceFirstPageParams(tenantId: TenantId, serviceId: Long, limit: Long) = Params
        .of("\$tenant_id", PrimitiveValue.utf8(tenantId.id),
            "\$service_id", PrimitiveValue.int64(serviceId),
            "\$limit", PrimitiveValue.uint64(limit))

    private fun toGetByServiceNextPageParams(tenantId: TenantId, serviceId: Long, limit: Long, fromProviderId: String,
                                             fromResourceId: String) = Params
        .of("\$tenant_id", PrimitiveValue.utf8(tenantId.id),
            "\$service_id", PrimitiveValue.int64(serviceId),
            "\$limit", PrimitiveValue.uint64(limit),
            "\$from_provider_id", PrimitiveValue.utf8(fromProviderId),
            "\$from_resource_id", PrimitiveValue.utf8(fromResourceId))

    private fun toGetByServicesFirstPageParams(tenantId: TenantId, serviceIds: Collection<Long>, limit: Long) = Params
        .of("\$tenant_id", PrimitiveValue.utf8(tenantId.id),
            "\$service_ids", ListValue.of(*serviceIds.map { PrimitiveValue.int64(it) }.toTypedArray()),
            "\$limit", PrimitiveValue.uint64(limit))

    private fun toGetByServicesNextPageParams(tenantId: TenantId, serviceIds: Collection<Long>, limit: Long,
                                              fromServiceId: Long, fromProviderId: String,
                                              fromResourceId: String) = Params
        .of("\$tenant_id", PrimitiveValue.utf8(tenantId.id),
            "\$service_ids", ListValue.of(*serviceIds.map { PrimitiveValue.int64(it) }.toTypedArray()),
            "\$limit", PrimitiveValue.uint64(limit),
            "\$from_service_id", PrimitiveValue.int64(fromServiceId),
            "\$from_provider_id", PrimitiveValue.utf8(fromProviderId),
            "\$from_resource_id", PrimitiveValue.utf8(fromResourceId))

    private fun toGetByServicesLastPageParams(tenantId: TenantId, limit: Long, fromServiceId: Long,
                                              fromProviderId: String, fromResourceId: String) = Params
        .of("\$tenant_id", PrimitiveValue.utf8(tenantId.id),
            "\$limit", PrimitiveValue.uint64(limit),
            "\$from_service_id", PrimitiveValue.int64(fromServiceId),
            "\$from_provider_id", PrimitiveValue.utf8(fromProviderId),
            "\$from_resource_id", PrimitiveValue.utf8(fromResourceId))

    private fun serviceIdsTail(sortedServiceIds: List<Long>, currentServiceId: Long) = sortedServiceIds
        .filter { it > currentServiceId }

    private fun resourceIdsTail(sortedResourceIds: List<String>, currentResourceId: String) = sortedResourceIds
        .filter { it > currentResourceId }

    private fun toGetKeysForOlderEpochsFirstPageParams(tenantId: TenantId, providerId: String, resourceId: String,
                                                       currentEpoch: Long, limit: Long) = Params
        .of("\$tenant_id", PrimitiveValue.utf8(tenantId.id),
            "\$provider_id", PrimitiveValue.utf8(providerId),
            "\$resource_id", PrimitiveValue.utf8(resourceId),
            "\$current_epoch", PrimitiveValue.int64(currentEpoch),
            "\$limit", PrimitiveValue.uint64(limit)
        )

    private fun toGetKeysForOlderEpochsNextPageParams(from: ServiceUsageKeyWithEpoch, limit: Long) = Params
        .of("\$tenant_id", PrimitiveValue.utf8(from.key.tenantId.id),
            "\$provider_id", PrimitiveValue.utf8(from.key.providerId),
            "\$resource_id", PrimitiveValue.utf8(from.key.resourceId),
            "\$from_epoch", PrimitiveValue.int64(from.epoch),
            "\$from_service_id", PrimitiveValue.int64(from.key.serviceId),
            "\$limit", PrimitiveValue.uint64(limit)
        )

    private fun toGetKeysForOlderEpochsMultiResourceFirstPageParams(tenantId: TenantId, providerId: String,
                                                                    resourceIds: Collection<String>, currentEpoch: Long,
                                                                    limit: Long) = Params
        .of("\$tenant_id", PrimitiveValue.utf8(tenantId.id),
            "\$provider_id", PrimitiveValue.utf8(providerId),
            "\$resource_ids", ListValue.of(*resourceIds.map { PrimitiveValue.utf8(it) }.toTypedArray()),
            "\$current_epoch", PrimitiveValue.int64(currentEpoch),
            "\$limit", PrimitiveValue.uint64(limit)
        )

    private fun toGetKeysForOlderEpochsMultiResourceNextPageParams(from: ServiceUsageKeyWithEpoch,
                                                                   resourceIds: Collection<String>, currentEpoch: Long,
                                                                   limit: Long) = Params
        .of("\$tenant_id", PrimitiveValue.utf8(from.key.tenantId.id),
            "\$provider_id", PrimitiveValue.utf8(from.key.providerId),
            "\$resource_ids", ListValue.of(*resourceIds.map { PrimitiveValue.utf8(it) }.toTypedArray()),
            "\$from_resource_id", PrimitiveValue.utf8(from.key.resourceId),
            "\$current_epoch", PrimitiveValue.int64(currentEpoch),
            "\$from_epoch", PrimitiveValue.int64(from.epoch),
            "\$from_service_id", PrimitiveValue.int64(from.key.serviceId),
            "\$limit", PrimitiveValue.uint64(limit)
        )

    private fun toGetKeysForOlderEpochsMultiResourceLastPageParams(from: ServiceUsageKeyWithEpoch,
                                                                   limit: Long) = Params
        .of("\$tenant_id", PrimitiveValue.utf8(from.key.tenantId.id),
            "\$provider_id", PrimitiveValue.utf8(from.key.providerId),
            "\$from_resource_id", PrimitiveValue.utf8(from.key.resourceId),
            "\$from_epoch", PrimitiveValue.int64(from.epoch),
            "\$from_service_id", PrimitiveValue.int64(from.key.serviceId),
            "\$limit", PrimitiveValue.uint64(limit)
        )

    private fun toGetByServiceAndProviderFirstPageParams(tenantId: TenantId, serviceId: Long, providerId: String,
                                                         limit: Long) = Params
        .of("\$tenant_id", PrimitiveValue.utf8(tenantId.id),
            "\$service_id", PrimitiveValue.int64(serviceId),
            "\$provider_id", PrimitiveValue.utf8(providerId),
            "\$limit", PrimitiveValue.uint64(limit))

    private fun toGetByServiceAndProviderNextPageParams(tenantId: TenantId, serviceId: Long, providerId: String,
                                                        limit: Long, fromResourceId: String) = Params
        .of("\$tenant_id", PrimitiveValue.utf8(tenantId.id),
            "\$service_id", PrimitiveValue.int64(serviceId),
            "\$limit", PrimitiveValue.uint64(limit),
            "\$provider_id", PrimitiveValue.utf8(providerId),
            "\$from_resource_id", PrimitiveValue.utf8(fromResourceId))

    private fun toGetByProviderResourcesFirstPageParams(tenantId: TenantId, providerId: String,
                                                        resourceIds: Collection<String>, limit: Long) = Params
        .of("\$tenant_id", PrimitiveValue.utf8(tenantId.id),
            "\$provider_id", PrimitiveValue.utf8(providerId),
            "\$resource_ids", ListValue.of(*resourceIds.map { PrimitiveValue.utf8(it) }.toTypedArray()),
            "\$limit", PrimitiveValue.uint64(limit))

    private fun toGetByProviderResourcesNextPageParams(tenantId: TenantId, providerId: String,
                                                       resourceIds: Collection<String>, limit: Long,
                                                       fromResourceId: String, fromServiceId: Long) = Params
        .of("\$tenant_id", PrimitiveValue.utf8(tenantId.id),
            "\$provider_id", PrimitiveValue.utf8(providerId),
            "\$resource_ids", ListValue.of(*resourceIds.map { PrimitiveValue.utf8(it) }.toTypedArray()),
            "\$limit", PrimitiveValue.uint64(limit),
            "\$from_resource_id", PrimitiveValue.utf8(fromResourceId),
            "\$from_service_id", PrimitiveValue.int64(fromServiceId))

    private fun toGetByProviderResourcesLastPageParams(tenantId: TenantId, providerId: String, limit: Long,
                                                       fromResourceId: String, fromServiceId: Long) = Params
        .of("\$tenant_id", PrimitiveValue.utf8(tenantId.id),
            "\$provider_id", PrimitiveValue.utf8(providerId),
            "\$limit", PrimitiveValue.uint64(limit),
            "\$from_resource_id", PrimitiveValue.utf8(fromResourceId),
            "\$from_service_id", PrimitiveValue.int64(fromServiceId))

    private fun toGetByProviderFirstPageParams(tenantId: TenantId, providerId: String, limit: Long) = Params
        .of("\$tenant_id", PrimitiveValue.utf8(tenantId.id),
            "\$provider_id", PrimitiveValue.utf8(providerId),
            "\$limit", PrimitiveValue.uint64(limit))

    private fun toGetByProviderNextPageParams(tenantId: TenantId, providerId: String, limit: Long,
                                              fromResourceId: String, fromServiceId: Long) = Params
        .of("\$tenant_id", PrimitiveValue.utf8(tenantId.id),
            "\$provider_id", PrimitiveValue.utf8(providerId),
            "\$limit", PrimitiveValue.uint64(limit),
            "\$from_resource_id", PrimitiveValue.utf8(fromResourceId),
            "\$from_service_id", PrimitiveValue.int64(fromServiceId))

    private fun toScanByProviderResourcesParams(tenantId: TenantId, providerId: String,
                                                resourceIds: Collection<String>) = Params
        .of("\$tenant_id", PrimitiveValue.utf8(tenantId.id),
            "\$provider_id", PrimitiveValue.utf8(providerId),
            "\$resource_ids", ListValue.of(*resourceIds.map { PrimitiveValue.utf8(it) }.toTypedArray()))

    private fun toScanByProviderParams(tenantId: TenantId, providerId: String) = Params
        .of("\$tenant_id", PrimitiveValue.utf8(tenantId.id),
            "\$provider_id", PrimitiveValue.utf8(providerId))

}
