package ru.yandex.intranet.d.dao.usage

import com.yandex.ydb.table.query.Params
import com.yandex.ydb.table.result.ResultSetReader
import com.yandex.ydb.table.values.ListValue
import com.yandex.ydb.table.values.PrimitiveValue
import com.yandex.ydb.table.values.StructValue
import kotlinx.coroutines.reactor.awaitSingle
import kotlinx.coroutines.reactor.awaitSingleOrNull
import org.springframework.stereotype.Component
import ru.yandex.intranet.d.dao.DaoReader
import ru.yandex.intranet.d.datasource.impl.YdbQuerySource
import ru.yandex.intranet.d.datasource.model.YdbTxSession
import ru.yandex.intranet.d.model.TenantId
import ru.yandex.intranet.d.model.usage.UsageEpochKey
import ru.yandex.intranet.d.model.usage.UsageEpochModel

/**
 * Usage epochs DAO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
class UsageEpochsDao(private val ydbQuerySource: YdbQuerySource) {

    suspend fun getById(session: YdbTxSession, id: UsageEpochKey): UsageEpochModel? {
        val query = ydbQuerySource.getQuery("yql.queries.usageEpochs.getById")
        val params = toKeyParams(id)
        return DaoReader.toModel(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toModel)
    }

    suspend fun getByIds(session: YdbTxSession,
                         ids: Collection<UsageEpochKey>): List<UsageEpochModel> {
        if (ids.isEmpty()) {
            return listOf()
        }
        return ids.distinct().chunked(1000).map {
            val query = ydbQuerySource.getQuery("yql.queries.usageEpochs.getByIds")
            val params = toKeyListParams(it)
            DaoReader.toModels(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toModel)
        }.flatten()
    }

    suspend fun upsertOneRetryable(session: YdbTxSession, value: UsageEpochModel) {
        val query = ydbQuerySource.getQuery("yql.queries.usageEpochs.upsertOne")
        val params = toUpsertOneParams(value)
        session.executeDataQueryRetryable(query, params).awaitSingleOrNull()
    }

    suspend fun upsertManyRetryable(session: YdbTxSession, values: Collection<UsageEpochModel>) {
        if (values.isEmpty()) {
            return
        }
        val query = ydbQuerySource.getQuery("yql.queries.usageEpochs.upsertMany")
        val params = toUpsertManyParams(values)
        session.executeDataQueryRetryable(query, params).awaitSingleOrNull()
    }

    private fun toModel(reader: ResultSetReader) = UsageEpochModel(
        key = toKey(reader),
        epoch = reader.getColumn("epoch").int64
    )

    private fun toKey(reader: ResultSetReader) = UsageEpochKey(
        tenantId = TenantId(reader.getColumn("tenant_id").utf8),
        providerId = reader.getColumn("provider_id").utf8,
        resourceId = reader.getColumn("resource_id").utf8
    )

    private fun toKeyParams(id: UsageEpochKey) = Params.of(
        "\$id", toKeyStruct(id)
    )

    private fun toKeyListParams(ids: Collection<UsageEpochKey>) = Params.of(
        "\$ids", ListValue.of(*ids.map { toKeyStruct(it) }.toTypedArray())
    )

    private fun toKeyStruct(id: UsageEpochKey) = StructValue.of(mapOf(
        "tenant_id" to PrimitiveValue.utf8(id.tenantId.id),
        "provider_id" to PrimitiveValue.utf8(id.providerId),
        "resource_id" to PrimitiveValue.utf8(id.resourceId)
    ))

    private fun toUpsertOneParams(value: UsageEpochModel) = Params.of(
        "\$value", toUpsertStruct(value)
    )

    private fun toUpsertManyParams(values: Collection<UsageEpochModel>) = Params.of(
        "\$values", ListValue.of(*values.map { toUpsertStruct(it) }.toTypedArray())
    )

    private fun toUpsertStruct(value: UsageEpochModel) = StructValue.of(
        mapOf(
            "tenant_id" to PrimitiveValue.utf8(value.key.tenantId.id),
            "provider_id" to PrimitiveValue.utf8(value.key.providerId),
            "resource_id" to PrimitiveValue.utf8(value.key.resourceId),
            "epoch" to PrimitiveValue.int64(value.epoch)
        )
    )

}
