package ru.yandex.intranet.d.dao.users;

import java.time.LocalDate;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Optional;

import com.yandex.ydb.table.query.DataQueryResult;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.result.ValueReader;
import com.yandex.ydb.table.values.PrimitiveValue;
import com.yandex.ydb.table.values.StructValue;
import com.yandex.ydb.table.values.Value;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.datasource.Ydb;
import ru.yandex.intranet.d.datasource.impl.YdbQuerySource;
import ru.yandex.intranet.d.datasource.model.YdbReadTableSettings;
import ru.yandex.intranet.d.datasource.model.YdbSession;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.model.users.AbcUserModel;

/**
 * DAO пользователей в таблице ABC.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 */
@Component
public class AbcIntranetStaffDao {
    public static final String INTRANET_STAFF_TABLE_NAME = "public_intranet_staff";

    public enum Fields {
        ID, FIRST_NAME, LAST_NAME, LOGIN, IS_DISMISSED, GENDER, WORK_EMAIL, FIRST_NAME_EN, LAST_NAME_EN, LANG_UI, TZ,
        UID, IS_ROBOT, AFFILIATION
    }

    private final YdbQuerySource ydbQuerySource;

    public AbcIntranetStaffDao(
            YdbQuerySource ydbQuerySource
    ) {
        this.ydbQuerySource = ydbQuerySource;
    }

    public Flux<AbcUserModel> getAllRows(YdbSession session, Fields... fields) {
        return session
                .readTable(
                        ydbQuerySource.preprocessAbcTableName(INTRANET_STAFF_TABLE_NAME),
                        toReadFieldsSettings(fields)
                )
                .flatMapIterable(this::readRowsToModel);
    }

    public Mono<Optional<Long>> getMaxId(YdbTxSession session) {
        String query = ydbQuerySource.getAbcQuery("yql.queries.intranet.staff.getMaxId");
        Params params = Params.empty();
        return session.executeDataQueryRetryable(query, params).map(this::toMaxId);
    }

    public Mono<Void> upsertOneRetryable(YdbTxSession session, AbcUserModel user) {
        String query = ydbQuerySource.getAbcQuery("yql.queries.intranet.staff.upsertOne");
        Map<String, Value> fields = prepareUserFields(user);
        Params params = Params.of("$user", StructValue.of(fields));
        return session.executeDataQueryRetryable(query, params).then();
    }

    public YdbReadTableSettings toReadFieldsSettings(Fields... fields) {
        YdbReadTableSettings.Builder settings = YdbReadTableSettings.builder()
                .ordered(true);
        for (Fields field : fields) {
            settings.addColumn(field.name().toLowerCase());
        }
        return settings.build();
    }

    private Iterable<AbcUserModel> readRowsToModel(ResultSetReader reader) {
        return () -> new Iterator<>() {
            @Override
            public boolean hasNext() {
                return reader.next();
            }

            @Override
            public AbcUserModel next() {
                return readOneRowToModel(reader);
            }
        };
    }

    @SuppressWarnings("MethodLength")
    private AbcUserModel readOneRowToModel(ResultSetReader reader) {
        AbcUserModel.Builder result = AbcUserModel.newBuilder();
        final int columnCount = reader.getColumnCount();
        for (int i = 0; i < columnCount; i++) {
            ValueReader value = reader.getColumn(i);
            String columnName = reader.getColumnName(i);
            try {
                Fields field = Fields.valueOf(columnName.toUpperCase());
                switch (field) {
                    case ID:
                        result.setId(value.getInt64());
                        break;
                    case FIRST_NAME:
                        result.setFirstName(value.getUtf8());
                        break;
                    case LAST_NAME:
                        result.setLastName(value.getUtf8());
                        break;
                    case LOGIN:
                        result.setLogin(value.getUtf8());
                        break;
                    case IS_DISMISSED:
                        result.setIsDismissed(value.getBool());
                        break;
                    case GENDER:
                        result.setGender(value.getUtf8().charAt(0));
                        break;
                    case WORK_EMAIL:
                        result.setWorkEmail(value.getUtf8());
                        break;
                    case FIRST_NAME_EN:
                        result.setFirstNameEn(value.getUtf8());
                        break;
                    case LAST_NAME_EN:
                        result.setLastNameEn(value.getUtf8());
                        break;
                    case LANG_UI:
                        result.setLangUi(value.getUtf8());
                        break;
                    case TZ:
                        result.setTz(value.getUtf8());
                        break;
                    case UID:
                        result.setUid(value.getUtf8());
                        break;
                    case IS_ROBOT:
                        result.setIsRobot(value.getBool());
                        break;
                    case AFFILIATION:
                        result.setAffiliation(value.getUtf8());
                        break;
                    default:
                        throw new IllegalArgumentException();
                }
            } catch (IllegalArgumentException e) {
                throw new IllegalArgumentException(
                        "Unsupported field " + columnName + "of table " + INTRANET_STAFF_TABLE_NAME
                );
            }
        }
        return result.build();
    }

    private Optional<Long> toMaxId(DataQueryResult result) {
        if (result.isEmpty()) {
            return Optional.empty();
        }
        if (result.getResultSetCount() > 1) {
            throw new IllegalStateException("Too many result sets");
        }
        ResultSetReader reader = result.getResultSet(0);
        if (!reader.next()) {
            return Optional.empty();
        }
        if (reader.getRowCount() > 1) {
            throw new IllegalStateException("Non unique max_id");
        }
        return Optional.ofNullable(Ydb.int64OrNull(reader.getColumn("max_id")));
    }

    private Map<String, Value> prepareUserFields(AbcUserModel user) {
        Map<String, Value> fields = new HashMap<>();
        fields.put("id", PrimitiveValue.int64(user.getId()));
        fields.put("affiliation", PrimitiveValue.utf8(user.getAffiliation()));
        fields.put("first_name", PrimitiveValue.utf8(user.getFirstName()));
        fields.put("first_name_en", PrimitiveValue.utf8(user.getFirstNameEn()));
        fields.put("gender", PrimitiveValue.utf8(String.valueOf(user.getGender())));
        fields.put("is_dismissed", PrimitiveValue.bool(user.isDismissed()));
        fields.put("is_robot", PrimitiveValue.bool(user.getRobot()));
        fields.put("lang_ui", PrimitiveValue.utf8(user.getLangUi()));
        fields.put("last_name", PrimitiveValue.utf8(user.getLastName()));
        fields.put("last_name_en", PrimitiveValue.utf8(user.getLastNameEn()));
        fields.put("login", PrimitiveValue.utf8(user.getLogin()));
        fields.put("tz", PrimitiveValue.utf8(user.getTz()));
        fields.put("uid", PrimitiveValue.utf8(user.getUid()));
        fields.put("work_email", PrimitiveValue.utf8(user.getWorkEmail()));
        return fields;
    }

    private String toDateTime(LocalDate value) {
        if (value == null) {
            return null;
        }
        return value.atStartOfDay().format(DateTimeFormatter.ofPattern("yyyy-MM-dd' 'HH:mm:ss' +0000 UTC'"));
    }

    private String toDateTime(ZonedDateTime value) {
        if (value == null) {
            return null;
        }
        return value.format(DateTimeFormatter.ofPattern("yyyy-MM-dd' 'HH:mm:ss' 'z"));
    }

}
