package ru.yandex.intranet.d.dao.users;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.yandex.ydb.table.query.DataQueryResult;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.result.ValueReader;
import com.yandex.ydb.table.values.ListValue;
import com.yandex.ydb.table.values.PrimitiveValue;
import com.yandex.ydb.table.values.StructValue;
import com.yandex.ydb.table.values.Value;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.datasource.Ydb;
import ru.yandex.intranet.d.datasource.impl.YdbQuerySource;
import ru.yandex.intranet.d.datasource.model.YdbReadTableSettings;
import ru.yandex.intranet.d.datasource.model.YdbSession;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.model.users.AbcServiceMemberModel;
import ru.yandex.intranet.d.model.users.AbcServiceMemberState;

/**
 * DAO ролей пользователей в таблице public_services_servicemember.
 *
 * @author Ruslan Kadriev <aqru@yandex-team.ru>
 * @since 19-11-2020
 */
@Component
public class AbcServiceMemberDao {
    private static final String SERVICE_SERVICEMEMBER_TABLE_NAME = "public_services_servicemember";

    public enum Fields {
        ID, STAFF_ID, SERVICE_ID, ROLE_ID, STATE
    }

    private final YdbQuerySource ydbQuerySource;

    public AbcServiceMemberDao(YdbQuerySource ydbQuerySource) {
        this.ydbQuerySource = ydbQuerySource;
    }

    public Flux<AbcServiceMemberModel> getAllRows(YdbSession session, Fields... fields) {
        return session
                .readTable(
                        ydbQuerySource.preprocessAbcTableName(SERVICE_SERVICEMEMBER_TABLE_NAME),
                        toReadFieldsSettings(fields)
                )
                .flatMapIterable(this::readRowsToModel);
    }

    public Mono<Optional<Long>> getMaxId(YdbTxSession session) {
        String query = ydbQuerySource.getAbcQuery("yql.queries.servicemember.getMaxId");
        Params params = Params.empty();
        return session.executeDataQueryRetryable(query, params).map(this::toMaxId);
    }

    public Mono<Void> upsertManyRetryable(YdbTxSession session, List<AbcServiceMemberModel> serviceMembers) {
        if (serviceMembers.isEmpty()) {
            return Mono.empty();
        }
        String query = ydbQuerySource.getAbcQuery("yql.queries.servicemember.upsertMany");
        Params params = Params.of("$servicemembers", ListValue.of(serviceMembers.stream().map(serviceMember ->
                StructValue.of(prepareServiceMemberFields(serviceMember))).toArray(StructValue[]::new)));
        return session.executeDataQueryRetryable(query, params).then();
    }

    // TODO support status in model
    public Mono<Void> depriveRoleRetryable(YdbTxSession session, Long id) {
        String query = ydbQuerySource.getAbcQuery("yql.queries.servicemember.depriveRole");
        Params params = Params.of("$id", PrimitiveValue.int64(id),
                "$role", Ydb.nullableUtf8("deprived"));
        return session.executeDataQueryRetryable(query, params).then();
    }

    public Mono<List<AbcServiceMemberModel>> getByUsersAndRoles(YdbTxSession session, Set<Long> staffIds,
                                                                Set<Long> roleIds) {
        return getByUsersAndRoles(session, staffIds, roleIds, Ydb.MAX_RESPONSE_ROWS);
    }

    public Mono<List<AbcServiceMemberModel>> getByServicesAndRoles(YdbTxSession session, Set<Long> serviceIds,
                                                                   Set<Long> roleIds) {
        return getByServicesAndRoles(session, serviceIds, roleIds, Ydb.MAX_RESPONSE_ROWS);
    }

    Mono<List<AbcServiceMemberModel>> getByUsersAndRoles(YdbTxSession session, Set<Long> staffIds,
                                                         Set<Long> roleIds, long limit) {
        if (staffIds.isEmpty() || roleIds.isEmpty()) {
            return Mono.just(List.of());
        }
        List<Long> sortedStaffIds = staffIds.stream().sorted().collect(Collectors.toList());
        List<Long> sortedRoleIds = roleIds.stream().sorted().collect(Collectors.toList());
        String firstPageQuery = ydbQuerySource.getAbcQuery("yql.queries.servicemember.getByStaffRoleIdsFirstPage");
        ListValue firstPageStaffIdsParam = ListValue.of(sortedStaffIds.stream()
                .map(PrimitiveValue::int64)
                .toArray(PrimitiveValue[]::new));
        ListValue firstPageRoleIdsParam = ListValue.of(sortedRoleIds.stream()
                .map(PrimitiveValue::int64)
                .toArray(PrimitiveValue[]::new));
        HashMap<String, Value<?>> firstPageParamsMap = new HashMap<>();
        firstPageParamsMap.put("$staff_ids", firstPageStaffIdsParam);
        firstPageParamsMap.put("$roles_ids", firstPageRoleIdsParam);
        firstPageParamsMap.put("$limit", PrimitiveValue.uint64(limit));
        Params firstPageParams = Params.copyOf(firstPageParamsMap);
        return session.executeDataQueryRetryable(firstPageQuery, firstPageParams).flatMap(firstPageResult -> {
            List<AbcServiceMemberModel> firstPageModels = toModels(firstPageResult);
            if (!firstPageResult.getResultSet(0).isTruncated() && firstPageModels.size() < limit) {
                return Mono.just(firstPageModels);
            }
            return getByUsersAndRolesNextPage(session, sortedStaffIds, sortedRoleIds,
                    firstPageModels.get(firstPageModels.size() - 1), limit).expand(tuple -> {
                if (!tuple.getT2() && tuple.getT1().size() < limit) {
                    return Mono.empty();
                } else {
                    return getByUsersAndRolesNextPage(session, sortedStaffIds, sortedRoleIds,
                            tuple.getT1().get(tuple.getT1().size() - 1), limit);
                }
            }).map(Tuple2::getT1).reduce(firstPageModels, (l, r) -> Stream.concat(l.stream(), r.stream())
                    .collect(Collectors.toList()));
        });
    }

    Mono<List<AbcServiceMemberModel>> getByServicesAndRoles(YdbTxSession session, Set<Long> serviceIds,
                                                            Set<Long> roleIds, long limit) {
        if (serviceIds.isEmpty() || roleIds.isEmpty()) {
            return Mono.just(List.of());
        }
        List<Long> sortedServiceIds = serviceIds.stream().sorted().collect(Collectors.toList());
        List<Long> sortedRoleIds = roleIds.stream().sorted().collect(Collectors.toList());
        String firstPageQuery = ydbQuerySource.getAbcQuery("yql.queries.servicemember.getByServiceRoleIdsFirstPage");
        ListValue firstPageServiceIdsParam = ListValue.of(sortedServiceIds.stream()
                .map(PrimitiveValue::int64)
                .toArray(PrimitiveValue[]::new));
        ListValue firstPageRoleIdsParam = ListValue.of(sortedRoleIds.stream()
                .map(PrimitiveValue::int64)
                .toArray(PrimitiveValue[]::new));
        HashMap<String, Value<?>> firstPageParamsMap = new HashMap<>();
        firstPageParamsMap.put("$service_ids", firstPageServiceIdsParam);
        firstPageParamsMap.put("$roles_ids", firstPageRoleIdsParam);
        firstPageParamsMap.put("$limit", PrimitiveValue.uint64(limit));
        Params firstPageParams = Params.copyOf(firstPageParamsMap);
        return session.executeDataQueryRetryable(firstPageQuery, firstPageParams).flatMap(firstPageResult -> {
            List<AbcServiceMemberModel> firstPageModels = toModels(firstPageResult);
            if (!firstPageResult.getResultSet(0).isTruncated() && firstPageModels.size() < limit) {
                return Mono.just(firstPageModels);
            }
            return getByServicesAndRolesNextPage(session, sortedServiceIds, sortedRoleIds,
                    firstPageModels.get(firstPageModels.size() - 1), limit).expand(tuple -> {
                if (!tuple.getT2() && tuple.getT1().size() < limit) {
                    return Mono.empty();
                } else {
                    return getByServicesAndRolesNextPage(session, sortedServiceIds, sortedRoleIds,
                            tuple.getT1().get(tuple.getT1().size() - 1), limit);
                }
            }).map(Tuple2::getT1).reduce(firstPageModels, (l, r) -> Stream.concat(l.stream(), r.stream())
                    .collect(Collectors.toList()));
        });
    }

    private Mono<Tuple2<List<AbcServiceMemberModel>, Boolean>> getByUsersAndRolesNextPage(
            YdbTxSession session, List<Long> sortedStaffIds, List<Long> sortedRoleIds,
            AbcServiceMemberModel from, long limit) {
        List<Long> remainingStaffIds = sortedStaffIds.stream().filter(id -> id.compareTo(from.getStaffId()) > 0)
                .toList();
        List<Long> remainingRoleIds = sortedRoleIds.stream().filter(id -> id.compareTo(from.getRoleId()) > 0).toList();
        String nextPageQuery;
        Map<String, Value<?>> nextPageParamsMap = new HashMap<>();
        nextPageParamsMap.put("$from_staff_id", PrimitiveValue.int64(from.getStaffId()));
        nextPageParamsMap.put("$from_role_id", PrimitiveValue.int64(from.getRoleId()));
        nextPageParamsMap.put("$from_id", PrimitiveValue.int64(from.getId()));
        nextPageParamsMap.put("$limit", PrimitiveValue.uint64(limit));
        ListValue roleIdsParam = ListValue.of(sortedRoleIds.stream()
                .map(PrimitiveValue::int64)
                .toArray(PrimitiveValue[]::new));
        if (remainingStaffIds.isEmpty() && remainingRoleIds.isEmpty()) {
            nextPageQuery = ydbQuerySource
                    .getAbcQuery("yql.queries.servicemember.getByStaffRoleIdsNextPagesLast");
        } else if (remainingStaffIds.isEmpty() && !remainingRoleIds.isEmpty()) {
            nextPageQuery = ydbQuerySource
                    .getAbcQuery("yql.queries.servicemember.getByStaffRoleIdsNextPagesLastStaffId");
        } else if (!remainingStaffIds.isEmpty() && remainingRoleIds.isEmpty()) {
            nextPageQuery = ydbQuerySource
                    .getAbcQuery("yql.queries.servicemember.getByStaffRoleIdsNextPagesLastRoleId");
            nextPageParamsMap.put("$roles_ids", roleIdsParam);
        } else {
            nextPageQuery = ydbQuerySource
                    .getAbcQuery("yql.queries.servicemember.getByStaffRoleIdsNextPagesFull");
            nextPageParamsMap.put("$roles_ids", roleIdsParam);
        }
        if (!remainingRoleIds.isEmpty()) {
            ListValue remainingRoleIdsParam = ListValue.of(remainingRoleIds.stream()
                    .map(PrimitiveValue::int64)
                    .toArray(PrimitiveValue[]::new));
            nextPageParamsMap.put("$from_roles_ids", remainingRoleIdsParam);
        }
        if (!remainingStaffIds.isEmpty()) {
            ListValue remainingStaffIdsParam = ListValue.of(remainingStaffIds.stream()
                    .map(PrimitiveValue::int64)
                    .toArray(PrimitiveValue[]::new));
            nextPageParamsMap.put("$from_staff_ids", remainingStaffIdsParam);
        }
        Params nextPageParams = Params.copyOf(nextPageParamsMap);
        return session.executeDataQueryRetryable(nextPageQuery, nextPageParams).map(nextPageResult -> {
            List<AbcServiceMemberModel> nextPageModels = toModels(nextPageResult);
            return Tuples.of(nextPageModels, nextPageResult.getResultSet(0).isTruncated());
        });
    }

    private Mono<Tuple2<List<AbcServiceMemberModel>, Boolean>> getByServicesAndRolesNextPage(
            YdbTxSession session, List<Long> sortedServiceIds, List<Long> sortedRoleIds,
            AbcServiceMemberModel from, long limit) {
        List<Long> remainingServiceIds = sortedServiceIds.stream().filter(id -> id.compareTo(from.getServiceId()) > 0)
                .toList();
        List<Long> remainingRoleIds = sortedRoleIds.stream().filter(id -> id.compareTo(from.getRoleId()) > 0).toList();
        String nextPageQuery;
        Map<String, Value<?>> nextPageParamsMap = new HashMap<>();
        nextPageParamsMap.put("$from_service_id", PrimitiveValue.int64(from.getServiceId()));
        nextPageParamsMap.put("$from_role_id", PrimitiveValue.int64(from.getRoleId()));
        nextPageParamsMap.put("$from_id", PrimitiveValue.int64(from.getId()));
        nextPageParamsMap.put("$limit", PrimitiveValue.uint64(limit));
        ListValue roleIdsParam = ListValue.of(sortedRoleIds.stream()
                .map(PrimitiveValue::int64)
                .toArray(PrimitiveValue[]::new));
        if (remainingServiceIds.isEmpty() && remainingRoleIds.isEmpty()) {
            nextPageQuery = ydbQuerySource
                    .getAbcQuery("yql.queries.servicemember.getByServiceRoleIdsNextPagesLast");
        } else if (remainingServiceIds.isEmpty() && !remainingRoleIds.isEmpty()) {
            nextPageQuery = ydbQuerySource
                    .getAbcQuery("yql.queries.servicemember.getByServiceRoleIdsNextPagesLastServiceId");
        } else if (!remainingServiceIds.isEmpty() && remainingRoleIds.isEmpty()) {
            nextPageQuery = ydbQuerySource
                    .getAbcQuery("yql.queries.servicemember.getByServiceRoleIdsNextPagesLastRoleId");
            nextPageParamsMap.put("$roles_ids", roleIdsParam);
        } else {
            nextPageQuery = ydbQuerySource
                    .getAbcQuery("yql.queries.servicemember.getByServiceRoleIdsNextPagesFull");
            nextPageParamsMap.put("$roles_ids", roleIdsParam);
        }
        if (!remainingRoleIds.isEmpty()) {
            ListValue remainingRoleIdsParam = ListValue.of(remainingRoleIds.stream()
                    .map(PrimitiveValue::int64)
                    .toArray(PrimitiveValue[]::new));
            nextPageParamsMap.put("$from_roles_ids", remainingRoleIdsParam);
        }
        if (!remainingServiceIds.isEmpty()) {
            ListValue remainingServiceIdsParam = ListValue.of(remainingServiceIds.stream()
                    .map(PrimitiveValue::int64)
                    .toArray(PrimitiveValue[]::new));
            nextPageParamsMap.put("$from_service_ids", remainingServiceIdsParam);
        }
        Params nextPageParams = Params.copyOf(nextPageParamsMap);
        return session.executeDataQueryRetryable(nextPageQuery, nextPageParams).map(nextPageResult -> {
            List<AbcServiceMemberModel> nextPageModels = toModels(nextPageResult);
            return Tuples.of(nextPageModels, nextPageResult.getResultSet(0).isTruncated());
        });
    }

    private List<AbcServiceMemberModel> toModels(DataQueryResult result) {
        if (result.isEmpty()) {
            return List.of();
        }
        if (result.getResultSetCount() > 1) {
            throw new IllegalStateException("Too many result sets");
        }
        ResultSetReader reader = result.getResultSet(0);
        List<AbcServiceMemberModel> models = new ArrayList<>();
        while (reader.next()) {
            models.add(readOneRowToModel(reader));
        }
        return models;
    }

    public YdbReadTableSettings toReadFieldsSettings(Fields... fields) {
        YdbReadTableSettings.Builder settings = YdbReadTableSettings.builder()
                .ordered(true);
        for (Fields field : fields) {
            settings.addColumn(field.name().toLowerCase());
        }
        return settings.build();
    }

    private Iterable<AbcServiceMemberModel> readRowsToModel(ResultSetReader reader) {
        return () -> new Iterator<>() {
            @Override
            public boolean hasNext() {
                return reader.next();
            }

            @Override
            public AbcServiceMemberModel next() {
                return readOneRowToModel(reader);
            }
        };
    }

    private AbcServiceMemberModel readOneRowToModel(ResultSetReader reader) {
        AbcServiceMemberModel.Builder result = AbcServiceMemberModel.newBuilder();
        final int columnCount = reader.getColumnCount();
        for (int i = 0; i < columnCount; i++) {
            ValueReader value = reader.getColumn(i);
            String columnName = reader.getColumnName(i);
            try {
                Fields field = Fields.valueOf(columnName.toUpperCase());
                switch (field) {
                    case ID -> result.id(value.getInt64());
                    case STAFF_ID -> result.staffId(value.getInt64());
                    case SERVICE_ID -> result.serviceId(value.getInt64());
                    case ROLE_ID -> result.roleId(value.getInt64());
                    case STATE -> result.state(AbcServiceMemberState.fromString(value.getUtf8()));
                    default -> throw new IllegalArgumentException();
                }
            } catch (IllegalArgumentException e) {
                throw new IllegalArgumentException(
                        "Unsupported field " + columnName + "of table " + SERVICE_SERVICEMEMBER_TABLE_NAME
                );
            }
        }
        return result.build();
    }

    private Optional<Long> toMaxId(DataQueryResult result) {
        if (result.isEmpty()) {
            return Optional.empty();
        }
        if (result.getResultSetCount() > 1) {
            throw new IllegalStateException("Too many result sets");
        }
        ResultSetReader reader = result.getResultSet(0);
        if (!reader.next()) {
            return Optional.empty();
        }
        if (reader.getRowCount() > 1) {
            throw new IllegalStateException("Non unique max_id");
        }
        return Optional.ofNullable(Ydb.int64OrNull(reader.getColumn("max_id")));
    }

    private Map<String, Value> prepareServiceMemberFields(AbcServiceMemberModel serviceMember) {
        Map<String, Value> fields = new HashMap<>();
        fields.put("id", PrimitiveValue.int64(serviceMember.getId()));
        fields.put("staff_id", PrimitiveValue.int64(serviceMember.getStaffId()));
        fields.put("service_id", PrimitiveValue.int64(serviceMember.getServiceId()));
        fields.put("role_id", PrimitiveValue.int64(serviceMember.getRoleId()));
        fields.put("state", Ydb.nullableUtf8(serviceMember.getState().toString()));
        return fields;
    }

}

