package ru.yandex.intranet.d.dao.users;

import java.io.UncheckedIOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectReader;
import com.fasterxml.jackson.databind.ObjectWriter;
import com.yandex.ydb.table.query.DataQueryResult;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.values.ListType;
import com.yandex.ydb.table.values.ListValue;
import com.yandex.ydb.table.values.OptionalValue;
import com.yandex.ydb.table.values.PrimitiveType;
import com.yandex.ydb.table.values.PrimitiveValue;
import com.yandex.ydb.table.values.StructValue;
import com.yandex.ydb.table.values.TupleType;
import com.yandex.ydb.table.values.TupleValue;
import com.yandex.ydb.table.values.Value;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;

import ru.yandex.intranet.d.datasource.Ydb;
import ru.yandex.intranet.d.datasource.impl.YdbQuerySource;
import ru.yandex.intranet.d.datasource.model.WithTxId;
import ru.yandex.intranet.d.datasource.model.YdbDataQuery;
import ru.yandex.intranet.d.datasource.model.YdbSession;
import ru.yandex.intranet.d.datasource.model.YdbTxDataQuery;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.users.StaffAffiliation;
import ru.yandex.intranet.d.model.users.UserModel;
import ru.yandex.intranet.d.model.users.UserServiceRoles;
import ru.yandex.intranet.d.util.ObjectMapperHolder;

/**
 * Users DAO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class UsersDao {

    private final YdbQuerySource ydbQuerySource;
    private final ObjectReader rolesReader;
    private final ObjectWriter rolesWriter;

    public UsersDao(
            YdbQuerySource ydbQuerySource,
            @Qualifier("ydbJsonObjectMapper") ObjectMapperHolder objectMapper
    ) {
        this.ydbQuerySource = ydbQuerySource;
        this.rolesReader = objectMapper.getObjectMapper().readerFor(
                new TypeReference<Map<UserServiceRoles, Set<Long>>>() { });
        this.rolesWriter = objectMapper.getObjectMapper().writerFor(
                new TypeReference<Map<UserServiceRoles, Set<Long>>>() { });
    }

    public Mono<Optional<UserModel>> getById(YdbTxSession session, String id, TenantId tenantId) {
        String query = ydbQuerySource.getQuery("yql.queries.users.getOneById");
        final Params params = Params.of("$id", PrimitiveValue.utf8(id),
                "$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
        return session.executeDataQueryRetryable(query, params).map(this::toUser);
    }

    public Mono<WithTxId<Optional<UserModel>>> getByIdTx(YdbTxSession session, String id, TenantId tenantId) {
        String query = ydbQuerySource.getQuery("yql.queries.users.getOneById");
        final Params params = Params.of("$id", PrimitiveValue.utf8(id),
                "$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
        return session.executeDataQueryRetryable(query, params)
                .map(r -> new WithTxId<>(toUser(r), r.getTxId()));
    }

    public Mono<Optional<UserModel>> getByPassportUid(YdbTxSession session, String passportUid,
                                                      TenantId tenantId) {
        String query = ydbQuerySource.getQuery("yql.queries.users.getOneByPassportUid");
        final Params params = Params.of("$passport_uid", PrimitiveValue.utf8(passportUid),
                "$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
        return session.executeDataQueryRetryable(query, params).map(this::toUser);
    }

    public Mono<WithTxId<Optional<UserModel>>> getByPassportUidTx(YdbTxSession session, String passportUid,
                                                      TenantId tenantId) {
        String query = ydbQuerySource.getQuery("yql.queries.users.getOneByPassportUid");
        final Params params = Params.of("$passport_uid", PrimitiveValue.utf8(passportUid),
                "$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
        return session.executeDataQueryRetryable(query, params)
                .map(r -> new WithTxId<>(toUser(r), r.getTxId()));
    }

    public Mono<Optional<UserModel>> getByPassportLogin(YdbTxSession session, String passportLogin,
                                                        TenantId tenantId) {
        String query = ydbQuerySource.getQuery("yql.queries.users.getOneByPassportLogin");
        final Params params = Params.of("$passport_login", PrimitiveValue.utf8(passportLogin),
                "$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
        return session.executeDataQueryRetryable(query, params).map(this::toUser);
    }

    public Mono<Optional<UserModel>> getByStaffId(YdbTxSession session, long staffId, TenantId tenantId) {
        String query = ydbQuerySource.getQuery("yql.queries.users.getOneByStaffId");
        final Params params = Params.of("$staff_id", PrimitiveValue.int64(staffId),
                "$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
        return session.executeDataQueryRetryable(query, params).map(this::toUser);
    }

    public Mono<List<UserModel>> getByIds(YdbTxSession session, List<Tuple2<String, TenantId>> ids) {
        if (ids.isEmpty()) {
            return Mono.just(List.of());
        }
        String query = ydbQuerySource.getQuery("yql.queries.users.getByIds");
        ListValue idsParam = ListValue.of(ids.stream().map(id -> TupleValue.of(PrimitiveValue.utf8(id.getT1()),
                PrimitiveValue.utf8(id.getT2().getId()))).toArray(TupleValue[]::new));
        Params params = Params.of("$ids", idsParam);
        return session.executeDataQueryRetryable(query, params).map(this::toUsers);
    }

    public Mono<List<UserModel>> getByPassportUids(YdbTxSession session,
                                                   List<Tuple2<String, TenantId>> passportUids) {
        if (passportUids.isEmpty()) {
            return Mono.just(List.of());
        }
        String query = ydbQuerySource.getQuery("yql.queries.users.getByPassportUids");
        ListValue idsParam = ListValue.of(passportUids.stream()
                .map(id -> TupleValue.of(PrimitiveValue.utf8(id.getT1()),
                        PrimitiveValue.utf8(id.getT2().getId()))).toArray(TupleValue[]::new));
        Params params = Params.of("$ids", idsParam);
        return session.executeDataQueryRetryable(query, params).map(this::toUsers);
    }

    public Mono<WithTxId<List<UserModel>>> getByPassportUidsTx(YdbTxSession session,
                                                               List<Tuple2<String, TenantId>> passportUids) {
        if (passportUids.isEmpty()) {
            return Mono.error(new IllegalArgumentException("At least one passportUid is required"));
        }
        String query = ydbQuerySource.getQuery("yql.queries.users.getByPassportUids");
        ListValue idsParam = ListValue.of(passportUids.stream()
                .map(id -> TupleValue.of(PrimitiveValue.utf8(id.getT1()),
                        PrimitiveValue.utf8(id.getT2().getId()))).toArray(TupleValue[]::new));
        Params params = Params.of("$ids", idsParam);
        return session.executeDataQueryRetryable(query, params)
                .map(r -> new WithTxId<>(toUsers(r), r.getTxId()));
    }

    public Mono<List<UserModel>> getByPassportLogins(YdbTxSession session,
                                                     List<Tuple2<String, TenantId>> passportLogins) {
        if (passportLogins.isEmpty()) {
            return Mono.just(List.of());
        }
        String query = ydbQuerySource.getQuery("yql.queries.users.getByPassportLogins");
        ListValue idsParam = ListValue.of(passportLogins.stream()
                .map(id -> TupleValue.of(PrimitiveValue.utf8(id.getT1()),
                        PrimitiveValue.utf8(id.getT2().getId()))).toArray(TupleValue[]::new));
        Params params = Params.of("$ids", idsParam);
        return session.executeDataQueryRetryable(query, params).map(this::toUsers);
    }

    public Mono<List<UserModel>> getByStaffIds(YdbTxSession session, List<Tuple2<Long, TenantId>> staffIds) {
        if (staffIds.isEmpty()) {
            return Mono.just(List.of());
        }
        String query = ydbQuerySource.getQuery("yql.queries.users.getByStaffIds");
        ListValue idsParam = ListValue.of(staffIds.stream()
                .map(id -> TupleValue.of(PrimitiveValue.int64(id.getT1()),
                        PrimitiveValue.utf8(id.getT2().getId()))).toArray(TupleValue[]::new));
        Params params = Params.of("$ids", idsParam);
        return session.executeDataQueryRetryable(query, params).map(this::toUsers);
    }

    public Mono<List<UserModel>> getByExternalIds(YdbTxSession session, List<Tuple2<String, TenantId>> passportUids,
                                                  List<Tuple2<String, TenantId>> passportLogins,
                                                  List<Tuple2<Long, TenantId>> staffIds) {
        if (passportUids.isEmpty() && passportLogins.isEmpty() && staffIds.isEmpty()) {
            return Mono.just(List.of());
        }
        Params params = prepareExtrnalIdsParams(passportUids, passportLogins, staffIds);
        String query = ydbQuerySource.getQuery("yql.queries.users.getByExternalIds");
        return session.executeDataQueryRetryable(query, params).map(this::toUsers);
    }

    public Mono<List<UserModel>> getDAdmins(YdbTxSession session, TenantId tenantId) {
        String query = ydbQuerySource.getQuery("yql.queries.users.getDAdmins");
        Params params = Params.of("$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
        return session.executeDataQueryRetryable(query, params).map(this::toUsers);
    }

    public Mono<YdbDataQuery> prepareGetByExternalIds(YdbSession session) {
        String query = ydbQuerySource.getQuery("yql.queries.users.getByExternalIds");
        return session.prepareDataQuery(query);
    }

    public Mono<List<UserModel>> executeGetByExternalIds(YdbTxDataQuery query,
                                                         List<Tuple2<String, TenantId>> passportUids,
                                                         List<Tuple2<String, TenantId>> passportLogins,
                                                         List<Tuple2<Long, TenantId>> staffIds) {
        if (passportUids.isEmpty() && passportLogins.isEmpty() && staffIds.isEmpty()) {
            return Mono.just(List.of());
        }
        Params params = prepareExtrnalIdsParams(passportUids, passportLogins, staffIds);
        return query.executeRetryable(params).map(this::toUsers);
    }

    public Mono<Void> upsertUserRetryable(YdbTxSession session, UserModel user) {
        String query = ydbQuerySource.getQuery("yql.queries.users.upsertOneUser");
        Map<String, Value> fields = prepareUserFields(user);
        Params params = Params.of("$user", StructValue.of(fields));
        return session.executeDataQueryRetryable(query, params).then();
    }

    public Mono<Void> upsertUsersRetryable(YdbTxSession session, List<UserModel> users) {
        if (users.isEmpty()) {
            return Mono.empty();
        }
        String query = ydbQuerySource.getQuery("yql.queries.users.upsertManyUsers");
        Params params = Params.of("$users", ListValue.of(users.stream().map(user -> {
            Map<String, Value> fields = prepareUserFields(user);
            return StructValue.of(fields);
        }).toArray(StructValue[]::new)));
        return session.executeDataQueryRetryable(query, params).then();
    }

    public Mono<Void> upsertUsersByPassportUidsRetryable(YdbTxSession session, List<UserModel> users) {
        if (users.isEmpty()) {
            return Mono.empty();
        }
        String query = ydbQuerySource.getQuery("yql.queries.users.upsertManyUsersByPassportUids");
        Params params = Params.of("$all_users", ListValue.of(users.stream().map(user -> {
            Map<String, Value> fields = prepareUserFields(user);
            return StructValue.of(fields);
        }).toArray(StructValue[]::new)));
        return session.executeDataQueryRetryable(query, params).then();
    }

    public Mono<YdbDataQuery> prepareUpsertUsers(YdbSession session) {
        String query = ydbQuerySource.getQuery("yql.queries.users.upsertManyUsers");
        return session.prepareDataQuery(query);
    }

    public Mono<Void> executeUpsertUsersRetryable(YdbTxDataQuery query, List<UserModel> users) {
        Params params = Params.of("$users", ListValue.of(users.stream().map(user -> {
            Map<String, Value> fields = prepareUserFields(user);
            return StructValue.of(fields);
        }).toArray(StructValue[]::new)));
        return query.executeRetryable(params).then();
    }

    public Mono<Void> updateUserRetryable(YdbTxSession session, UserModel user) {
        String query = ydbQuerySource.getQuery("yql.queries.users.updateOneUser");
        Map<String, Value> fields = prepareUserFields(user);
        Params params = Params.of("$user", StructValue.of(fields));
        return session.executeDataQueryRetryable(query, params).then();
    }

    public Mono<Void> updateUsersRetryable(YdbTxSession session, List<UserModel> users) {
        if (users.isEmpty()) {
            return Mono.empty();
        }
        String query = ydbQuerySource.getQuery("yql.queries.users.updateManyUsers");
        Params params = Params.of("$users", ListValue.of(users.stream().map(user -> {
            Map<String, Value> fields = prepareUserFields(user);
            return StructValue.of(fields);
        }).toArray(StructValue[]::new)));
        return session.executeDataQueryRetryable(query, params).then();
    }

    public Mono<YdbDataQuery> prepareUpdateUsers(YdbSession session) {
        String query = ydbQuerySource.getQuery("yql.queries.users.updateManyUsers");
        return session.prepareDataQuery(query);
    }

    public Mono<Void> executeUpdateUsersRetryable(YdbTxDataQuery query, List<UserModel> users) {
        Params params = Params.of("$users", ListValue.of(users.stream().map(user -> {
            Map<String, Value> fields = prepareUserFields(user);
            return StructValue.of(fields);
        }).toArray(StructValue[]::new)));
        return query.executeRetryable(params).then();
    }

    public Mono<Void> removeUserRetryable(YdbTxSession session, UserModel user) {
        final Params params = Params.of("$id", PrimitiveValue.utf8(user.getId()),
                "$tenant_id", PrimitiveValue.utf8(user.getTenantId().getId()));
        String query = ydbQuerySource.getQuery("yql.queries.users.deleteUser");
        return session.executeDataQueryRetryable(query, params).then();
    }

    public Mono<Void> removeUsersRetryable(YdbTxSession session, List<UserModel> users) {
        if (users.isEmpty()) {
            return Mono.empty();
        }
        String query = ydbQuerySource.getQuery("yql.queries.users.deleteUsers");
        ListValue idsParam = ListValue.of(users.stream().map(user -> StructValue.of("id",
                PrimitiveValue.utf8(user.getId()), "tenant_id",
                PrimitiveValue.utf8(user.getTenantId().getId()))).toArray(StructValue[]::new));
        Params params = Params.of("$ids", idsParam);
        return session.executeDataQueryRetryable(query, params).then();
    }

    private Params prepareExtrnalIdsParams(List<Tuple2<String, TenantId>> passportUids,
                                           List<Tuple2<String, TenantId>> passportLogins,
                                           List<Tuple2<Long, TenantId>> staffIds) {
        ListValue passportUidsValue = !passportUids.isEmpty() ? ListValue.of(passportUids.stream()
                .map(id -> TupleValue.of(PrimitiveValue.utf8(id.getT1()),
                        PrimitiveValue.utf8(id.getT2().getId()))).toArray(TupleValue[]::new)) : null;
        ListValue passportLoginsValue = !passportLogins.isEmpty() ? ListValue.of(passportLogins.stream()
                .map(id -> TupleValue.of(PrimitiveValue.utf8(id.getT1()),
                        PrimitiveValue.utf8(id.getT2().getId()))).toArray(TupleValue[]::new)) : null;
        ListValue staffIdsValue = !staffIds.isEmpty() ? ListValue.of(staffIds.stream()
                .map(id -> TupleValue.of(PrimitiveValue.int64(id.getT1()),
                        PrimitiveValue.utf8(id.getT2().getId()))).toArray(TupleValue[]::new)) : null;
        OptionalValue passportUidsParam = Ydb.nullableValue(ListType.of(TupleType.of(List.of(PrimitiveType.utf8(),
                PrimitiveType.utf8()))), passportUidsValue);
        OptionalValue passportLoginsParam = Ydb.nullableValue(ListType.of(TupleType.of(List.of(PrimitiveType.utf8(),
                PrimitiveType.utf8()))), passportLoginsValue);
        OptionalValue staffIdsParam = Ydb.nullableValue(ListType.of(TupleType.of(List.of(PrimitiveType.int64(),
                PrimitiveType.utf8()))), staffIdsValue);
        return Params.of("$passport_uids", passportUidsParam, "$passport_logins", passportLoginsParam,
                "$staff_ids", staffIdsParam);
    }

    private Map<String, Value> prepareUserFields(UserModel user) {
        Map<String, Value> fields = new HashMap<>();
        fields.put("id", PrimitiveValue.utf8(user.getId()));
        fields.put("tenant_id", PrimitiveValue.utf8(user.getTenantId().getId()));
        fields.put("passport_uid", Ydb.nullableUtf8(user.getPassportUid().orElse(null)));
        fields.put("passport_login", Ydb.nullableUtf8(user.getPassportLogin().orElse(null)));
        fields.put("staff_id", Ydb.nullableInt64(user.getStaffId().orElse(null)));
        fields.put("staff_dismissed", Ydb.nullableBool(user.getStaffDismissed().orElse(null)));
        fields.put("staff_robot", Ydb.nullableBool(user.getStaffRobot().orElse(null)));
        fields.put("staff_affiliation", Ydb.nullableUtf8(user.getStaffAffiliation()
                .map(Enum::name).orElse(null)));
        fields.put("first_name_en", PrimitiveValue.utf8(user.getFirstNameEn()));
        fields.put("first_name_ru", PrimitiveValue.utf8(user.getFirstNameRu()));
        fields.put("last_name_en", PrimitiveValue.utf8(user.getLastNameEn()));
        fields.put("last_name_ru", PrimitiveValue.utf8(user.getLastNameRu()));
        fields.put("d_admin", PrimitiveValue.bool(user.getDAdmin()));
        fields.put("deleted", PrimitiveValue.bool(user.isDeleted()));
        fields.put("roles", PrimitiveValue.jsonDocument(writeRoles(user.getRoles())));
        fields.put("gender", PrimitiveValue.utf8(String.valueOf(user.getGender())));
        fields.put("work_email", Ydb.nullableUtf8(user.getWorkEmail().orElse(null)));
        fields.put("lang_ui", Ydb.nullableUtf8(user.getLangUi().orElse(null)));
        fields.put("time_zone", Ydb.nullableUtf8(user.getTimeZone().orElse(null)));
        return fields;
    }

    private Optional<UserModel> toUser(DataQueryResult result) {
        if (result.isEmpty()) {
            return Optional.empty();
        }
        if (result.getResultSetCount() > 1) {
            throw new IllegalStateException("Too many result sets");
        }
        ResultSetReader reader = result.getResultSet(0);
        if (!reader.next()) {
            return Optional.empty();
        }
        if (reader.getRowCount() > 1) {
            throw new IllegalStateException("Non unique user");
        }
        UserModel user = readOneUser(reader, new HashMap<>());
        return Optional.of(user);
    }

    private List<UserModel> toUsers(DataQueryResult result) {
        if (result.isEmpty()) {
            return List.of();
        }
        if (result.getResultSetCount() > 1) {
            throw new IllegalStateException("Too many result sets");
        }
        ResultSetReader reader = result.getResultSet(0);
        List<UserModel> users = new ArrayList<>();
        Map<String, TenantId> tenantIdCache = new HashMap<>();
        while (reader.next()) {
            users.add(readOneUser(reader, tenantIdCache));
        }
        return users;
    }

    private UserModel readOneUser(ResultSetReader reader, Map<String, TenantId> tenantIdCache) {
        String tenantIdString = reader.getColumn("tenant_id").getUtf8();

        return UserModel.builder()
                .id(reader.getColumn("id").getUtf8())
                .tenantId(tenantIdCache.computeIfAbsent(tenantIdString, TenantId::new))
                .passportUid(Ydb.utf8OrNull(reader.getColumn("passport_uid")))
                .passportLogin(Ydb.utf8OrNull(reader.getColumn("passport_login")))
                .staffId(Ydb.int64OrNull(reader.getColumn("staff_id")))
                .staffDismissed(Ydb.boolOrNull(reader.getColumn("staff_dismissed")))
                .staffRobot(Ydb.boolOrNull(reader.getColumn("staff_robot")))
                .staffAffiliation(toAffiliation(Ydb.utf8OrNull(reader.getColumn("staff_affiliation"))))
                .roles(readRoles(reader.getColumn("roles").getJsonDocument()))
                .firstNameEn(reader.getColumn("first_name_en").getUtf8())
                .firstNameRu(reader.getColumn("first_name_ru").getUtf8())
                .lastNameEn(reader.getColumn("last_name_en").getUtf8())
                .lastNameRu(reader.getColumn("last_name_ru").getUtf8())
                .dAdmin(reader.getColumn("d_admin").getBool())
                .deleted(reader.getColumn("deleted").getBool())
                .gender(reader.getColumn("gender").getUtf8())
                .workEmail(Ydb.utf8OrNull(reader.getColumn("work_email")))
                .langUi(Ydb.utf8OrNull(reader.getColumn("lang_ui")))
                .timeZone(Ydb.utf8OrNull(reader.getColumn("time_zone")))
                .build();
    }

    private StaffAffiliation toAffiliation(String value) {
        if (value == null) {
            return null;
        }
        return StaffAffiliation.valueOf(value);
    }

    private Map<UserServiceRoles, Set<Long>> readRoles(String roles) {
        try {
            return rolesReader.readValue(roles);
        } catch (JsonProcessingException e) {
            throw new UncheckedIOException(e);
        }
    }

    private String writeRoles(Map<UserServiceRoles, Set<Long>> roles) {
        try {
            return rolesWriter.writeValueAsString(roles);
        } catch (JsonProcessingException e) {
            throw new UncheckedIOException(e);
        }
    }

}
