package ru.yandex.intranet.d.datasource

import com.yandex.ydb.table.query.DataQueryResult
import com.yandex.ydb.table.query.Params
import com.yandex.ydb.table.transaction.TransactionMode
import kotlinx.coroutines.reactor.awaitSingle
import kotlinx.coroutines.reactor.awaitSingleOrNull
import reactor.util.function.Tuples
import ru.yandex.intranet.d.dao.WithTx
import ru.yandex.intranet.d.datasource.model.YdbSession
import ru.yandex.intranet.d.datasource.model.YdbTableClient
import ru.yandex.intranet.d.datasource.model.YdbTxSession
import ru.yandex.intranet.d.kotlin.mono

/**
 * Session wrapper.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
class Session constructor(val session: YdbSession) {
    suspend fun <T> rwTxRetryable(statement: suspend TxSession.() -> T): T? {
        return session.usingTxMonoRetryable(TransactionMode.SERIALIZABLE_READ_WRITE) {s ->
            mono {statement(TxSession(s))}}.awaitSingleOrNull()
    }
    suspend fun <T> roRetryable(statement: suspend TxSession.() -> T): T? {
        return session.usingTxMonoRetryable(TransactionMode.ONLINE_READ_ONLY) {s ->
            mono {statement(TxSession(s))}}.awaitSingleOrNull()
    }
    suspend fun <T> roStaleRetryable(statement: suspend TxSession.() -> T): T? {
        return session.usingTxMonoRetryable(TransactionMode.STALE_READ_ONLY) {s ->
            mono {statement(TxSession(s))}}.awaitSingleOrNull()
    }
    suspend fun <T> rwTx(statement: suspend TxSession.() -> T): T? {
        return session.usingTxMono(TransactionMode.SERIALIZABLE_READ_WRITE) {s ->
            mono {statement(TxSession(s))}}.awaitSingleOrNull()
    }
    suspend fun <T> ro(statement: suspend TxSession.() -> T): T? {
        return session.usingTxMono(TransactionMode.ONLINE_READ_ONLY) {s ->
            mono {statement(TxSession(s))}}.awaitSingleOrNull()
    }
    suspend fun <T> roStale(statement: suspend TxSession.() -> T): T? {
        return session.usingTxMono(TransactionMode.STALE_READ_ONLY) {s ->
            mono {statement(TxSession(s))}}.awaitSingleOrNull()
    }
    suspend fun <U, V, W> rwTxRetryableOptimized(preamble: suspend TxSession.() -> WithTx<U>,
                                                 body: suspend TxSession.(param: U) -> V?,
                                                 trailer: suspend TxSession.(param: V) -> W): W? {
        return session.usingCompTxMonoRetryable(TransactionMode.SERIALIZABLE_READ_WRITE,
            { s -> mono {
                val withTx = preamble(TxSession(s))
                Tuples.of(withTx.value, withTx.txId)
            }},
            { s, param -> mono { body(TxSession(s), param) }},
            { s, param -> mono { trailer(TxSession(s), param) }}).awaitSingleOrNull()
    }
    suspend fun <U, V, W> rwTxOptimized(preamble: suspend TxSession.() -> WithTx<U>,
                                        body: suspend TxSession.(param: U) -> V?,
                                        trailer: suspend TxSession.(param: V) -> W): W? {
        return session.usingCompTxMono(TransactionMode.SERIALIZABLE_READ_WRITE,
            { s -> mono {
                val withTx = preamble(TxSession(s))
                Tuples.of(withTx.value, withTx.txId)
            }},
            { s, param -> mono { body(TxSession(s), param) }},
            { s, param -> mono { trailer(TxSession(s), param) }}).awaitSingleOrNull()
    }
    fun roStaleSingleRetryableCommit(): YdbTxSession {
        return session.asTxCommitRetryable(TransactionMode.STALE_READ_ONLY)
    }
    fun rwSingleRetryableCommit(): YdbTxSession {
        return session.asTxCommitRetryable(TransactionMode.SERIALIZABLE_READ_WRITE)
    }
}

/**
 * Transaction session wrapper.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
class TxSession constructor(val txSession: YdbTxSession) {
    suspend fun executeDataQuery(query: String, params: Params): DataQueryResult {
        return txSession.executeDataQuery(query, params).awaitSingle()
    }
    suspend fun executeDataQueryRetryable(query: String, params: Params): DataQueryResult {
        return txSession.executeDataQueryRetryable(query, params).awaitSingle()
    }
}

suspend fun <T> dbSessionRetryable(client: YdbTableClient, statement: suspend Session.() -> T): T? {
    return client.usingSessionMonoRetryable { s -> mono {statement(Session(s))} }.awaitSingleOrNull()
}

suspend fun <T> dbSession(client: YdbTableClient, statement: suspend Session.() -> T): T? {
    return client.usingSessionMonoRetryable { s -> mono {statement(Session(s))} }.awaitSingleOrNull()
}
