package ru.yandex.intranet.d.datasource;

import java.time.Instant;
import java.util.List;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Stream;

import javax.annotation.Nullable;

import com.yandex.ydb.table.result.ValueReader;
import com.yandex.ydb.table.values.ListValue;
import com.yandex.ydb.table.values.OptionalType;
import com.yandex.ydb.table.values.OptionalValue;
import com.yandex.ydb.table.values.PrimitiveType;
import com.yandex.ydb.table.values.PrimitiveValue;
import com.yandex.ydb.table.values.TupleValue;
import com.yandex.ydb.table.values.Type;
import com.yandex.ydb.table.values.Value;
import reactor.util.function.Tuple2;

import ru.yandex.intranet.d.model.StringIdWithTenant;
import ru.yandex.intranet.d.model.TenantId;

/**
 * YDB utils.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class Ydb {

    public static final long MAX_RESPONSE_ROWS = 1000;

    private Ydb() {
    }

    public static Boolean boolOrNull(ValueReader valueReader) {
        if (valueReader.isOptionalItemPresent()) {
            return valueReader.getBool();
        }
        return null;
    }

    public static boolean boolOrDefault(ValueReader valueReader, boolean defaultValue) {
        if (valueReader.isOptionalItemPresent()) {
            return valueReader.getBool();
        }
        return defaultValue;
    }

    public static Boolean bool(ValueReader valueReader) {
        return valueReader.getBool();
    }

    public static Long int64OrNull(ValueReader valueReader) {
        if (valueReader.isOptionalItemPresent()) {
            return valueReader.getInt64();
        }
        return null;
    }

    public static long int64OrDefault(ValueReader valueReader, long defaultValue) {
        if (valueReader.isOptionalItemPresent()) {
            return valueReader.getInt64();
        }
        return defaultValue;
    }

    public static int int32OrDefault(ValueReader valueReader, int defaultValue) {
        if (valueReader.isOptionalItemPresent()) {
            return valueReader.getInt32();
        }
        return defaultValue;
    }

    public static Long int64(ValueReader valueReader) {
        return valueReader.getInt64();
    }

    public static String utf8OrNull(ValueReader valueReader) {
        if (valueReader.isOptionalItemPresent()) {
            return valueReader.getUtf8();
        }
        return null;
    }

    public static String utf8(ValueReader valueReader) {
        return valueReader.getUtf8();
    }

    public static String utf8EmptyToNull(ValueReader valueReader) {
        String value = valueReader.getUtf8();
        return "".equals(value) ? null : value;
    }

    public static <T> T jsonOrNull(ValueReader valueReader, Function<String, T> objectReader) {
        if (valueReader.isOptionalItemPresent()) {
            return objectReader.apply(valueReader.getJson());
        }
        return null;
    }

    public static <T> T jsonDocumentOrNull(ValueReader valueReader, Function<String, T> objectReader) {
        if (valueReader.isOptionalItemPresent()) {
            return objectReader.apply(valueReader.getJsonDocument());
        }
        return null;
    }

    public static Instant timestampOrNull(ValueReader valueReader) {
        if (valueReader.isOptionalItemPresent()) {
            return valueReader.getTimestamp();
        }
        return null;
    }

    public static Instant timestamp(ValueReader valueReader) {
        return valueReader.getTimestamp();
    }

    public static OptionalValue nullableTimestamp(Instant value) {
        OptionalType type = OptionalType.of(PrimitiveType.timestamp());
        return value != null ? type.newValue(PrimitiveValue.timestamp(value)) : type.emptyValue();
    }

    public static OptionalValue nullableUtf8(String value) {
        if (value != null) {
            return OptionalType.of(PrimitiveType.utf8()).newValue(PrimitiveValue.utf8(value));
        } else {
            return OptionalType.of(PrimitiveType.utf8()).emptyValue();
        }
    }

    public static PrimitiveValue nullToEmptyUtf8(Optional<String> value) {
        return PrimitiveValue.utf8(value.orElse(""));
    }

    public static PrimitiveValue nullToEmptyUtf8(@Nullable String value) {
        return PrimitiveValue.utf8(value == null ? "" : value);
    }

    public static OptionalValue nullableInt64(Long value) {
        if (value != null) {
            return OptionalType.of(PrimitiveType.int64()).newValue(PrimitiveValue.int64(value));
        } else {
            return OptionalType.of(PrimitiveType.int64()).emptyValue();
        }
    }

    public static OptionalValue nullableInt32(Integer value) {
        if (value != null) {
            return OptionalType.of(PrimitiveType.int32()).newValue(PrimitiveValue.int32(value));
        } else {
            return OptionalType.of(PrimitiveType.int32()).emptyValue();
        }
    }

    public static OptionalValue nullableBool(Boolean value) {
        if (value != null) {
            return OptionalType.of(PrimitiveType.bool()).newValue(PrimitiveValue.bool(value));
        } else {
            return OptionalType.of(PrimitiveType.bool()).emptyValue();
        }
    }

    public static OptionalValue nullableDouble(Double value) {
        if (value != null) {
            return OptionalType.of(PrimitiveType.float64()).newValue(PrimitiveValue.float64(value));
        } else {
            return OptionalType.of(PrimitiveType.float64()).emptyValue();
        }
    }

    public static <T> OptionalValue nullableJson(T value, Function<T, String> objectWriter) {
        if (value != null) {
            return OptionalType.of(PrimitiveType.json())
                    .newValue(PrimitiveValue.json(objectWriter.apply(value)));
        } else {
            return OptionalType.of(PrimitiveType.json()).emptyValue();
        }
    }

    public static <T> OptionalValue nullableJsonDocument(T value, Function<T, String> objectWriter) {
        if (value != null) {
            return OptionalType.of(PrimitiveType.jsonDocument())
                    .newValue(PrimitiveValue.jsonDocument(objectWriter.apply(value)));
        } else {
            return OptionalType.of(PrimitiveType.jsonDocument()).emptyValue();
        }
    }

    public static OptionalValue nullableValue(Type type, Value value) {
        if (value != null) {
            return OptionalType.of(type).newValue(value);
        } else {
            return OptionalType.of(type).emptyValue();
        }
    }

    public static ListValue toIdsListValue(List<Tuple2<String, TenantId>> ids) {
        return ListValue.of(ids.stream().map(id -> TupleValue.of(PrimitiveValue.utf8(id.getT1()),
                PrimitiveValue.utf8(id.getT2().getId()))).toArray(TupleValue[]::new));
    }

    public static ListValue toIdWithTenantsListValue(List<StringIdWithTenant> ids) {
        return toIdWithTenantsListValue(ids.stream());
    }

    public static ListValue toIdWithTenantsListValue(Stream<StringIdWithTenant> ids) {
        return ListValue.of(ids.map(id -> TupleValue.of(
                PrimitiveValue.utf8(id.getId()),
                PrimitiveValue.utf8(id.getTenantId().getId())
        )).toArray(TupleValue[]::new));
    }

    public static ListValue toIdWithTenantsListValue(List<String> ids, TenantId tenantId) {
        return ListValue.of(ids.stream().map(id -> TupleValue.of(
                PrimitiveValue.utf8(id),
                PrimitiveValue.utf8(tenantId.getId())
        )).toArray(TupleValue[]::new));
    }

    public static PrimitiveValue utf8(String value) {
        return PrimitiveValue.utf8(value);
    }

    public static PrimitiveValue bool(boolean value) {
        return PrimitiveValue.bool(value);
    }

    public static PrimitiveValue int8(byte value) {
        return PrimitiveValue.int8(value);
    }

    public static PrimitiveValue uint8(byte value) {
        return PrimitiveValue.uint8(value);
    }

    public static PrimitiveValue int16(short value) {
        return PrimitiveValue.int16(value);
    }

    public static PrimitiveValue uint16(short value) {
        return PrimitiveValue.uint16(value);
    }

    public static PrimitiveValue int32(int value) {
        return PrimitiveValue.int32(value);
    }

    public static PrimitiveValue uint32(int value) {
        return PrimitiveValue.uint32(value);
    }

    public static PrimitiveValue int64(long value) {
        return PrimitiveValue.int64(value);
    }

    public static PrimitiveValue uint64(long value) {
        return PrimitiveValue.uint64(value);
    }

    public static PrimitiveValue float32(float value) {
        return PrimitiveValue.float32(value);
    }

    public static PrimitiveValue float64(double value) {
        return PrimitiveValue.float64(value);
    }

    public static PrimitiveValue timestamp(Instant value) {
        return PrimitiveValue.timestamp(value);
    }
}
