package ru.yandex.intranet.d.datasource;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.math.BigInteger;
import java.time.Duration;
import java.util.concurrent.TimeUnit;

import com.fasterxml.jackson.annotation.JsonFormat;
import com.fasterxml.jackson.core.json.JsonWriteFeature;
import com.fasterxml.jackson.databind.MapperFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.databind.json.JsonMapper;
import com.fasterxml.jackson.datatype.jdk8.Jdk8Module;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import com.fasterxml.jackson.module.kotlin.KotlinModule;
import com.fasterxml.jackson.module.paramnames.ParameterNamesModule;
import com.yandex.ydb.core.grpc.BalancingPolicy;
import com.yandex.ydb.core.grpc.BalancingSettings;
import com.yandex.ydb.core.grpc.DiscoveryMode;
import com.yandex.ydb.core.grpc.GrpcTransport;
import com.yandex.ydb.core.grpc.TransportImplType;
import com.yandex.ydb.core.rpc.RpcTransport;
import com.yandex.ydb.table.SchemeClient;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.rpc.SchemeRpc;
import com.yandex.ydb.table.rpc.TableRpc;
import com.yandex.ydb.table.rpc.grpc.GrpcSchemeRpc;
import com.yandex.ydb.table.rpc.grpc.GrpcTableRpc;
import io.netty.channel.ChannelOption;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.beans.factory.config.PropertiesFactoryBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.io.Resource;
import org.springframework.core.io.support.PathMatchingResourcePatternResolver;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import ru.yandex.intranet.d.datasource.impl.YdbRpcTransport;
import ru.yandex.intranet.d.datasource.impl.YdbSchemeClientImpl;
import ru.yandex.intranet.d.datasource.impl.YdbSchemeRpc;
import ru.yandex.intranet.d.datasource.impl.YdbTableClientImpl;
import ru.yandex.intranet.d.datasource.impl.YdbTableRpc;
import ru.yandex.intranet.d.datasource.model.YdbSchemeClient;
import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.datasource.security.YdbAuthProvider;
import ru.yandex.intranet.d.datasource.security.YdbDummyAuthProvider;
import ru.yandex.intranet.d.datasource.utils.ReadinessYdbTableClientHolder;
import ru.yandex.intranet.d.metrics.YdbMetrics;
import ru.yandex.intranet.d.util.MdcTaskDecorator;
import ru.yandex.intranet.d.util.ObjectMapperHolder;

/**
 * YDB configuration.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Configuration
public class YdbConfiguration {

    @Bean
    @SuppressWarnings("ParameterNumber")
    public YdbRpcTransport ydbRpcTransportReal(@Value("${ydb.endpoint}") String endpoint,
                                               @Value("${ydb.database}") String database,
                                               @Value("${ydb.readTimeoutMs}") long readTimeoutMillis,
                                               @Value("${ydb.discoveryPeriodSec}") long discoveryPeriodSeconds,
                                               @Value("${ydb.connectTimeoutMs}") int connectTimeoutMillis,
                                               @Value("${ydb.keepAliveTimeSec}") long keepAliveTimeSeconds,
                                               @Value("${ydb.keepAliveTimeoutSec}") long keepAliveTimeoutSeconds,
                                               @Value("${ydb.idleTimeoutMinutes}") long idleTimeoutMinutes,
                                               @Value("${ydb.keepAliveWithoutCalls}") boolean keepAliveWithoutCalls,
                                               @Qualifier("ydbCallTaskExecutor") ThreadPoolTaskExecutor executor,
                                               YdbAuthProvider authProvider) {
        GrpcTransport.Builder rpcTransportBuilder = GrpcTransport
                .forEndpoint(endpoint, database)
                .withReadTimeout(readTimeoutMillis, TimeUnit.MILLISECONDS)
                .withEndpointsDiscoveryPeriod(Duration.ofSeconds(discoveryPeriodSeconds))
                .withCallExecutor(executor)
                .withTransportImplType(TransportImplType.GRPC_TRANSPORT_IMPL)
                .withDiscoveryMode(DiscoveryMode.SYNC)
                .withBalancingSettings(BalancingSettings.fromPolicy(BalancingPolicy.USE_ALL_NODES))
                .withChannelInitializer(channel -> channel
                        .withOption(ChannelOption.CONNECT_TIMEOUT_MILLIS, connectTimeoutMillis)
                        .keepAliveTime(keepAliveTimeSeconds, TimeUnit.SECONDS)
                        .keepAliveWithoutCalls(keepAliveWithoutCalls)
                        .keepAliveTimeout(keepAliveTimeoutSeconds, TimeUnit.SECONDS)
                        .idleTimeout(idleTimeoutMinutes, TimeUnit.MINUTES));
        if (!(authProvider instanceof YdbDummyAuthProvider)) {
            rpcTransportBuilder = rpcTransportBuilder.withAuthProvider(authProvider);
        }
        RpcTransport rpcTransport = rpcTransportBuilder.build();
        return new YdbRpcTransport(rpcTransport);
    }

    @Bean
    public YdbTableRpc ydbTableRpc(YdbRpcTransport rpcTransport) {
        TableRpc tableRpc = GrpcTableRpc.useTransport(rpcTransport.getRpcTransport());
        return new YdbTableRpc(tableRpc);
    }

    @Bean
    public YdbSchemeRpc ydbSchemeRpc(YdbRpcTransport rpcTransport) {
        SchemeRpc schemeRpc = GrpcSchemeRpc.useTransport(rpcTransport.getRpcTransport());
        return new YdbSchemeRpc(schemeRpc);
    }

    @Bean
    @SuppressWarnings("ParameterNumber")
    public TableClient tableClient(
            @Value("${ydb.sessionPoolMinSize}") int sessionPoolMinSize,
            @Value("${ydb.sessionPoolMaxSize}") int sessionPoolMaxSize,
            @Value("${ydb.sessionMaxRetries}") int sessionMaxRetries,
            @Value("${ydb.sessionMaxIdleTimeSec}") long sessionMaxIdleTimeSeconds,
            @Value("${ydb.sessionKeepAliveTimeSec}") long sessionKeepAliveTimeSeconds,
            YdbTableRpc tableRpc
    ) {
        return TableClient.newClient(tableRpc.getTableRpc())
                .sessionPoolSize(sessionPoolMinSize, sessionPoolMaxSize)
                .sessionCreationMaxRetries(sessionMaxRetries)
                .sessionMaxIdleTime(Duration.ofSeconds(sessionMaxIdleTimeSeconds))
                .sessionKeepAliveTime(Duration.ofSeconds(sessionKeepAliveTimeSeconds))
                .keepQueryText(true)
                // Client cache is not used, see DISPENSER-4676
                .queryCacheSize(0)
                .build();
    }

    @Bean
    @SuppressWarnings("ParameterNumber")
    public YdbTableClient ydbTableClient(
            @Value("${ydb.sessionWaitTimeoutMs}") long sessionWaitTimeoutMillis,
            @Value("${ydb.sessionWaitRetries}") long sessionWaitRetries,
            @Value("${ydb.queryTimeoutMs}") long queryTimeoutMillis,
            @Value("${ydb.queryRetries}") long queryRetries,
            @Value("${ydb.sessionRetries}") long sessionRetries,
            @Value("${ydb.transactionRetries}") long transactionRetries,
            @Value("${ydb.sessionClientWaitTimeoutMs}") long sessionClientWaitTimeoutMillis,
            @Value("${ydb.queryClientTimeoutMs}") long queryClientTimeoutMillis,
            TableClient tableClient,
            YdbMetrics ydbMetrics
    ) {
        return new YdbTableClientImpl(tableClient, Duration.ofMillis(sessionWaitTimeoutMillis), sessionWaitRetries,
                queryTimeoutMillis, queryRetries, sessionRetries, transactionRetries, ydbMetrics,
                Duration.ofMillis(sessionClientWaitTimeoutMillis), Duration.ofMillis(queryClientTimeoutMillis));
    }

    @Bean
    @SuppressWarnings("ParameterNumber")
    public ReadinessYdbTableClientHolder readinessYdbTableClient(
            @Value("${ydb.readiness.sessionPoolMinSize}") int sessionPoolMinSize,
            @Value("${ydb.readiness.sessionPoolMaxSize}") int sessionPoolMaxSize,
            @Value("${ydb.readiness.sessionMaxRetries}") int sessionMaxRetries,
            @Value("${ydb.readiness.sessionMaxIdleTimeSec}") long sessionMaxIdleTimeSeconds,
            @Value("${ydb.readiness.sessionKeepAliveTimeSec}") long sessionKeepAliveTimeSeconds,
            @Value("${ydb.readiness.sessionWaitTimeoutMs}") long sessionWaitTimeoutMillis,
            @Value("${ydb.readiness.sessionWaitRetries}") long sessionWaitRetries,
            @Value("${ydb.readiness.queryTimeoutMs}") long queryTimeoutMillis,
            @Value("${ydb.readiness.queryRetries}") long queryRetries,
            @Value("${ydb.readiness.sessionRetries}") long sessionRetries,
            @Value("${ydb.readiness.transactionRetries}") long transactionRetries,
            @Value("${ydb.readiness.sessionClientWaitTimeoutMs}") long sessionClientWaitTimeoutMillis,
            @Value("${ydb.readiness.queryClientTimeoutMs}") long queryClientTimeoutMillis,
            YdbTableRpc tableRpc,
            YdbMetrics ydbMetrics) {
        TableClient tableClient = TableClient.newClient(tableRpc.getTableRpc())
                .sessionPoolSize(sessionPoolMinSize, sessionPoolMaxSize)
                .sessionCreationMaxRetries(sessionMaxRetries)
                .sessionMaxIdleTime(Duration.ofSeconds(sessionMaxIdleTimeSeconds))
                .sessionKeepAliveTime(Duration.ofSeconds(sessionKeepAliveTimeSeconds))
                .keepQueryText(true)
                // Client cache is not used, see DISPENSER-4676
                .queryCacheSize(0)
                .build();
        YdbTableClient ydbTableClient = new YdbTableClientImpl(tableClient, Duration.ofMillis(sessionWaitTimeoutMillis),
                sessionWaitRetries, queryTimeoutMillis, queryRetries, sessionRetries, transactionRetries, ydbMetrics,
                Duration.ofMillis(sessionClientWaitTimeoutMillis), Duration.ofMillis(queryClientTimeoutMillis));
        return new ReadinessYdbTableClientHolder(ydbTableClient);
    }

    @Bean
    public YdbSchemeClient ydbSchemeClient(YdbSchemeRpc schemeRpc,
                                           @Value("${ydb.queryTimeoutMs}") long queryTimeoutMillis,
                                           @Value("${ydb.queryRetries}") long queryRetries,
                                           @Value("${ydb.queryClientTimeoutMs}") long queryClientTimeoutMillis,
                                           YdbMetrics ydbMetrics) {
        SchemeClient schemeClient = SchemeClient.newClient(schemeRpc.getSchemeRpc()).build();
        return new YdbSchemeClientImpl(schemeClient, queryRetries, queryTimeoutMillis, ydbMetrics,
                Duration.ofMillis(queryClientTimeoutMillis));
    }

    @Bean("ydbCallTaskExecutor")
    public ThreadPoolTaskExecutor ydbCallTaskExecutor(
            @Value("${ydb.executorShutdownTimeoutMs}") long executorShutdownTimeoutMs) {
        ThreadPoolTaskExecutor executor = new ThreadPoolTaskExecutor();
        executor.setQueueCapacity(10000);
        executor.setMaxPoolSize(30);
        executor.setCorePoolSize(10);
        executor.setDaemon(true);
        executor.setThreadNamePrefix("ydb-call-executor-");
        executor.setAwaitTerminationMillis(executorShutdownTimeoutMs);
        executor.setWaitForTasksToCompleteOnShutdown(true);
        executor.setTaskDecorator(new MdcTaskDecorator());
        return executor;
    }

    @Bean("yqlQueries")
    public PropertiesFactoryBean yqlQueries() {
        PathMatchingResourcePatternResolver resourceResolver = new PathMatchingResourcePatternResolver();
        try {
            Resource[] resources = resourceResolver.getResources("classpath:queries/*.xml");
            PropertiesFactoryBean factory = new PropertiesFactoryBean();
            factory.setSingleton(true);
            factory.setLocations(resources);
            return factory;
        } catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    @Bean("ydbJsonObjectMapper")
    public ObjectMapperHolder ydbJsonObjectMapper() {
        ObjectMapper mapper = JsonMapper.builder()
                .enable(JsonWriteFeature.WRITE_NUMBERS_AS_STRINGS)
                .enable(MapperFeature.ALLOW_COERCION_OF_SCALARS)
                .disable(SerializationFeature.WRITE_DURATIONS_AS_TIMESTAMPS)
                .withConfigOverride(BigInteger.class,
                        o -> o.setFormat(JsonFormat.Value.forShape(JsonFormat.Shape.STRING)))
                .addModule(new Jdk8Module())
                .addModule(new JavaTimeModule())
                .addModule(new ParameterNamesModule())
                .addModule(new KotlinModule.Builder().build())
                .build();
        return new ObjectMapperHolder(mapper);
    }

}
