package ru.yandex.intranet.d.datasource.coordination;

import java.time.Duration;
import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;

import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.datasource.coordination.impl.ClusterManagerBuilderImpl;
import ru.yandex.intranet.d.datasource.coordination.model.cluster.ClusterLeader;
import ru.yandex.intranet.d.datasource.coordination.model.cluster.ClusterMembership;
import ru.yandex.intranet.d.datasource.coordination.model.cluster.NodeInfo;
import ru.yandex.intranet.d.datasource.coordination.model.cluster.NodeLeadershipStatus;
import ru.yandex.intranet.d.datasource.coordination.model.session.SessionState;

/**
 * Cluster manager.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public interface ClusterManager {

    /**
     * Schedule cluster manager start, non-blocking method.
     */
    void start();

    /**
     * Schedule cluster manager stop, non-blocking method.
     * @param onStop On stop asynchronous callback.
     */
    void stop(Runnable onStop);

    /**
     * Check if cluster manager is running.
     * @return True if running, false otherwise.
     */
    boolean isRunning();

    /**
     * Check if current node is the leader, asynchronously.
     * Request to YDB coordinator is performed to check leadership status.
     *
     * @return True if leader, false otherwise.
     */
    Mono<Boolean> isLeader();

    /**
     * Check if current node is the leader, use cached status.
     * @return True if leader, false otherwise, empty if no cached data is present.
     */
    Optional<Boolean> isLeaderCached();

    /**
     * Get node info for the current leader, asynchronously.
     * Request to YDB coordinator is performed to check leadership status.
     *
     * @return Node info for the leader or empty if no leader is present.
     */
    Mono<Optional<NodeInfo>> getLeader();

    /**
     * Get node info for the current leader.
     *
     * @return Node info for the leader or empty if no leader is present.
     */
    Optional<NodeInfo> getLeaderCached();

    /**
     * Check if current node is a member of the cluster, asynchronously.
     * Request to YDB coordinator is performed to check membership status.
     *
     * @return True if member, false otherwise.
     */
    Mono<Boolean> isMember();

    /**
     * Check if current node is a member of the cluster, use cached status.
     * @return True if member, false otherwise, empty if no cached data is present.
     */
    Optional<Boolean> isMemberCached();

    /**
     * Get node info for all cluster members, asynchronously.
     * Request to YDB coordinator is performed to check membership status.
     *
     * @return Node info for each cluster member.
     */
    Mono<Set<NodeInfo>> getMembers();

    /**
     * Get node info for all cluster members, use cached data.
     * @return Node info for each cluster member, empty if no cached data is present.
     */
    Optional<Set<NodeInfo>> getMembersCached();

    /**
     * Get last seen cluster members, returns last seen members even when coordination session is broken
     * @return Last seen cluster members.
     */
    Set<NodeInfo> getLastSeenClusterMembers();

    /**
     * Subscribe to the current node leadership status.
     * @param consumer Consumer.
     */
    void addLeadershipSubscriber(Consumer<NodeLeadershipStatus> consumer);

    /**
     * Subscribe to the cluster leader.
     * @param consumer Consumer.
     */
    void addLeaderSubscriber(Consumer<ClusterLeader> consumer);

    /**
     * Subscribe to the cluster membership changes.
     * @param consumer Consumer.
     */
    void addMembershipSubscriber(Consumer<ClusterMembership> consumer);

    /**
     * Get this node id.
     * @return This node id.
     */
    String getNodeId();

    /**
     * Get current version.
     * @return Current version.
     */
    String getCurrentVersion();

    /**
     * Get session state.
     * @return Session state.
     */
    SessionState getSessionState();

    /**
     * Create new cluster manager builder.
     * @param leadershipSemaphoreName YDB semaphore name for leader election.
     * @param membershipSemaphoreName YDB semaphore name for membership discovery.
     * @param coordinatorBuilder YDB coordinator builder.
     * @return Cluster manager builder.
     */
    static Builder newClusterManager(String leadershipSemaphoreName, String membershipSemaphoreName,
                                     Coordinator.Builder coordinatorBuilder) {
        return new ClusterManagerBuilderImpl(leadershipSemaphoreName, membershipSemaphoreName, coordinatorBuilder);
    }

    interface Builder {

        /**
         * Set time to wait until semaphore acquisition for leader election.
         * Determines how often unsuccessful semaphore acquisition request will be retried.
         *
         * @param duration Target duration.
         * @return Builder.
         */
        Builder leadershipAcquisitionWaitDuration(Duration duration);

        /**
         * Set time to wait until semaphore acquisition for membership discovery.
         * Determines how often unsuccessful semaphore acquisition request will be retried.
         *
         * @param duration Target duration.
         * @return Builder.
         */
        Builder membershipAcquisitionWaitDuration(Duration duration);

        /**
         * Set timeout for blocking when waiting for semaphore description.
         * @param timeout Timeout.
         * @return Builder.
         */
        Builder describeSemaphoreBlockTimeout(Duration timeout);

        /**
         * Set timeout for blocking when waiting for semaphore creation.
         * @param timeout Timeout.
         * @return Builder.
         */
        Builder createSemaphoreBlockTimeout(Duration timeout);

        /**
         * Set scheduler queue capacity for describe requests.
         * @param capacity Capacity.
         * @return Builder.
         */
        Builder describeQueueCapacity(int capacity);

        /**
         * Set internal executor service shutdown timeout.
         * @param timeout Timeout.
         * @return Builder.
         */
        Builder executorShutdownTimeout(Duration timeout);

        /**
         * Set host info supplier.
         * @param supplier Supplier.
         * @return Builder.
         */
        Builder hostInfoSupplier(HostInfoSupplier supplier);

        /**
         * Set version supplier.
         * @param supplier Supplier.
         * @return Builder.
         */
        Builder versionSupplier(VersionSupplier supplier);

        /**
         * Timeout for blocking on semaphore acquisition should be greater
         * than time to wait until semaphore acquisition.
         * @param timeout Timeout.
         * @return Builder.
         */
        Builder acquireSemaphoreBlockTimeout(Duration timeout);

        /**
         * Timeout for blocking on semaphore release.
         * @param timeout Timeout.
         * @return Builder.
         */
        Builder releaseSemaphoreBlockTimeout(Duration timeout);

        /**
         * Build new cluster manager.
         * @return New cluster manager.
         */
        ClusterManager build();

    }

}
