package ru.yandex.intranet.d.datasource.coordination;

import java.io.Closeable;

import com.yandex.ydb.core.rpc.OutStreamObserver;
import com.yandex.ydb.core.rpc.StreamObserver;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.datasource.coordination.impl.CoordinationClientBuilderImpl;
import ru.yandex.intranet.d.datasource.coordination.model.AlterCoordinationNodeRequest;
import ru.yandex.intranet.d.datasource.coordination.model.CoordinationNode;
import ru.yandex.intranet.d.datasource.coordination.model.CoordinationNodeDescription;
import ru.yandex.intranet.d.datasource.coordination.model.CreateCoordinationNodeRequest;
import ru.yandex.intranet.d.datasource.coordination.model.DescribeCoordinationNodeRequest;
import ru.yandex.intranet.d.datasource.coordination.model.DropCoordinationNodeRequest;
import ru.yandex.intranet.d.datasource.coordination.model.session.CoordinationSessionRequest;
import ru.yandex.intranet.d.datasource.coordination.model.session.CoordinationSessionResponse;
import ru.yandex.intranet.d.datasource.coordination.model.session.SessionOperationParameters;
import ru.yandex.intranet.d.datasource.coordination.rpc.CoordinationRpc;

/**
 * YDB coordination client.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public interface CoordinationClient extends Closeable {

    /**
     * Create YDB coordination node asynchronously.
     * @param request Request parameters.
     * @return Created node name.
     */
    Mono<CoordinationNode> createNode(CreateCoordinationNodeRequest request);

    /**
     * Alter YDB coordination node asynchronously.
     * @param request Request parameters.
     * @return Nothing.
     */
    Mono<Void> alterNode(AlterCoordinationNodeRequest request);

    /**
     * Drop YDB coordination node asynchronously.
     * @param request Request parameters.
     * @return Nothing.
     */
    Mono<Void> dropNode(DropCoordinationNodeRequest request);

    /**
     * Describe YDB coordination node.
     * @param request Request parameters.
     * @return Node description.
     */
    Mono<CoordinationNodeDescription> describeNode(DescribeCoordinationNodeRequest request);

    /**
     * Run YDB coordination session asynchronously.
     * @param params Session parameters.
     * @param response Observer of the session responses.
     * @return Publisher of the session requests.
     */
    OutStreamObserver<CoordinationSessionRequest> session(SessionOperationParameters params,
                                                          StreamObserver<CoordinationSessionResponse> response);

    /**
     * Create new YDB coordination client builder.
     * @param coordinationRpc YDB coordination RPC.
     * @return Builder.
     */
    static Builder newClient(CoordinationRpc coordinationRpc) {
        return new CoordinationClientBuilderImpl(coordinationRpc);
    }

    interface Builder {

        /**
         * Create new YDB coordination client.
         * @return YDB coordination client.
         */
        CoordinationClient build();

    }

}
