package ru.yandex.intranet.d.datasource.coordination;

import java.time.Duration;
import java.util.function.Consumer;

import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.datasource.coordination.impl.CoordinatorBuilderImpl;
import ru.yandex.intranet.d.datasource.coordination.model.NodeConsistencyMode;
import ru.yandex.intranet.d.datasource.coordination.model.session.ChangesSubscription;
import ru.yandex.intranet.d.datasource.coordination.model.session.CoordinationSemaphore;
import ru.yandex.intranet.d.datasource.coordination.model.session.CoordinationSemaphoreDescription;
import ru.yandex.intranet.d.datasource.coordination.model.session.SessionState;

/**
 * YDB coordinator.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public interface Coordinator {

    /**
     * Schedule coordinator start, non-blocking method.
     */
    void start();

    /**
     * Schedule coordinator stop, non-blocking method.
     * @param onStop On stop asynchronous callback.
     */
    void stop(Runnable onStop);

    /**
     * Acquire semaphore asynchronously.
     * @param name Semaphore name.
     * @param timeout Maximum time to wait until semaphore acquisition
     * @param count How much resource needs to be acquired
     * @param data Owner-specific data
     * @param ephemeral True if acquiring ephemeral semaphore
     * @return True if acquired, false otherwise.
     */
    Mono<Boolean> acquireSemaphore(String name, Duration timeout, long count, byte[] data, boolean ephemeral);

    /**
     * Release semaphore asynchronously.
     * @param name Semaphore name.
     * @return True if released, false otherwise.
     */
    Mono<Boolean> releaseSemaphore(String name);

    /**
     * Create semaphore asynchronously.
     * @param name Semaphore name.
     * @param limit Resource limit for a new semaphore.
     * @param data Semaphore-specific data.
     * @return Created semaphore name.
     */
    Mono<CoordinationSemaphore> createSemaphore(String name, long limit, byte[] data);

    /**
     * Update semaphore data asynchronously.
     * @param name Semaphore name.
     * @param data Semaphore-specific data.
     * @return Nothing.
     */
    Mono<Void> updateSemaphore(String name, byte[] data);

    /**
     * Delete semaphore asynchronously.
     * @param name Semaphore name.
     * @param force Force delete, even if used.
     * @return Nothing.
     */
    Mono<Void> deleteSemaphore(String name, boolean force);

    /**
     * Describe semaphore asynchronously.
     * IMPORTANT DETAILS BELOW!
     * This call disables previous semaphore change subscription, take care to not wait on such cancelled subscription.
     *
     * @param name Semaphore name.
     * @param includeOwners Return semaphore owners.
     * @param includeWaiters Return semaphore waiters.
     * @return Semaphore description.
     */
    Mono<CoordinationSemaphoreDescription> describeSemaphore(String name, boolean includeOwners,
                                                             boolean includeWaiters);

    /**
     * Describe semaphore and subscribe to the next semaphore change, asynchronously.
     * IMPORTANT DETAILS BELOW!
     * This call disables previous semaphore change subscription, take care to not wait on such cancelled subscription.
     *
     * @param name Semaphore name.
     * @param includeOwners Return semaphore owners.
     * @param includeWaiters Return semaphore waiters.
     * @param watchData Subscribe to semaphore data change.
     * @param watchOwners Subscribe to semaphore owners change.
     * @return Semaphore description and subscription.
     */
    Mono<ChangesSubscription> subscribeToChanges(String name, boolean includeOwners, boolean includeWaiters,
                                                 boolean watchData, boolean watchOwners);

    /**
     * Subscribe to session state changes.
     *
     * @param consumer Session state consumer.
     */
    void subscribeToSessionState(Consumer<SessionState> consumer);

    /**
     * Returns current coordination session state.
     * @return Coordination session state.
     */
    SessionState getSessionState();

    /**
     * Unblock coordination session reconnection
     */
    void reconnect();

    /**
     * Create new YDB coordinator builder.
     * @param client YDB coordination client.
     * @param nodePath YDB coordination node path.
     * @return YDB coordinator builder.
     */
    static Coordinator.Builder newCoordinator(CoordinationClient client, String nodePath) {
        return new CoordinatorBuilderImpl(client, nodePath);
    }

    interface Builder {

        Builder operationTimeout(Duration timeout);

        Builder cancelAfter(Duration timeout);

        Builder deadlineAfter(Duration timeout);

        Builder nodeSelfCheckPeriod(Duration period);

        Builder nodeSessionGracePeriod(Duration period);

        Builder nodeReadConsistencyMode(NodeConsistencyMode mode);

        Builder nodeAttachConsistencyMode(NodeConsistencyMode mode);

        Builder sessionTimeout(Duration timeout);

        Builder sessionDescription(String description);

        Builder reconnectionPause(Duration duration);

        Builder blockTimeout(Duration timeout);

        Builder sessionStartTimeout(Duration timeout);

        Builder sessionStopTimeout(Duration timeout);

        Builder schedulerShutdownTimeout(Duration timeout);

        Builder pingPeriod(Duration period);

        Builder pingTimeout(Duration timeout);

        Builder lostPingsThreshold(long count);

        Builder sessionDeadlineAfter(Duration timeout);

        Builder semaphoreManagementTimeout(Duration timeout);

        Builder onSessionStart(Runnable action);

        Builder onSessionStop(Runnable action);

        Builder acquireSemaphoreTimeout(Duration timeout);

        Builder releaseSemaphoreTimeout(Duration timeout);

        Builder describeSemaphoreTimeout(Duration timeout);

        Builder acquireEnqueuedSemaphoreTimeout(Duration timeout);

        Builder executorPoolThreads(int count);

        Coordinator build();

        Runnable getOnSessionStart();

        Runnable getOnSessionStop();

    }

}
