package ru.yandex.intranet.d.datasource.coordination.impl;

import java.time.Duration;
import java.util.Set;

import ru.yandex.intranet.d.datasource.coordination.ClusterManager;
import ru.yandex.intranet.d.datasource.coordination.Coordinator;
import ru.yandex.intranet.d.datasource.coordination.HostInfoSupplier;
import ru.yandex.intranet.d.datasource.coordination.VersionSupplier;
import ru.yandex.intranet.d.datasource.coordination.model.cluster.ClusterLeader;
import ru.yandex.intranet.d.datasource.coordination.model.cluster.ClusterMembership;
import ru.yandex.intranet.d.datasource.coordination.model.cluster.NodeLeadershipStatus;
import ru.yandex.intranet.d.util.Barrier;

/**
 * YDB cluster manager builder.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public class ClusterManagerBuilderImpl implements ClusterManager.Builder {

    private final String leadershipSemaphoreName;
    private final String membershipSemaphoreName;
    private final Coordinator.Builder coordinatorBuilder;

    private Duration leadershipAcquisitionWaitDuration = Duration.ofMinutes(1);
    private Duration membershipAcquisitionWaitDuration = Duration.ofMinutes(1);
    private Duration describeSemaphoreBlockTimeout = Duration.ofSeconds(15);
    private Duration createSemaphoreBlockTimeout = Duration.ofSeconds(15);
    private int describeQueueCapacity = 1000;
    private Duration executorShutdownTimeout = Duration.ofSeconds(1);
    private HostInfoSupplier hostInfoSupplier = new DeployHostInfoSupplier();
    private VersionSupplier versionSupplier = new ArcadiaVersionSupplier();
    private Duration acquireSemaphoreBlockTimeout = Duration.ofSeconds(70);
    private Duration releaseSemaphoreBlockTimeout = Duration.ofSeconds(20);

    public ClusterManagerBuilderImpl(String leadershipSemaphoreName, String membershipSemaphoreName,
                                     Coordinator.Builder coordinatorBuilder) {
        this.leadershipSemaphoreName = leadershipSemaphoreName;
        this.membershipSemaphoreName = membershipSemaphoreName;
        this.coordinatorBuilder = coordinatorBuilder;
    }

    @Override
    public ClusterManager.Builder leadershipAcquisitionWaitDuration(Duration duration) {
        this.leadershipAcquisitionWaitDuration = duration;
        return this;
    }

    @Override
    public ClusterManager.Builder membershipAcquisitionWaitDuration(Duration duration) {
        this.membershipAcquisitionWaitDuration = duration;
        return this;
    }

    @Override
    public ClusterManager.Builder describeSemaphoreBlockTimeout(Duration timeout) {
        this.describeSemaphoreBlockTimeout = timeout;
        return this;
    }

    @Override
    public ClusterManager.Builder createSemaphoreBlockTimeout(Duration timeout) {
        this.createSemaphoreBlockTimeout = timeout;
        return this;
    }

    @Override
    public ClusterManager.Builder describeQueueCapacity(int capacity) {
        this.describeQueueCapacity = capacity;
        return this;
    }

    @Override
    public ClusterManager.Builder executorShutdownTimeout(Duration timeout) {
        this.executorShutdownTimeout = timeout;
        return this;
    }

    @Override
    public ClusterManager.Builder hostInfoSupplier(HostInfoSupplier supplier) {
        this.hostInfoSupplier = supplier;
        return this;
    }

    @Override
    public ClusterManager.Builder versionSupplier(VersionSupplier supplier) {
        this.versionSupplier = supplier;
        return this;
    }

    @Override
    public ClusterManager.Builder acquireSemaphoreBlockTimeout(Duration timeout) {
        this.acquireSemaphoreBlockTimeout = timeout;
        return this;
    }

    @Override
    public ClusterManager.Builder releaseSemaphoreBlockTimeout(Duration timeout) {
        this.releaseSemaphoreBlockTimeout = timeout;
        return this;
    }

    @Override
    public ClusterManager build() {
        Barrier sessionReadyBarrier = new Barrier();
        sessionReadyBarrier.close();
        Barrier leadershipRefreshBarrier = new Barrier();
        leadershipRefreshBarrier.close();
        Barrier membershipRefreshBarrier = new Barrier();
        membershipRefreshBarrier.close();
        LeadershipPublisher leadershipPublisher = new LeadershipPublisher();
        LeaderPublisher leaderPublisher = new LeaderPublisher();
        MembershipPublisher membershipPublisher = new MembershipPublisher();
        Runnable originalOnSessionStart = coordinatorBuilder.getOnSessionStart();
        coordinatorBuilder.onSessionStart(() -> {
            sessionReadyBarrier.open();
            originalOnSessionStart.run();
        });
        Runnable originalOnSessionStop = coordinatorBuilder.getOnSessionStop();
        coordinatorBuilder.onSessionStop(() -> {
            sessionReadyBarrier.close();
            leadershipRefreshBarrier.open();
            membershipRefreshBarrier.open();
            leadershipPublisher.nextStatus(NodeLeadershipStatus.UNDEFINED);
            leaderPublisher.nextLeader(new ClusterLeader(null, false));
            membershipPublisher.nextMembership(new ClusterMembership(Set.of(), false));
            originalOnSessionStop.run();
        });
        return new ClusterManagerImpl(coordinatorBuilder.build(), sessionReadyBarrier, leadershipRefreshBarrier,
                membershipRefreshBarrier, leadershipPublisher, leaderPublisher, membershipPublisher,
                leadershipSemaphoreName, membershipSemaphoreName, leadershipAcquisitionWaitDuration,
                membershipAcquisitionWaitDuration, describeSemaphoreBlockTimeout, createSemaphoreBlockTimeout,
                describeQueueCapacity, executorShutdownTimeout, hostInfoSupplier, versionSupplier,
                acquireSemaphoreBlockTimeout, releaseSemaphoreBlockTimeout);
    }

}
