package ru.yandex.intranet.d.datasource.coordination.impl;

import java.time.Duration;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.function.Consumer;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectReader;
import com.fasterxml.jackson.databind.ObjectWriter;
import com.fasterxml.jackson.datatype.jdk8.Jdk8Module;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import com.fasterxml.jackson.module.kotlin.KotlinModule;
import com.fasterxml.jackson.module.paramnames.ParameterNamesModule;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.datasource.coordination.ClusterManager;
import ru.yandex.intranet.d.datasource.coordination.Coordinator;
import ru.yandex.intranet.d.datasource.coordination.HostInfoSupplier;
import ru.yandex.intranet.d.datasource.coordination.VersionSupplier;
import ru.yandex.intranet.d.datasource.coordination.model.cluster.ClusterLeader;
import ru.yandex.intranet.d.datasource.coordination.model.cluster.ClusterMembership;
import ru.yandex.intranet.d.datasource.coordination.model.cluster.NodeInfo;
import ru.yandex.intranet.d.datasource.coordination.model.cluster.NodeLeadershipStatus;
import ru.yandex.intranet.d.datasource.coordination.model.session.SessionState;
import ru.yandex.intranet.d.util.Barrier;

/**
 * YDB cluster manager.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public class ClusterManagerImpl implements ClusterManager {

    private static final Logger LOG = LoggerFactory.getLogger(ClusterManagerImpl.class);

    private final Coordinator coordinator;
    private final Barrier sessionReadyBarrier;
    private final Barrier leaderElectionRefreshBarrier;
    private final Barrier nodeMembershipRefreshBarrier;
    private final LeadershipPublisher leadershipPublisher;
    private final LeaderPublisher leaderPublisher;
    private final MembershipPublisher membershipPublisher;
    private final String leadershipSemaphoreName;
    private final String membershipSemaphoreName;
    private final Duration leadershipAcquisitionWaitDuration;
    private final Duration membershipAcquisitionWaitDuration;
    private final Duration describeSemaphoreBlockTimeout;
    private final Duration createSemaphoreBlockTimeout;
    private final int describeQueueCapacity;
    private final Duration executorShutdownTimeout;
    private final HostInfoSupplier hostInfoSupplier;
    private final Duration acquireSemaphoreBlockTimeout;
    private final Duration releaseSemaphoreBlockTimeout;
    private final String nodeId = UUID.randomUUID().toString();
    private final String currentVersion;

    private volatile boolean started;
    private volatile LeaderElection leaderElection;
    private volatile NodeMembership nodeMembership;
    private volatile Thread leaderElectionThread;
    private volatile Thread nodeMembershipThread;

    @SuppressWarnings("ParameterNumber")
    public ClusterManagerImpl(Coordinator coordinator, Barrier sessionReadyBarrier,
                              Barrier leaderElectionRefreshBarrier, Barrier nodeMembershipRefreshBarrier,
                              LeadershipPublisher leadershipPublisher, LeaderPublisher leaderPublisher,
                              MembershipPublisher membershipPublisher, String leadershipSemaphoreName,
                              String membershipSemaphoreName, Duration leadershipAcquisitionWaitDuration,
                              Duration membershipAcquisitionWaitDuration, Duration describeSemaphoreBlockTimeout,
                              Duration createSemaphoreBlockTimeout, int describeQueueCapacity,
                              Duration executorShutdownTimeout, HostInfoSupplier hostInfoSupplier,
                              VersionSupplier versionSupplier, Duration acquireSemaphoreBlockTimeout,
                              Duration releaseSemaphoreBlockTimeout) {
        this.coordinator = coordinator;
        this.sessionReadyBarrier = sessionReadyBarrier;
        this.leaderElectionRefreshBarrier = leaderElectionRefreshBarrier;
        this.nodeMembershipRefreshBarrier = nodeMembershipRefreshBarrier;
        this.leadershipPublisher = leadershipPublisher;
        this.leaderPublisher = leaderPublisher;
        this.membershipPublisher = membershipPublisher;
        this.leadershipSemaphoreName = leadershipSemaphoreName;
        this.membershipSemaphoreName = membershipSemaphoreName;
        this.leadershipAcquisitionWaitDuration = leadershipAcquisitionWaitDuration;
        this.membershipAcquisitionWaitDuration = membershipAcquisitionWaitDuration;
        this.describeSemaphoreBlockTimeout = describeSemaphoreBlockTimeout;
        this.createSemaphoreBlockTimeout = createSemaphoreBlockTimeout;
        this.describeQueueCapacity = describeQueueCapacity;
        this.executorShutdownTimeout = executorShutdownTimeout;
        this.hostInfoSupplier = hostInfoSupplier;
        this.acquireSemaphoreBlockTimeout = acquireSemaphoreBlockTimeout;
        this.releaseSemaphoreBlockTimeout = releaseSemaphoreBlockTimeout;
        this.currentVersion = versionSupplier.getVersion();
    }

    @Override
    public void start() {
        synchronized (this) {
            LOG.info("Starting cluster manager...");
            if (started) {
                LOG.info("Cluster manager was already started");
                return;
            }
            coordinator.start();
            ObjectMapper objectMapper = createObjectMapper();
            ObjectReader nodeInfoReader = objectMapper.readerFor(NodeInfo.class);
            ObjectWriter nodeInfoWriter = objectMapper.writerFor(NodeInfo.class);
            NodeInfo nodeInfo = prepareNodeInfo();
            leaderElection = new LeaderElection(coordinator, sessionReadyBarrier,
                    leaderElectionRefreshBarrier, leadershipPublisher, leaderPublisher, nodeId,
                    leadershipSemaphoreName, leadershipAcquisitionWaitDuration, describeSemaphoreBlockTimeout,
                    createSemaphoreBlockTimeout, acquireSemaphoreBlockTimeout, releaseSemaphoreBlockTimeout,
                    executorShutdownTimeout, nodeInfoReader, nodeInfoWriter, nodeInfo, describeQueueCapacity);
            nodeMembership = new NodeMembership(coordinator, sessionReadyBarrier,
                    nodeMembershipRefreshBarrier, membershipPublisher, nodeId, membershipSemaphoreName,
                    membershipAcquisitionWaitDuration, describeSemaphoreBlockTimeout, createSemaphoreBlockTimeout,
                    acquireSemaphoreBlockTimeout, releaseSemaphoreBlockTimeout, executorShutdownTimeout,
                    nodeInfoReader, nodeInfoWriter, nodeInfo, describeQueueCapacity);
            leaderElectionThread = new Thread(leaderElection, "ClusterManagerLeaderElection");
            nodeMembershipThread = new Thread(nodeMembership, "ClusterManagerNodeMembership");
            leaderElectionThread.setDaemon(true);
            nodeMembershipThread.setDaemon(true);
            leaderElectionThread.start();
            nodeMembershipThread.start();
            started = true;
            LOG.info("Cluster manager started");
        }
    }

    @Override
    public void stop(Runnable onStop) {
        synchronized (this) {
            LOG.info("Stopping cluster manager...");
            if (!started) {
                LOG.info("Cluster manager is already stopped");
                onStop.run();
                return;
            }
            leaderElection.stop();
            leaderElectionThread.interrupt();
            leaderElectionRefreshBarrier.open();
            nodeMembership.stop();
            nodeMembershipThread.interrupt();
            nodeMembershipRefreshBarrier.open();
            sessionReadyBarrier.open();
            Thread waitForStopThread = new Thread(() -> {
                LOG.info("Waiting until cluster manager leader election thread is stopped...");
                boolean interrupted = false;
                try {
                    leaderElectionThread.join();
                } catch (InterruptedException e) {
                    interrupted = true;
                }
                leaderElectionThread = null;
                leaderElection = null;
                LOG.info("Cluster manager leader election thread is stopped");
                LOG.info("Waiting until cluster manager node membership thread is stopped...");
                try {
                    nodeMembershipThread.join();
                } catch (InterruptedException e) {
                    interrupted = true;
                }
                nodeMembershipThread = null;
                nodeMembership = null;
                LOG.info("Cluster manager node membership thread is stopped");
                if (interrupted) {
                    Thread.currentThread().interrupt();
                }
                coordinator.stop(() -> {
                    started = false;
                    onStop.run();
                    LOG.info("Cluster manager stopped");
                });
            }, "ClusterManagerFinalizer");
            waitForStopThread.setDaemon(true);
            waitForStopThread.start();
        }
    }

    @Override
    public boolean isRunning() {
        return started;
    }

    @Override
    public Mono<Boolean> isLeader() {
        LeaderElection election = this.leaderElection;
        if (election != null) {
            return election.isLeader();
        } else {
            return Mono.error(new IllegalStateException("Cluster manager is not started"));
        }
    }

    @Override
    public Optional<Boolean> isLeaderCached() {
        LeaderElection election = this.leaderElection;
        if (election != null) {
            return election.isLeaderCached();
        } else {
            return Optional.empty();
        }
    }

    @Override
    public Mono<Optional<NodeInfo>> getLeader() {
        LeaderElection election = this.leaderElection;
        if (election != null) {
            return election.getLeader();
        } else {
            return Mono.error(new IllegalStateException("Cluster manager is not started"));
        }
    }

    @Override
    public Optional<NodeInfo> getLeaderCached() {
        LeaderElection election = this.leaderElection;
        if (election != null) {
            return election.getLeaderCached();
        } else {
            return Optional.empty();
        }
    }

    @Override
    public Mono<Boolean> isMember() {
        NodeMembership membership = this.nodeMembership;
        if (membership != null) {
            return membership.isMember();
        } else {
            return Mono.error(new IllegalStateException("Cluster manager is not started"));
        }
    }

    @Override
    public Optional<Boolean> isMemberCached() {
        NodeMembership membership = this.nodeMembership;
        if (membership != null) {
            return membership.isMemberCached();
        } else {
            return Optional.empty();
        }
    }

    @Override
    public Mono<Set<NodeInfo>> getMembers() {
        NodeMembership membership = this.nodeMembership;
        if (membership != null) {
            return membership.getMembers();
        } else {
            return Mono.error(new IllegalStateException("Cluster manager is not started"));
        }
    }

    @Override
    public Optional<Set<NodeInfo>> getMembersCached() {
        NodeMembership membership = this.nodeMembership;
        if (membership != null) {
            return membership.getMembersCached();
        } else {
            return Optional.empty();
        }
    }

    @Override
    public Set<NodeInfo> getLastSeenClusterMembers() {
        NodeMembership membership = this.nodeMembership;
        if (membership != null) {
            return membership.getLastSeenClusterMembers();
        } else {
            return Set.of();
        }
    }

    @Override
    public void addLeadershipSubscriber(Consumer<NodeLeadershipStatus> consumer) {
        leadershipPublisher.addSubscriber(consumer);
    }

    @Override
    public void addLeaderSubscriber(Consumer<ClusterLeader> consumer) {
        leaderPublisher.addSubscriber(consumer);
    }

    @Override
    public void addMembershipSubscriber(Consumer<ClusterMembership> consumer) {
        membershipPublisher.addSubscriber(consumer);
    }

    @Override
    public String getNodeId() {
        return nodeId;
    }

    @Override
    public String getCurrentVersion() {
        return currentVersion;
    }

    @Override
    public SessionState getSessionState() {
        return coordinator.getSessionState();
    }

    private NodeInfo prepareNodeInfo() {
        return new NodeInfo(nodeId, hostInfoSupplier.getTransientFqdn(), hostInfoSupplier.getPersistentFqdn(),
                hostInfoSupplier.getHostIps(), currentVersion);
    }

    private static ObjectMapper createObjectMapper() {
        ObjectMapper mapper = new ObjectMapper();
        mapper.registerModule(new Jdk8Module());
        mapper.registerModule(new JavaTimeModule());
        mapper.registerModule(new ParameterNamesModule());
        mapper.registerModule(new KotlinModule.Builder().build());
        return mapper;
    }

}
