package ru.yandex.intranet.d.datasource.coordination.impl;

import java.io.IOException;
import java.time.Duration;
import java.util.List;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.protobuf.ByteString;
import com.google.protobuf.Message;
import com.google.protobuf.util.Durations;
import com.yandex.ydb.OperationProtos;
import com.yandex.ydb.StatusCodesProtos;
import com.yandex.ydb.coordination.AlterNodeRequest;
import com.yandex.ydb.coordination.Config;
import com.yandex.ydb.coordination.ConsistencyMode;
import com.yandex.ydb.coordination.CreateNodeRequest;
import com.yandex.ydb.coordination.DescribeNodeRequest;
import com.yandex.ydb.coordination.DescribeNodeResult;
import com.yandex.ydb.coordination.DropNodeRequest;
import com.yandex.ydb.coordination.RateLimiterCountersMode;
import com.yandex.ydb.coordination.SemaphoreDescription;
import com.yandex.ydb.coordination.SemaphoreSession;
import com.yandex.ydb.coordination.SessionRequest;
import com.yandex.ydb.coordination.SessionResponse;
import com.yandex.ydb.core.Issue;
import com.yandex.ydb.core.Operations;
import com.yandex.ydb.core.Result;
import com.yandex.ydb.core.Status;
import com.yandex.ydb.core.StatusCode;
import com.yandex.ydb.core.rpc.OutStreamObserver;
import com.yandex.ydb.core.rpc.StreamObserver;
import com.yandex.ydb.scheme.SchemeOperationProtos;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.datasource.coordination.CoordinationClient;
import ru.yandex.intranet.d.datasource.coordination.model.AlterCoordinationNodeRequest;
import ru.yandex.intranet.d.datasource.coordination.model.CoordinationNode;
import ru.yandex.intranet.d.datasource.coordination.model.CoordinationNodeDescription;
import ru.yandex.intranet.d.datasource.coordination.model.CreateCoordinationNodeRequest;
import ru.yandex.intranet.d.datasource.coordination.model.DescribeCoordinationNodeRequest;
import ru.yandex.intranet.d.datasource.coordination.model.DropCoordinationNodeRequest;
import ru.yandex.intranet.d.datasource.coordination.model.NodeConfig;
import ru.yandex.intranet.d.datasource.coordination.model.NodeConsistencyMode;
import ru.yandex.intranet.d.datasource.coordination.model.NodeRateLimiterCountersMode;
import ru.yandex.intranet.d.datasource.coordination.model.OperationMode;
import ru.yandex.intranet.d.datasource.coordination.model.OperationParameters;
import ru.yandex.intranet.d.datasource.coordination.model.SchemeEntry;
import ru.yandex.intranet.d.datasource.coordination.model.SchemeEntryPermission;
import ru.yandex.intranet.d.datasource.coordination.model.SchemeEntryType;
import ru.yandex.intranet.d.datasource.coordination.model.TargetNodeConfig;
import ru.yandex.intranet.d.datasource.coordination.model.session.AcquireSemaphorePendingResponse;
import ru.yandex.intranet.d.datasource.coordination.model.session.AcquireSemaphoreRequest;
import ru.yandex.intranet.d.datasource.coordination.model.session.AcquireSemaphoreResultResponse;
import ru.yandex.intranet.d.datasource.coordination.model.session.CoordinationSemaphoreDescription;
import ru.yandex.intranet.d.datasource.coordination.model.session.CoordinationSemaphoreSession;
import ru.yandex.intranet.d.datasource.coordination.model.session.CoordinationSessionRequest;
import ru.yandex.intranet.d.datasource.coordination.model.session.CoordinationSessionResponse;
import ru.yandex.intranet.d.datasource.coordination.model.session.CreateSemaphoreRequest;
import ru.yandex.intranet.d.datasource.coordination.model.session.CreateSemaphoreResultResponse;
import ru.yandex.intranet.d.datasource.coordination.model.session.DeleteSemaphoreRequest;
import ru.yandex.intranet.d.datasource.coordination.model.session.DeleteSemaphoreResultResponse;
import ru.yandex.intranet.d.datasource.coordination.model.session.DescribeSemaphoreChangedResponse;
import ru.yandex.intranet.d.datasource.coordination.model.session.DescribeSemaphoreRequest;
import ru.yandex.intranet.d.datasource.coordination.model.session.DescribeSemaphoreResultResponse;
import ru.yandex.intranet.d.datasource.coordination.model.session.FailureResponse;
import ru.yandex.intranet.d.datasource.coordination.model.session.PingPongRequest;
import ru.yandex.intranet.d.datasource.coordination.model.session.PingPongResponse;
import ru.yandex.intranet.d.datasource.coordination.model.session.ReleaseSemaphoreRequest;
import ru.yandex.intranet.d.datasource.coordination.model.session.ReleaseSemaphoreResultResponse;
import ru.yandex.intranet.d.datasource.coordination.model.session.SessionOperationParameters;
import ru.yandex.intranet.d.datasource.coordination.model.session.SessionStartRequest;
import ru.yandex.intranet.d.datasource.coordination.model.session.SessionStartedResponse;
import ru.yandex.intranet.d.datasource.coordination.model.session.SessionStopRequest;
import ru.yandex.intranet.d.datasource.coordination.model.session.SessionStoppedResponse;
import ru.yandex.intranet.d.datasource.coordination.model.session.UpdateSemaphoreRequest;
import ru.yandex.intranet.d.datasource.coordination.model.session.UpdateSemaphoreResultResponse;
import ru.yandex.intranet.d.datasource.coordination.rpc.CoordinationRpc;

/**
 * YDB coordination client.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public class CoordinationClientImpl implements CoordinationClient {

    private static final Logger LOG = LoggerFactory.getLogger(CoordinationClientImpl.class);

    private final CoordinationRpc coordinationRpc;
    private final Object sendMutex = new Object();

    public CoordinationClientImpl(CoordinationRpc coordinationRpc) {
        this.coordinationRpc = coordinationRpc;
    }

    @Override
    public void close() throws IOException {
        coordinationRpc.close();
    }

    @Override
    public Mono<CoordinationNode> createNode(CreateCoordinationNodeRequest request) {
        return Mono.fromFuture(() -> coordinationRpc.createNode(toCreateNodeRequest(request),
                request.getOperationParameters().map(OperationParameters::getDeadlineAfterNanos).orElse(0L)))
                .map(r -> r.expect("Failed to create coordination node"))
                .map(r -> {
                    checkOperationStatus(r.getOperation()).expect("Failed to create coordination node");
                    return new CoordinationNode(request.getPath());
                });
    }

    @Override
    public Mono<Void> alterNode(AlterCoordinationNodeRequest request) {
        return Mono.fromFuture(() -> coordinationRpc.alterNode(toAlterNodeRequest(request),
                request.getOperationParameters().map(OperationParameters::getDeadlineAfterNanos).orElse(0L)))
                .map(r -> r.expect("Failed to alter coordination node"))
                .flatMap(r -> {
                    checkOperationStatus(r.getOperation()).expect("Failed to alter coordination node");
                    return Mono.empty();
                });
    }

    @Override
    public Mono<Void> dropNode(DropCoordinationNodeRequest request) {
        return Mono.fromFuture(() -> coordinationRpc.dropNode(toDropNodeRequest(request),
                request.getOperationParameters().map(OperationParameters::getDeadlineAfterNanos).orElse(0L)))
                .map(r -> r.expect("Failed to drop coordination node"))
                .flatMap(r -> Mono.justOrEmpty(checkOperationStatus(r.getOperation())
                        .expect("Failed to drop coordination node")));
    }

    @Override
    public Mono<CoordinationNodeDescription> describeNode(DescribeCoordinationNodeRequest request) {
        return Mono.fromFuture(() -> coordinationRpc.describeNode(toDescribeNodeRequest(request),
                request.getOperationParameters().map(OperationParameters::getDeadlineAfterNanos).orElse(0L)))
                .map(r -> r.expect("Failed to describe coordination node"))
                .flatMap(r -> Mono.justOrEmpty(unpackOperationResult(r.getOperation(), DescribeNodeResult.class,
                        this::fromDescribeNodeResult).expect("Failed to describe coordination node")));
    }

    @Override
    public OutStreamObserver<CoordinationSessionRequest> session(SessionOperationParameters params,
                                                          StreamObserver<CoordinationSessionResponse> response) {
        OutStreamObserver<SessionRequest> request = coordinationRpc.session(new StreamObserver<>() {
            @Override
            public void onNext(SessionResponse value) {
                fromSessionResponse(value).ifPresent(response::onNext);
            }
            @Override
            public void onError(Status status) {
                response.onError(status);
            }
            @Override
            public void onCompleted() {
                response.onCompleted();
            }
        }, params.getDeadlineAfterNanos());
        return new OutStreamObserver<>() {
            @Override
            public void onNext(CoordinationSessionRequest value) {
                synchronized (sendMutex) {
                    request.onNext(toSessionRequest(value));
                }
            }
            @Override
            public void onError(Throwable t) {
                synchronized (sendMutex) {
                    request.onError(t);
                }
            }
            @Override
            public void onCompleted() {
                synchronized (sendMutex) {
                    request.onCompleted();
                }
            }
        };
    }

    private Result<Void> checkOperationStatus(OperationProtos.Operation operation) {
        if (!operation.getReady()) {
            throw new IllegalStateException("Operation is not ready");
        }
        Status status = Operations.status(operation);
        if (status.isSuccess()) {
            return Result.success(null);
        } else {
            return Result.fail(status);
        }
    }

    private <M extends Message, V> Result<V> unpackOperationResult(OperationProtos.Operation operation,
                                                                   Class<M> resultClass, Function<M, V> mapper) {
        if (!operation.getReady()) {
            throw new IllegalStateException("Operation is not ready");
        }
        Status status = Operations.status(operation);
        if (status.isSuccess()) {
            M resultMessage = Operations.unpackResult(operation, resultClass);
            return Result.success(mapper.apply(resultMessage));
        } else {
            return Result.fail(status);
        }
    }

    private CreateNodeRequest toCreateNodeRequest(CreateCoordinationNodeRequest request) {
        CreateNodeRequest.Builder builder = CreateNodeRequest.newBuilder();
        builder.setPath(request.getPath());
        request.getConfiguration().ifPresent(v -> builder.setConfig(toConfig("", v)));
        request.getOperationParameters().ifPresent(v -> builder.setOperationParams(toOperationParams(v)));
        return builder.build();
    }

    private AlterNodeRequest toAlterNodeRequest(AlterCoordinationNodeRequest request) {
        AlterNodeRequest.Builder builder = AlterNodeRequest.newBuilder();
        builder.setPath(request.getPath());
        request.getConfiguration().ifPresent(v -> builder.setConfig(toConfig("", v)));
        request.getOperationParameters().ifPresent(v -> builder.setOperationParams(toOperationParams(v)));
        return builder.build();
    }

    private DropNodeRequest toDropNodeRequest(DropCoordinationNodeRequest request) {
        DropNodeRequest.Builder builder = DropNodeRequest.newBuilder();
        builder.setPath(request.getPath());
        request.getOperationParameters().ifPresent(v -> builder.setOperationParams(toOperationParams(v)));
        return builder.build();
    }

    private DescribeNodeRequest toDescribeNodeRequest(DescribeCoordinationNodeRequest request) {
        DescribeNodeRequest.Builder builder = DescribeNodeRequest.newBuilder();
        builder.setPath(request.getPath());
        request.getOperationParameters().ifPresent(v -> builder.setOperationParams(toOperationParams(v)));
        return builder.build();
    }

    private Config toConfig(String path, TargetNodeConfig config) {
        Config.Builder builder = Config.newBuilder();
        builder.setPath(path);
        builder.setSelfCheckPeriodMillis((int) config.getSelfCheckPeriod().toMillis());
        builder.setSessionGracePeriodMillis((int) config.getSessionGracePeriod().toMillis());
        builder.setReadConsistencyMode(toConsistencyMode(config.getReadConsistencyMode()));
        builder.setAttachConsistencyMode(toConsistencyMode(config.getAttachConsistencyMode()));
        builder.setRateLimiterCountersMode(toRateLimiterCountersMode(config.getRateLimiterCountersMode()));
        return builder.build();
    }

    private OperationProtos.OperationParams toOperationParams(OperationParameters params) {
        OperationProtos.OperationParams.Builder builder = OperationProtos.OperationParams.newBuilder();
        builder.setOperationMode(toOperationMode(params.getOperationMode()));
        params.getOperationTimeout().ifPresent(v -> builder.setOperationTimeout(Durations.fromNanos(v.toNanos())));
        params.getCancelAfter().ifPresent(v -> builder.setCancelAfter(Durations.fromNanos(v.toNanos())));
        builder.putAllLabels(params.getLabels());
        return builder.build();
    }

    private CoordinationNodeDescription fromDescribeNodeResult(DescribeNodeResult value) {
        return new CoordinationNodeDescription(fromSchemeEntry(value.getSelf()), fromConfig(value.getConfig()));
    }

    private SchemeEntry fromSchemeEntry(SchemeOperationProtos.Entry value) {
        return new SchemeEntry(value.getName(), value.getOwner(), fromSchemeEntryType(value.getType()),
                value.getEffectivePermissionsList().stream().map(this::fromPermission).collect(Collectors.toList()),
                value.getPermissionsList().stream().map(this::fromPermission).collect(Collectors.toList()));
    }

    private NodeConfig fromConfig(Config value) {
        return new NodeConfig(value.getPath(), Duration.ofMillis(value.getSelfCheckPeriodMillis()),
                Duration.ofMillis(value.getSessionGracePeriodMillis()),
                fromConsistencyMode(value.getReadConsistencyMode()),
                fromConsistencyMode(value.getAttachConsistencyMode()),
                fromRateLimiterCountersMode(value.getRateLimiterCountersMode()));
    }

    private SchemeEntryPermission fromPermission(SchemeOperationProtos.Permissions value) {
        return new SchemeEntryPermission(value.getSubject(), List.copyOf(value.getPermissionNamesList()));
    }

    private Optional<CoordinationSessionResponse> fromSessionResponse(SessionResponse response) {
        if (response.hasPing()) {
            return Optional.of(CoordinationSessionResponse.ping(new PingPongResponse(response
                    .getPing().getOpaque())));
        } else if (response.hasPong()) {
            return Optional.of(CoordinationSessionResponse.pong(new PingPongResponse(response.getPong()
                    .getOpaque())));
        } else if (response.hasFailure()) {
            return Optional.of(CoordinationSessionResponse.failure(new FailureResponse(fromStatus(response
                    .getFailure().getStatus(), response.getFailure().getIssuesList()))));
        } else if (response.hasSessionStarted()) {
            return Optional.of(CoordinationSessionResponse.sessionStarted(new SessionStartedResponse(response
                    .getSessionStarted().getSessionId(), Duration.ofMillis(response.getSessionStarted()
                    .getTimeoutMillis()))));
        } else if (response.hasSessionStopped()) {
            return Optional.of(CoordinationSessionResponse.sessionStopped(new SessionStoppedResponse(response
                    .getSessionStopped().getSessionId())));
        } else if (response.hasAcquireSemaphorePending()) {
            return Optional.of(CoordinationSessionResponse
                    .acquireSemaphorePending(new AcquireSemaphorePendingResponse(response
                            .getAcquireSemaphorePending().getReqId())));
        } else if (response.hasAcquireSemaphoreResult()) {
            return Optional.of(CoordinationSessionResponse
                    .acquireSemaphoreResult(new AcquireSemaphoreResultResponse(response
                            .getAcquireSemaphoreResult().getReqId(), fromStatus(response.getAcquireSemaphoreResult()
                            .getStatus(), response.getAcquireSemaphoreResult().getIssuesList()), response
                            .getAcquireSemaphoreResult().getAcquired())));
        } else if (response.hasReleaseSemaphoreResult()) {
            return Optional.of(CoordinationSessionResponse
                    .releaseSemaphoreResult(new ReleaseSemaphoreResultResponse(response
                            .getReleaseSemaphoreResult().getReqId(), fromStatus(response.getReleaseSemaphoreResult()
                            .getStatus(), response.getReleaseSemaphoreResult().getIssuesList()), response
                            .getReleaseSemaphoreResult().getReleased())));
        } else if (response.hasDescribeSemaphoreResult()) {
            return Optional.of(CoordinationSessionResponse
                    .describeSemaphoreResult(new DescribeSemaphoreResultResponse(response
                            .getDescribeSemaphoreResult().getReqId(), fromStatus(response.getDescribeSemaphoreResult()
                            .getStatus(), response.getDescribeSemaphoreResult().getIssuesList()),
                            fromDescription(response.getDescribeSemaphoreResult().getSemaphoreDescription()),
                            response.getDescribeSemaphoreResult().getWatchAdded())));
        } else if (response.hasDescribeSemaphoreChanged()) {
            return Optional.of(CoordinationSessionResponse
                    .describeSemaphoreChanged(new DescribeSemaphoreChangedResponse(response
                            .getDescribeSemaphoreChanged().getReqId(), response.getDescribeSemaphoreChanged()
                            .getDataChanged(), response.getDescribeSemaphoreChanged().getOwnersChanged())));
        } else if (response.hasCreateSemaphoreResult()) {
            return Optional.of(CoordinationSessionResponse
                    .createSemaphoreResult(new CreateSemaphoreResultResponse(response
                            .getCreateSemaphoreResult().getReqId(), fromStatus(response.getCreateSemaphoreResult()
                            .getStatus(), response.getCreateSemaphoreResult().getIssuesList()))));
        } else if (response.hasUpdateSemaphoreResult()) {
            return Optional.of(CoordinationSessionResponse
                    .updateSemaphoreResult(new UpdateSemaphoreResultResponse(response
                            .getUpdateSemaphoreResult().getReqId(), fromStatus(response.getUpdateSemaphoreResult()
                            .getStatus(), response.getUpdateSemaphoreResult().getIssuesList()))));
        } else if (response.hasDeleteSemaphoreResult()) {
            return Optional.of(CoordinationSessionResponse
                    .deleteSemaphoreResult(new DeleteSemaphoreResultResponse(response
                            .getDeleteSemaphoreResult().getReqId(), fromStatus(response.getDeleteSemaphoreResult()
                            .getStatus(), response.getDeleteSemaphoreResult().getIssuesList()))));
        } else {
            LOG.warn("Unexpected coordination response type {}", response.getResponseCase());
            return Optional.empty();
        }
    }

    private Status fromStatus(StatusCodesProtos.StatusIds.StatusCode statusCode,
                              List<com.yandex.ydb.YdbIssueMessage.IssueMessage> issues) {
        return Status.of(StatusCode.fromProto(statusCode), Issue.fromPb(issues));
    }

    private CoordinationSemaphoreDescription fromDescription(SemaphoreDescription description) {
        return new CoordinationSemaphoreDescription(description.getName(), description.getData().toByteArray(),
                description.getCount(), description.getLimit(), description.getEphemeral(), description
                .getOwnersList().stream().map(this::fromSemaphoreSession).collect(Collectors.toList()),
                description.getWaitersList().stream().map(this::fromSemaphoreSession).collect(Collectors.toList()));
    }

    private CoordinationSemaphoreSession fromSemaphoreSession(SemaphoreSession semaphoreSession) {
        return new CoordinationSemaphoreSession(semaphoreSession.getOrderId(), semaphoreSession.getSessionId(),
                Duration.ofMillis(semaphoreSession.getTimeoutMillis()), semaphoreSession.getCount(),
                semaphoreSession.getData().toByteArray());
    }

    private SessionRequest toSessionRequest(CoordinationSessionRequest request) {
        return request.match(new CoordinationSessionRequest.Cases<>() {
            @Override
            public SessionRequest ping(PingPongRequest request) {
                return SessionRequest.newBuilder().setPing(SessionRequest.PingPong.newBuilder()
                        .setOpaque(request.getOpaque()).build()).build();
            }
            @Override
            public SessionRequest pong(PingPongRequest request) {
                return SessionRequest.newBuilder().setPong(SessionRequest.PingPong.newBuilder()
                        .setOpaque(request.getOpaque()).build()).build();
            }
            @Override
            public SessionRequest sessionStart(SessionStartRequest request) {
                return SessionRequest.newBuilder().setSessionStart(SessionRequest.SessionStart.newBuilder()
                        .setPath(request.getPath()).setSessionId(request.getSessionId())
                        .setTimeoutMillis(request.getTimeout().toMillis()).setDescription(request.getDescription())
                        .setSeqNo(request.getSeqNo())
                        .setProtectionKey(ByteString.copyFrom(request.getProtectionKey()))
                        .build()).build();
            }
            @Override
            public SessionRequest sessionStop(SessionStopRequest request) {
                return SessionRequest.newBuilder().setSessionStop(SessionRequest.SessionStop.newBuilder()
                        .build()).build();
            }
            @Override
            public SessionRequest acquireSemaphore(AcquireSemaphoreRequest request) {
                return SessionRequest.newBuilder().setAcquireSemaphore(SessionRequest.AcquireSemaphore.newBuilder()
                        .setReqId(request.getReqId()).setName(request.getName())
                        .setTimeoutMillis(request.getTimeout().toMillis()).setCount(request.getCount())
                        .setData(ByteString.copyFrom(request.getData()))
                        .setEphemeral(request.isEphemeral())
                        .build()).build();
            }
            @Override
            public SessionRequest releaseSemaphore(ReleaseSemaphoreRequest request) {
                return SessionRequest.newBuilder().setReleaseSemaphore(SessionRequest.ReleaseSemaphore.newBuilder()
                        .setReqId(request.getReqId()).setName(request.getName()).build()).build();
            }
            @Override
            public SessionRequest describeSemaphore(DescribeSemaphoreRequest request) {
                return SessionRequest.newBuilder().setDescribeSemaphore(SessionRequest.DescribeSemaphore.newBuilder()
                        .setReqId(request.getReqId()).setName(request.getName())
                        .setIncludeOwners(request.isIncludeOwners()).setIncludeWaiters(request.isIncludeWaiters())
                        .setWatchData(request.isWatchData()).setWatchOwners(request.isWatchOwners()).build()).build();
            }
            @Override
            public SessionRequest createSemaphore(CreateSemaphoreRequest request) {
                return SessionRequest.newBuilder().setCreateSemaphore(SessionRequest.CreateSemaphore.newBuilder()
                        .setReqId(request.getReqId()).setName(request.getName())
                        .setLimit(request.getLimit()).setData(ByteString.copyFrom(request.getData()))
                        .build()).build();
            }
            @Override
            public SessionRequest updateSemaphore(UpdateSemaphoreRequest request) {
                return SessionRequest.newBuilder().setUpdateSemaphore(SessionRequest.UpdateSemaphore.newBuilder()
                        .setReqId(request.getReqId()).setName(request.getName())
                        .setData(ByteString.copyFrom(request.getData())).build()).build();
            }
            @Override
            public SessionRequest deleteSemaphore(DeleteSemaphoreRequest request) {
                return SessionRequest.newBuilder().setDeleteSemaphore(SessionRequest.DeleteSemaphore.newBuilder()
                        .setReqId(request.getReqId()).setName(request.getName())
                        .setForce(request.isForce()).build()).build();
            }
        });
    }

    private OperationProtos.OperationParams.OperationMode toOperationMode(OperationMode mode) {
        switch (mode) {
            case UNSPECIFIED:
                return OperationProtos.OperationParams.OperationMode.OPERATION_MODE_UNSPECIFIED;
            case SYNC:
                return OperationProtos.OperationParams.OperationMode.SYNC;
            case ASYNC:
                return OperationProtos.OperationParams.OperationMode.ASYNC;
            default:
                throw new IllegalArgumentException("Unsupported operation mode: " + mode);
        }
    }

    private ConsistencyMode toConsistencyMode(NodeConsistencyMode mode) {
        switch (mode) {
            case UNSET:
                return ConsistencyMode.CONSISTENCY_MODE_UNSET;
            case STRICT:
                return ConsistencyMode.CONSISTENCY_MODE_STRICT;
            case RELAXED:
                return ConsistencyMode.CONSISTENCY_MODE_RELAXED;
            default:
                throw new IllegalArgumentException("Unsupported consistency mode: " + mode);
        }
    }

    private RateLimiterCountersMode toRateLimiterCountersMode(NodeRateLimiterCountersMode mode) {
        switch (mode) {
            case UNSET:
                return RateLimiterCountersMode.RATE_LIMITER_COUNTERS_MODE_UNSET;
            case AGGREGATED:
                return RateLimiterCountersMode.RATE_LIMITER_COUNTERS_MODE_AGGREGATED;
            case DETAILED:
                return RateLimiterCountersMode.RATE_LIMITER_COUNTERS_MODE_DETAILED;
            default:
                throw new IllegalArgumentException("Unsupported rate limiter counters mode: " + mode);
        }
    }

    private SchemeEntryType fromSchemeEntryType(SchemeOperationProtos.Entry.Type type) {
        switch (type) {
            case TYPE_UNSPECIFIED:
                return SchemeEntryType.UNSPECIFIED;
            case DIRECTORY:
                return SchemeEntryType.DIRECTORY;
            case TABLE:
                return SchemeEntryType.TABLE;
            case PERS_QUEUE_GROUP:
                return SchemeEntryType.PERS_QUEUE_GROUP;
            case DATABASE:
                return SchemeEntryType.DATABASE;
            case RTMR_VOLUME:
                return SchemeEntryType.RTMR_VOLUME;
            case BLOCK_STORE_VOLUME:
                return SchemeEntryType.BLOCK_STORE_VOLUME;
            case COORDINATION_NODE:
                return SchemeEntryType.COORDINATION_NODE;
            default:
                throw new IllegalArgumentException("Unsupported scheme entry type: " + type);
        }
    }

    private NodeConsistencyMode fromConsistencyMode(ConsistencyMode mode) {
        switch (mode) {
            case CONSISTENCY_MODE_UNSET:
                return NodeConsistencyMode.UNSET;
            case CONSISTENCY_MODE_STRICT:
                return NodeConsistencyMode.STRICT;
            case CONSISTENCY_MODE_RELAXED:
                return NodeConsistencyMode.RELAXED;
            default:
                throw new IllegalArgumentException("Unsupported consistency mode: " + mode);
        }
    }

    private NodeRateLimiterCountersMode fromRateLimiterCountersMode(RateLimiterCountersMode mode) {
        switch (mode) {
            case RATE_LIMITER_COUNTERS_MODE_UNSET:
                return NodeRateLimiterCountersMode.UNSET;
            case RATE_LIMITER_COUNTERS_MODE_AGGREGATED:
                return NodeRateLimiterCountersMode.AGGREGATED;
            case RATE_LIMITER_COUNTERS_MODE_DETAILED:
                return NodeRateLimiterCountersMode.DETAILED;
            default:
                throw new IllegalArgumentException("Unsupported rate limiter counters mode: " + mode);
        }
    }

}
