package ru.yandex.intranet.d.datasource.coordination.impl;

import java.time.Duration;

import ru.yandex.intranet.d.datasource.coordination.CoordinationClient;
import ru.yandex.intranet.d.datasource.coordination.Coordinator;
import ru.yandex.intranet.d.datasource.coordination.model.NodeConsistencyMode;

/**
 * YDB coordinator builder.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public class CoordinatorBuilderImpl implements Coordinator.Builder {

    private final CoordinationClient client;
    private final String nodePath;

    private Duration operationTimeout = Duration.ofSeconds(15);
    private Duration cancelAfter = Duration.ofSeconds(15);
    private Duration deadlineAfter = Duration.ofSeconds(15);
    private Duration nodeSelfCheckPeriod = Duration.ofSeconds(1);
    private Duration nodeSessionGracePeriod = Duration.ofSeconds(10);
    private NodeConsistencyMode nodeReadConsistencyMode = NodeConsistencyMode.STRICT;
    private NodeConsistencyMode nodeAttachConsistencyMode = NodeConsistencyMode.STRICT;
    private Duration sessionTimeout = Duration.ofSeconds(60);
    private String sessionDescription = "ABCD coordination session";
    private Duration reconnectionPause = Duration.ofSeconds(5);
    private Duration blockTimeout = Duration.ofSeconds(15);
    private Duration sessionStartTimeout = Duration.ofSeconds(15);
    private Duration sessionStopTimeout = Duration.ofSeconds(1);
    private Duration schedulerShutdownTimeout = Duration.ofSeconds(1);
    private Duration pingPeriod = Duration.ofSeconds(5);
    private Duration pingTimeout = Duration.ofSeconds(5);
    private long lostPingsThreshold = 3;
    private Duration sessionDeadlineAfter = Duration.ofDays(365);
    private Duration semaphoreManagementTimeout = Duration.ofSeconds(15);
    private Runnable onSessionStart = () -> { };
    private Runnable onSessionStop = () -> { };
    private Duration acquireSemaphoreTimeout = Duration.ofMinutes(2);
    private Duration acquireEnqueuedSemaphoreTimeout = Duration.ofSeconds(15);
    private Duration releaseSemaphoreTimeout = Duration.ofSeconds(15);
    private Duration describeSemaphoreTimeout = Duration.ofSeconds(15);
    private int executorPoolThreads = 10;

    public CoordinatorBuilderImpl(CoordinationClient client, String nodePath) {
        this.client = client;
        this.nodePath = nodePath;
    }

    @Override
    public Coordinator.Builder operationTimeout(Duration timeout) {
        this.operationTimeout = timeout;
        return this;
    }

    @Override
    public Coordinator.Builder cancelAfter(Duration timeout) {
        this.cancelAfter = timeout;
        return this;
    }

    @Override
    public Coordinator.Builder deadlineAfter(Duration timeout) {
        this.deadlineAfter = timeout;
        return this;
    }

    @Override
    public Coordinator.Builder nodeSelfCheckPeriod(Duration period) {
        this.nodeSelfCheckPeriod = period;
        return this;
    }

    @Override
    public Coordinator.Builder nodeSessionGracePeriod(Duration period) {
        this.nodeSessionGracePeriod = period;
        return this;
    }

    @Override
    public Coordinator.Builder nodeReadConsistencyMode(NodeConsistencyMode mode) {
        this.nodeReadConsistencyMode = mode;
        return this;
    }

    @Override
    public Coordinator.Builder nodeAttachConsistencyMode(NodeConsistencyMode mode) {
        this.nodeAttachConsistencyMode = mode;
        return this;
    }

    @Override
    public Coordinator.Builder sessionTimeout(Duration timeout) {
        this.sessionTimeout = timeout;
        return this;
    }

    @Override
    public Coordinator.Builder sessionDescription(String description) {
        this.sessionDescription = description;
        return this;
    }

    @Override
    public Coordinator.Builder reconnectionPause(Duration duration) {
        this.reconnectionPause = duration;
        return this;
    }

    @Override
    public Coordinator.Builder blockTimeout(Duration timeout) {
        this.blockTimeout = timeout;
        return this;
    }

    @Override
    public Coordinator.Builder sessionStartTimeout(Duration timeout) {
        this.sessionStartTimeout = timeout;
        return this;
    }

    @Override
    public Coordinator.Builder sessionStopTimeout(Duration timeout) {
        this.sessionStopTimeout = timeout;
        return this;
    }

    @Override
    public Coordinator.Builder schedulerShutdownTimeout(Duration timeout) {
        this.schedulerShutdownTimeout = timeout;
        return this;
    }

    @Override
    public Coordinator.Builder pingPeriod(Duration period) {
        this.pingPeriod = period;
        return this;
    }

    @Override
    public Coordinator.Builder pingTimeout(Duration timeout) {
        this.pingTimeout = timeout;
        return this;
    }

    @Override
    public Coordinator.Builder lostPingsThreshold(long count) {
        this.lostPingsThreshold = count;
        return this;
    }

    @Override
    public Coordinator.Builder sessionDeadlineAfter(Duration timeout) {
        this.sessionDeadlineAfter = timeout;
        return this;
    }

    @Override
    public Coordinator.Builder semaphoreManagementTimeout(Duration timeout) {
        this.semaphoreManagementTimeout = timeout;
        return this;
    }

    @Override
    public Coordinator.Builder onSessionStart(Runnable action) {
        this.onSessionStart = action;
        return this;
    }

    @Override
    public Coordinator.Builder onSessionStop(Runnable action) {
        this.onSessionStop = action;
        return this;
    }

    @Override
    public Coordinator.Builder acquireSemaphoreTimeout(Duration timeout) {
        this.acquireSemaphoreTimeout = timeout;
        return this;
    }

    @Override
    public Coordinator.Builder releaseSemaphoreTimeout(Duration timeout) {
        this.releaseSemaphoreTimeout = timeout;
        return this;
    }

    @Override
    public Coordinator.Builder describeSemaphoreTimeout(Duration timeout) {
        this.describeSemaphoreTimeout = timeout;
        return this;
    }

    @Override
    public Coordinator.Builder acquireEnqueuedSemaphoreTimeout(Duration timeout) {
        this.acquireEnqueuedSemaphoreTimeout = timeout;
        return this;
    }

    @Override
    public Coordinator.Builder executorPoolThreads(int count) {
        this.executorPoolThreads = count;
        return this;
    }

    @Override
    public Coordinator build() {
        return new CoordinatorImpl(client, nodePath, operationTimeout, cancelAfter, deadlineAfter,
                nodeSelfCheckPeriod, nodeSessionGracePeriod, nodeReadConsistencyMode, nodeAttachConsistencyMode,
                sessionTimeout, sessionDescription, reconnectionPause, blockTimeout, sessionStartTimeout,
                sessionStopTimeout, schedulerShutdownTimeout, pingPeriod, pingTimeout, lostPingsThreshold,
                sessionDeadlineAfter, semaphoreManagementTimeout, onSessionStart, onSessionStop,
                acquireSemaphoreTimeout, releaseSemaphoreTimeout, describeSemaphoreTimeout,
                acquireEnqueuedSemaphoreTimeout, executorPoolThreads);
    }

    @Override
    public Runnable getOnSessionStart() {
        return onSessionStart;
    }

    @Override
    public Runnable getOnSessionStop() {
        return onSessionStop;
    }

}
