package ru.yandex.intranet.d.datasource.coordination.impl;

import java.time.Duration;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.function.Consumer;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.datasource.coordination.CoordinationClient;
import ru.yandex.intranet.d.datasource.coordination.Coordinator;
import ru.yandex.intranet.d.datasource.coordination.model.NodeConsistencyMode;
import ru.yandex.intranet.d.datasource.coordination.model.session.ChangesSubscription;
import ru.yandex.intranet.d.datasource.coordination.model.session.CoordinationSemaphore;
import ru.yandex.intranet.d.datasource.coordination.model.session.CoordinationSemaphoreDescription;
import ru.yandex.intranet.d.datasource.coordination.model.session.SessionState;

/**
 * YDB coordinator.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public class CoordinatorImpl implements Coordinator {

    private static final Logger LOG = LoggerFactory.getLogger(CoordinatorImpl.class);

    private final CoordinationClient client;
    private final String nodePath;
    private final Duration operationTimeout;
    private final Duration cancelAfter;
    private final Duration deadlineAfter;
    private final Duration nodeSelfCheckPeriod;
    private final Duration nodeSessionGracePeriod;
    private final NodeConsistencyMode nodeReadConsistencyMode;
    private final NodeConsistencyMode nodeAttachConsistencyMode;
    private final Duration sessionTimeout;
    private final String sessionDescription;
    private final Duration reconnectionPause;
    private final Duration blockTimeout;
    private final Duration sessionStartTimeout;
    private final Duration sessionStopTimeout;
    private final Duration schedulerShutdownTimeout;
    private final Duration pingPeriod;
    private final Duration pingTimeout;
    private final long lostPingsThreshold;
    private final Duration sessionDeadlineAfter;
    private final Duration semaphoreManagementTimeout;
    private final Runnable onSessionStart;
    private final Runnable onSessionStop;
    private final Duration acquireSemaphoreTimeout;
    private final Duration releaseSemaphoreTimeout;
    private final Duration describeSemaphoreTimeout;
    private final Duration acquireEnqueuedSemaphoreTimeout;
    private final int executorPoolThreads;
    private final ConcurrentLinkedQueue<Consumer<SessionState>> stateSubscribers = new ConcurrentLinkedQueue<>();

    private volatile boolean started;
    private volatile SessionInitiator sessionInitiator;
    private volatile Thread sessionInitiatorThread;

    @SuppressWarnings("ParameterNumber")
    public CoordinatorImpl(CoordinationClient client, String nodePath, Duration operationTimeout,
                           Duration cancelAfter, Duration deadlineAfter, Duration nodeSelfCheckPeriod,
                           Duration nodeSessionGracePeriod, NodeConsistencyMode nodeReadConsistencyMode,
                           NodeConsistencyMode nodeAttachConsistencyMode, Duration sessionTimeout,
                           String sessionDescription, Duration reconnectionPause, Duration blockTimeout,
                           Duration sessionStartTimeout, Duration sessionStopTimeout,
                           Duration schedulerShutdownTimeout, Duration pingPeriod, Duration pingTimeout,
                           long lostPingsThreshold, Duration sessionDeadlineAfter,
                           Duration semaphoreManagementTimeout, Runnable onSessionStart, Runnable onSessionStop,
                           Duration acquireSemaphoreTimeout, Duration releaseSemaphoreTimeout,
                           Duration describeSemaphoreTimeout, Duration acquireEnqueuedSemaphoreTimeout,
                           int executorPoolThreads) {
        this.client = client;
        this.nodePath = nodePath;
        this.operationTimeout = operationTimeout;
        this.cancelAfter = cancelAfter;
        this.deadlineAfter = deadlineAfter;
        this.nodeSelfCheckPeriod = nodeSelfCheckPeriod;
        this.nodeSessionGracePeriod = nodeSessionGracePeriod;
        this.nodeReadConsistencyMode = nodeReadConsistencyMode;
        this.nodeAttachConsistencyMode = nodeAttachConsistencyMode;
        this.sessionTimeout = sessionTimeout;
        this.sessionDescription = sessionDescription;
        this.reconnectionPause = reconnectionPause;
        this.blockTimeout = blockTimeout;
        this.sessionStartTimeout = sessionStartTimeout;
        this.sessionStopTimeout = sessionStopTimeout;
        this.schedulerShutdownTimeout = schedulerShutdownTimeout;
        this.pingPeriod = pingPeriod;
        this.pingTimeout = pingTimeout;
        this.lostPingsThreshold = lostPingsThreshold;
        this.sessionDeadlineAfter = sessionDeadlineAfter;
        this.semaphoreManagementTimeout = semaphoreManagementTimeout;
        this.onSessionStart = onSessionStart;
        this.onSessionStop = onSessionStop;
        this.acquireSemaphoreTimeout = acquireSemaphoreTimeout;
        this.releaseSemaphoreTimeout = releaseSemaphoreTimeout;
        this.describeSemaphoreTimeout = describeSemaphoreTimeout;
        this.acquireEnqueuedSemaphoreTimeout = acquireEnqueuedSemaphoreTimeout;
        this.executorPoolThreads = executorPoolThreads;
    }

    @Override
    public void start() {
        synchronized (this) {
            LOG.info("Starting YDB coordinator...");
            if (started) {
                LOG.info("YDB coordinator was already started");
                return;
            }
            sessionInitiator = new SessionInitiator(client, nodePath, operationTimeout, cancelAfter, deadlineAfter,
                    nodeSelfCheckPeriod, nodeSessionGracePeriod, nodeReadConsistencyMode, nodeAttachConsistencyMode,
                    sessionTimeout, sessionDescription, reconnectionPause, blockTimeout, sessionStartTimeout,
                    sessionStopTimeout, schedulerShutdownTimeout, pingPeriod, pingTimeout, lostPingsThreshold,
                    sessionDeadlineAfter, semaphoreManagementTimeout, onSessionStart, onSessionStop,
                    acquireSemaphoreTimeout, releaseSemaphoreTimeout, describeSemaphoreTimeout,
                    acquireEnqueuedSemaphoreTimeout, stateSubscribers, executorPoolThreads);
            sessionInitiatorThread = new Thread(sessionInitiator, "YdbCoordinatorSessionInitiator");
            sessionInitiatorThread.setUncaughtExceptionHandler((t, e) -> LOG
                    .error("Uncaught exception in thread " + t, e));
            sessionInitiatorThread.setDaemon(true);
            sessionInitiatorThread.start();
            started = true;
            LOG.info("YDB coordinator started");
        }
    }

    @Override
    public void stop(Runnable onStop) {
        synchronized (this) {
            LOG.info("Stopping YDB coordinator...");
            if (!started) {
                LOG.info("YDB coordinator is already stopped");
                onStop.run();
                return;
            }
            sessionInitiator.stop();
            sessionInitiatorThread.interrupt();
            sessionInitiator.unblockReconnect();
            Thread waitForStopThread = new Thread(() -> {
                LOG.info("Waiting until YDB coordination session initiator thread is stopped...");
                try {
                    sessionInitiatorThread.join();
                } catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                }
                sessionInitiatorThread = null;
                sessionInitiator = null;
                started = false;
                onStop.run();
                LOG.info("YDB coordinator stopped");
            }, "YdbCoordinatorFinalizer");
            waitForStopThread.setDaemon(true);
            waitForStopThread.start();
        }
    }

    @Override
    public Mono<Boolean> acquireSemaphore(String name, Duration timeout, long count, byte[] data, boolean ephemeral) {
        SessionInitiator initiator = this.sessionInitiator;
        if (initiator != null) {
            return initiator.acquireSemaphore(name, timeout, count, data, ephemeral);
        } else {
            return Mono.error(new IllegalStateException("Coordinator is not started"));
        }
    }

    @Override
    public Mono<Boolean> releaseSemaphore(String name) {
        SessionInitiator initiator = this.sessionInitiator;
        if (initiator != null) {
            return initiator.releaseSemaphore(name);
        } else {
            return Mono.error(new IllegalStateException("Coordinator is not started"));
        }
    }

    @Override
    public Mono<CoordinationSemaphore> createSemaphore(String name, long limit, byte[] data) {
        SessionInitiator initiator = this.sessionInitiator;
        if (initiator != null) {
            return initiator.createSemaphore(name, limit, data);
        } else {
            return Mono.error(new IllegalStateException("Coordinator is not started"));
        }
    }

    @Override
    public Mono<Void> updateSemaphore(String name, byte[] data) {
        SessionInitiator initiator = this.sessionInitiator;
        if (initiator != null) {
            return initiator.updateSemaphore(name, data);
        } else {
            return Mono.error(new IllegalStateException("Coordinator is not started"));
        }
    }

    @Override
    public Mono<Void> deleteSemaphore(String name, boolean force) {
        SessionInitiator initiator = this.sessionInitiator;
        if (initiator != null) {
            return initiator.deleteSemaphore(name, force);
        } else {
            return Mono.error(new IllegalStateException("Coordinator is not started"));
        }
    }

    @Override
    public Mono<CoordinationSemaphoreDescription> describeSemaphore(String name, boolean includeOwners,
                                                                    boolean includeWaiters) {
        SessionInitiator initiator = this.sessionInitiator;
        if (initiator != null) {
            return initiator.describeSemaphore(name, includeOwners, includeWaiters);
        } else {
            return Mono.error(new IllegalStateException("Coordinator is not started"));
        }
    }

    @Override
    public Mono<ChangesSubscription> subscribeToChanges(String name, boolean includeOwners, boolean includeWaiters,
                                                        boolean watchData, boolean watchOwners) {
        SessionInitiator initiator = this.sessionInitiator;
        if (initiator != null) {
            if (!watchData && !watchOwners) {
                return Mono.error(new IllegalArgumentException("At least one subscription is required"));
            }
            return initiator.subscribeToChanges(name, includeOwners, includeWaiters, watchData, watchOwners);
        } else {
            return Mono.error(new IllegalStateException("Coordinator is not started"));
        }
    }

    @Override
    public void subscribeToSessionState(Consumer<SessionState> consumer) {
        stateSubscribers.add(consumer);
    }

    @Override
    public SessionState getSessionState() {
        SessionInitiator currentSessionInitiator = sessionInitiator;
        if (currentSessionInitiator != null) {
            return currentSessionInitiator.getSessionState();
        }
        return SessionState.INVALID;
    }

    @Override
    public void reconnect() {
        sessionInitiator.unblockReconnect();
    }

}
