package ru.yandex.intranet.d.datasource.coordination.impl;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.time.Duration;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectReader;
import com.fasterxml.jackson.databind.ObjectWriter;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.yandex.ydb.core.StatusCode;
import com.yandex.ydb.core.UnexpectedResultException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.core.Exceptions;
import reactor.core.publisher.Mono;
import reactor.core.scheduler.Schedulers;

import ru.yandex.intranet.d.datasource.coordination.Coordinator;
import ru.yandex.intranet.d.datasource.coordination.model.cluster.ClusterLeader;
import ru.yandex.intranet.d.datasource.coordination.model.cluster.NodeInfo;
import ru.yandex.intranet.d.datasource.coordination.model.cluster.NodeLeadershipStatus;
import ru.yandex.intranet.d.datasource.coordination.model.session.ChangesSubscription;
import ru.yandex.intranet.d.datasource.coordination.model.session.CoordinationSemaphoreDescription;
import ru.yandex.intranet.d.datasource.coordination.model.session.SemaphoreEvent;
import ru.yandex.intranet.d.datasource.coordination.model.session.SessionState;
import ru.yandex.intranet.d.datasource.coordination.model.session.SubscriptionCancelledException;
import ru.yandex.intranet.d.util.Barrier;

/**
 * Leader election logic.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public class LeaderElection implements Runnable {

    private static final Logger LOG = LoggerFactory.getLogger(LeaderElection.class);

    private final Coordinator coordinator;
    private final Barrier sessionReadyBarrier;
    private final Barrier refreshBarrier;
    private final LeadershipPublisher leadershipPublisher;
    private final LeaderPublisher leaderPublisher;
    private final String nodeId;
    private final String semaphoreName;
    private final Duration semaphoreAcquisitionWaitDuration;
    private final Duration describeSemaphoreBlockTimeout;
    private final Duration createSemaphoreBlockTimeout;
    private final Duration acquireSemaphoreBlockTimeout;
    private final Duration releaseSemaphoreBlockTimeout;
    private final Duration executorShutdownTimeout;
    private final ObjectReader nodeInfoReader;
    private final ObjectWriter nodeInfoWriter;
    private final NodeInfo nodeInfo;
    private final ExecutorService executorService;

    private volatile boolean stopped = false;

    @SuppressWarnings("ParameterNumber")
    public LeaderElection(Coordinator coordinator, Barrier sessionReadyBarrier, Barrier refreshBarrier,
                          LeadershipPublisher leadershipPublisher, LeaderPublisher leaderPublisher,
                          String nodeId, String semaphoreName, Duration semaphoreAcquisitionWaitDuration,
                          Duration describeSemaphoreBlockTimeout, Duration createSemaphoreBlockTimeout,
                          Duration acquireSemaphoreBlockTimeout, Duration releaseSemaphoreBlockTimeout,
                          Duration executorShutdownTimeout, ObjectReader nodeInfoReader,
                          ObjectWriter nodeInfoWriter, NodeInfo nodeInfo, int describeQueueCapacity) {
        this.coordinator = coordinator;
        this.sessionReadyBarrier = sessionReadyBarrier;
        this.refreshBarrier = refreshBarrier;
        this.leadershipPublisher = leadershipPublisher;
        this.leaderPublisher = leaderPublisher;
        this.nodeId = nodeId;
        this.semaphoreName = semaphoreName;
        this.semaphoreAcquisitionWaitDuration = semaphoreAcquisitionWaitDuration;
        this.describeSemaphoreBlockTimeout = describeSemaphoreBlockTimeout;
        this.createSemaphoreBlockTimeout = createSemaphoreBlockTimeout;
        this.acquireSemaphoreBlockTimeout = acquireSemaphoreBlockTimeout;
        this.releaseSemaphoreBlockTimeout = releaseSemaphoreBlockTimeout;
        this.executorShutdownTimeout = executorShutdownTimeout;
        this.nodeInfoReader = nodeInfoReader;
        this.nodeInfoWriter = nodeInfoWriter;
        this.nodeInfo = nodeInfo;
        this.executorService = createExecutorService(describeQueueCapacity);
    }

    @Override
    public void run() {
        LOG.info("Running cluster manager leader election thread...");
        while (!stopped) {
            try {
                refreshLoop();
            } catch (Throwable e) {
                LOG.error("Failed to perform cluster manager leader election", e);
                Exceptions.throwIfJvmFatal(e);
                freeResources();
                if (stopped) {
                    LOG.info("Finishing cluster manager leader election thread...");
                    break;
                }
            }
        }
        leadershipPublisher.complete();
        leaderPublisher.complete();
        shutdownExecutor();
        LOG.info("Cluster manager leader election thread finished");
    }

    public void stop() {
        this.stopped = true;
    }

    public Mono<Boolean> isLeader() {
        LOG.info("Checking if this node is leader...");
        return getLeader().map(i -> i.map(n -> nodeId.equals(n.getUuid())).orElse(false));
    }

    public Optional<Boolean> isLeaderCached() {
        LOG.info("Checking if this node is leader using cache...");
        if (coordinator.getSessionState() == SessionState.INVALID) {
            LOG.info("Coordination session is not ready");
            leadershipPublisher.nextStatus(NodeLeadershipStatus.UNDEFINED);
            leaderPublisher.nextLeader(new ClusterLeader(null, false));
            refreshBarrier.open();
            return Optional.empty();
        }
        NodeLeadershipStatus status = leadershipPublisher.getLeadershipStatus();
        switch (status) {
            case LEADER:
                LOG.info("This node is cached as leader");
                return Optional.of(true);
            case FOLLOWER:
                LOG.info("This node is cached as follower");
                return Optional.of(false);
            case UNDEFINED:
                LOG.info("This node is cached as undefined");
                return Optional.empty();
            default:
                throw new IllegalStateException("Unexpected leadership status: " + status);
        }
    }

    public Mono<Optional<NodeInfo>> getLeader() {
        LOG.info("Getting current leader...");
        if (coordinator.getSessionState() == SessionState.INVALID) {
            LOG.info("Coordination session is not ready");
            leadershipPublisher.nextStatus(NodeLeadershipStatus.UNDEFINED);
            leaderPublisher.nextLeader(new ClusterLeader(null, false));
            refreshBarrier.open();
            return Mono.error(new IllegalStateException("YDB coordination session is not available"));
        }
        LOG.info("Scheduling semaphore {} description request...", semaphoreName);
        Mono<Optional<NodeInfo>> result = Mono.fromFuture(() -> CompletableFuture.supplyAsync(() -> {
            LOG.info("Preparing semaphore {} description request", semaphoreName);
            if (coordinator.getSessionState() == SessionState.INVALID) {
                LOG.info("Coordination session is not ready");
                leadershipPublisher.nextStatus(NodeLeadershipStatus.UNDEFINED);
                leaderPublisher.nextLeader(new ClusterLeader(null, false));
                refreshBarrier.open();
                throw new IllegalStateException("YDB coordination session is not available");
            }
            LOG.info("Waiting for semaphore {} description response...", semaphoreName);
            ChangesSubscription subscription = coordinator
                    .subscribeToChanges(semaphoreName, true, false, false, true)
                    .block(describeSemaphoreBlockTimeout);
            LOG.info("Received semaphore {} description response", semaphoreName);
            if (subscription != null) {
                subscribeToSemaphore(subscription.getSubscription());
            }
            Optional<NodeInfo> leader = Optional.ofNullable(subscription).flatMap(s -> s.getDescription()
                    .getOwners().stream().findFirst().map(v -> readNodeInfo(v.getData())));
            NodeLeadershipStatus leadershipStatus = leader.filter(n -> nodeId.equals(n.getUuid()))
                    .map(n -> NodeLeadershipStatus.LEADER).orElse(NodeLeadershipStatus.FOLLOWER);
            if (coordinator.getSessionState() == SessionState.INVALID) {
                LOG.info("Coordination session is not ready");
                leadershipPublisher.nextStatus(NodeLeadershipStatus.UNDEFINED);
                leaderPublisher.nextLeader(new ClusterLeader(null, false));
                refreshBarrier.open();
                throw new IllegalStateException("YDB coordination session is not available");
            }
            leaderPublisher.nextLeader(new ClusterLeader(leader.orElse(null), true));
            leadershipPublisher.nextStatus(leadershipStatus);
            if (leadershipStatus != NodeLeadershipStatus.LEADER) {
                LOG.info("Not a leader, opening semaphore acquisition sequence");
                refreshBarrier.open();
            }
            if (coordinator.getSessionState() == SessionState.INVALID) {
                LOG.info("Coordination session is not ready");
                leadershipPublisher.nextStatus(NodeLeadershipStatus.UNDEFINED);
                leaderPublisher.nextLeader(new ClusterLeader(null, false));
                refreshBarrier.open();
                throw new IllegalStateException("YDB coordination session is not available");
            }
            LOG.info("Current leader is: {}", leader);
            return leader;
        }, executorService));
        LOG.info("Scheduled semaphore {} description request", semaphoreName);
        return result;
    }

    public Optional<NodeInfo> getLeaderCached() {
        LOG.info("Getting current leader using cache...");
        if (coordinator.getSessionState() == SessionState.INVALID) {
            LOG.info("Coordination session is not ready");
            leadershipPublisher.nextStatus(NodeLeadershipStatus.UNDEFINED);
            leaderPublisher.nextLeader(new ClusterLeader(null, false));
            refreshBarrier.open();
            return Optional.empty();
        }
        Optional<NodeInfo> result = leaderPublisher.getClusterLeader().getLeader();
        LOG.info("Cached leader is: {}", result);
        return result;
    }

    @SuppressWarnings("MethodLength")
    private void refreshLoop() {
        while (!stopped) {
            refreshBarrier.close();
            if (stopped) {
                freeResources();
                LOG.info("Finishing cluster manager leader election thread...");
                return;
            }
            LOG.info("Waiting until coordination session is ready...");
            try {
                sessionReadyBarrier.passThrough();
            } catch (InterruptedException e) {
                LOG.info("Waking up after shutdown was requested (through interruption)...");
                freeResources();
                Thread.currentThread().interrupt();
                LOG.info("Finishing cluster manager leader election thread...");
                return;
            }
            if (stopped) {
                freeResources();
                LOG.info("Finishing cluster manager leader election thread...");
                return;
            }
            if (coordinator.getSessionState() == SessionState.INVALID) {
                leadershipPublisher.nextStatus(NodeLeadershipStatus.UNDEFINED);
                leaderPublisher.nextLeader(new ClusterLeader(null, false));
                LOG.info("Coordination session is not ready");
                continue;
            }
            LOG.info("Coordination session is ready");
            CoordinationSemaphoreDescription semaphoreDescription;
            try {
                semaphoreDescription = checkSemaphore().get();
            } catch (Exception e) {
                if (e instanceof InterruptedException || e.getCause() instanceof InterruptedException
                        || (e instanceof ExecutionException && e.getCause() != null
                        && e.getCause().getCause() instanceof InterruptedException)) {
                    freeResources();
                    Thread.currentThread().interrupt();
                    LOG.info("Finishing cluster manager leader election thread...");
                    return;
                }
                LOG.info("Failed to check YDB semaphore " + semaphoreName, e);
                freeResources();
                if (stopped) {
                    LOG.info("Finishing cluster manager leader election thread...");
                    return;
                }
                continue;
            }
            if (stopped) {
                freeResources();
                LOG.info("Finishing cluster manager leader election thread...");
                return;
            }
            if (coordinator.getSessionState() == SessionState.INVALID) {
                leadershipPublisher.nextStatus(NodeLeadershipStatus.UNDEFINED);
                leaderPublisher.nextLeader(new ClusterLeader(null, false));
                LOG.info("Coordination session is not ready");
                continue;
            }
            try {
                semaphoreDescription = createSemaphoreIfNotExists(semaphoreDescription).get();
            } catch (Exception e) {
                if (e instanceof InterruptedException || e.getCause() instanceof InterruptedException
                        || (e instanceof ExecutionException && e.getCause() != null
                        && e.getCause().getCause() instanceof InterruptedException)) {
                    freeResources();
                    Thread.currentThread().interrupt();
                    LOG.info("Finishing cluster manager leader election thread...");
                    return;
                }
                LOG.info("Failed to create and check YDB semaphore " + semaphoreName, e);
                freeResources();
                if (stopped) {
                    LOG.info("Finishing cluster manager leader election thread...");
                    return;
                }
                continue;
            }
            if (stopped) {
                freeResources();
                LOG.info("Finishing cluster manager leader election thread...");
                return;
            }
            if (coordinator.getSessionState() == SessionState.INVALID) {
                leadershipPublisher.nextStatus(NodeLeadershipStatus.UNDEFINED);
                leaderPublisher.nextLeader(new ClusterLeader(null, false));
                LOG.info("Coordination session is not ready");
                continue;
            }
            boolean acquired;
            try {
                acquired = acquireSemaphore();
            } catch (Exception e) {
                if (e.getCause() instanceof InterruptedException) {
                    freeResources();
                    Thread.currentThread().interrupt();
                    LOG.info("Finishing cluster manager leader election thread...");
                    return;
                }
                LOG.info("Failed to check YDB semaphore " + semaphoreName, e);
                freeResources();
                if (stopped) {
                    LOG.info("Finishing cluster manager leader election thread...");
                    return;
                }
                continue;
            }
            if (stopped) {
                freeResources();
                LOG.info("Finishing cluster manager leader election thread...");
                return;
            }
            if (coordinator.getSessionState() == SessionState.INVALID) {
                leadershipPublisher.nextStatus(NodeLeadershipStatus.UNDEFINED);
                leaderPublisher.nextLeader(new ClusterLeader(null, false));
                LOG.info("Coordination session is not ready");
                continue;
            }
            if (acquired) {
                refreshBarrier.close();
                LOG.info("Semaphore {} is acquired, may wait until the next change", semaphoreName);
                try {
                    semaphoreDescription = checkSemaphore().get();
                } catch (Exception e) {
                    if (e instanceof InterruptedException || e.getCause() instanceof InterruptedException
                            || (e instanceof ExecutionException && e.getCause() != null
                            && e.getCause().getCause() instanceof InterruptedException)) {
                        freeResources();
                        Thread.currentThread().interrupt();
                        LOG.info("Finishing cluster manager leader election thread...");
                        return;
                    }
                    LOG.info("Failed to check YDB semaphore " + semaphoreName, e);
                    freeResources();
                    if (stopped) {
                        LOG.info("Finishing cluster manager leader election thread...");
                        return;
                    }
                    continue;
                }
            } else {
                LOG.info("Semaphore {} is not acquired, need to try again", semaphoreName);
                continue;
            }
            if (stopped) {
                freeResources();
                LOG.info("Finishing cluster manager leader election thread...");
                return;
            }
            if (coordinator.getSessionState() == SessionState.INVALID) {
                leadershipPublisher.nextStatus(NodeLeadershipStatus.UNDEFINED);
                leaderPublisher.nextLeader(new ClusterLeader(null, false));
                LOG.info("Coordination session is not ready");
                continue;
            }
            LOG.info("Sleeping until refresh or shutdown...");
            try {
                refreshBarrier.passThrough();
            } catch (InterruptedException e) {
                LOG.info("Waking up after shutdown was requested (through interruption)...");
                freeResources();
                Thread.currentThread().interrupt();
                LOG.info("Finishing cluster manager leader election thread...");
                return;
            }
            if (stopped) {
                LOG.info("Waking up after shutdown was requested (through unlock)...");
                freeResources();
                LOG.info("Finishing cluster manager leader election thread...");
                return;
            } else {
                LOG.info("Waking up after refresh was requested...");
            }
        }
    }

    private void freeResources() {
        if (leadershipPublisher.getLeadershipStatus() == NodeLeadershipStatus.LEADER) {
            LOG.info("Releasing semaphore {}...", semaphoreName);
            try {
                Boolean released = coordinator
                        .releaseSemaphore(semaphoreName).block(releaseSemaphoreBlockTimeout);
                LOG.info("Released semaphore {}: {}", semaphoreName, released);
            } catch (Exception e) {
                if (e.getCause() instanceof InterruptedException) {
                    LOG.info("Semaphore {} release was interrupted", semaphoreName);
                    try {
                        Boolean released = coordinator
                                .releaseSemaphore(semaphoreName).block(releaseSemaphoreBlockTimeout);
                        LOG.info("Released semaphore {}: {}", semaphoreName, released);
                    } catch (Exception ex) {
                        if (ex.getCause() instanceof InterruptedException) {
                            LOG.warn("Semaphore {} release was interrupted again", semaphoreName);
                            Thread.currentThread().interrupt();
                        } else {
                            LOG.error("Failed to release semaphore " + semaphoreName, e);
                        }
                    }
                } else {
                    LOG.error("Failed to release semaphore " + semaphoreName, e);
                }
            }
        }
        if (coordinator.getSessionState() == SessionState.INVALID || stopped) {
            leadershipPublisher.nextStatus(NodeLeadershipStatus.UNDEFINED);
            leaderPublisher.nextLeader(new ClusterLeader(null, false));
        }
    }

    private boolean acquireSemaphore() {
        LOG.info("Acquiring semaphore {}...", semaphoreName);
        Boolean acquired = coordinator.acquireSemaphore(semaphoreName, semaphoreAcquisitionWaitDuration,
                1L, writeNodeInfo(nodeInfo), false)
                .block(acquireSemaphoreBlockTimeout);
        LOG.info("Acquisition status for {} is: {}", semaphoreName, acquired);
        if (acquired == null) {
            return false;
        }
        return acquired;
    }

    private CompletableFuture<CoordinationSemaphoreDescription> checkSemaphore() {
        LOG.info("Scheduling YDB semaphore {} check...", semaphoreName);
        CompletableFuture<CoordinationSemaphoreDescription> result = CompletableFuture.supplyAsync(() -> {
            try {
                LOG.info("Checking YDB semaphore {}...", semaphoreName);
                ChangesSubscription subscription = coordinator
                        .subscribeToChanges(semaphoreName, true, false, false, true)
                        .block(describeSemaphoreBlockTimeout);
                LOG.info("YDB semaphore {} exists", semaphoreName);
                if (subscription != null) {
                    subscribeToSemaphore(subscription.getSubscription());
                } else {
                    throw new IllegalStateException("Missing YDB semaphore subscription result");
                }
                Optional<NodeInfo> leader = subscription.getDescription()
                        .getOwners().stream().findFirst().map(v -> readNodeInfo(v.getData()));
                NodeLeadershipStatus leadershipStatus = leader.filter(n -> nodeId.equals(n.getUuid()))
                        .map(n -> NodeLeadershipStatus.LEADER).orElse(NodeLeadershipStatus.FOLLOWER);
                if (coordinator.getSessionState() == SessionState.INVALID) {
                    leadershipPublisher.nextStatus(NodeLeadershipStatus.UNDEFINED);
                    leaderPublisher.nextLeader(new ClusterLeader(null, false));
                    throw new IllegalStateException("Coordination session is not available");
                }
                leaderPublisher.nextLeader(new ClusterLeader(leader.orElse(null), true));
                leadershipPublisher.nextStatus(leadershipStatus);
                if (leadershipStatus != NodeLeadershipStatus.LEADER) {
                    LOG.info("Not a leader, opening semaphore acquisition sequence");
                    refreshBarrier.open();
                }
                if (coordinator.getSessionState() == SessionState.INVALID) {
                    leadershipPublisher.nextStatus(NodeLeadershipStatus.UNDEFINED);
                    leaderPublisher.nextLeader(new ClusterLeader(null, false));
                    throw new IllegalStateException("Coordination session is not available");
                }
                return subscription.getDescription();
            } catch (UnexpectedResultException e) {
                if (e.getStatusCode() == StatusCode.NOT_FOUND) {
                    LOG.info("YDB semaphore {} not yet exists", semaphoreName);
                    return null;
                }
                throw e;
            }
        }, executorService);
        LOG.info("Scheduled YDB semaphore {} check", semaphoreName);
        return result;
    }

    private CompletableFuture<CoordinationSemaphoreDescription> createSemaphoreIfNotExists(
            CoordinationSemaphoreDescription description) {
        if (description != null) {
            return CompletableFuture.completedFuture(description);
        }
        LOG.info("Creating YDB semaphore {}...", semaphoreName);
        coordinator.createSemaphore(semaphoreName, 1L, new byte[0]).block(createSemaphoreBlockTimeout);
        LOG.info("Created YDB semaphore {}", semaphoreName);
        LOG.info("Scheduling created YDB semaphore {} check...", semaphoreName);
        CompletableFuture<CoordinationSemaphoreDescription> result = CompletableFuture.supplyAsync(() -> {
            LOG.info("Checking created YDB semaphore {}...", semaphoreName);
            ChangesSubscription newSubscription = coordinator
                    .subscribeToChanges(semaphoreName, true, false, false, true)
                    .block(describeSemaphoreBlockTimeout);
            LOG.info("Checked created YDB semaphore {}", semaphoreName);
            if (newSubscription != null) {
                subscribeToSemaphore(newSubscription.getSubscription());
            } else {
                throw new IllegalStateException("Missing YDB semaphore subscription result");
            }
            Optional<NodeInfo> leader = newSubscription.getDescription()
                    .getOwners().stream().findFirst().map(v -> readNodeInfo(v.getData()));
            NodeLeadershipStatus leadershipStatus = leader.filter(n -> nodeId.equals(n.getUuid()))
                    .map(n -> NodeLeadershipStatus.LEADER).orElse(NodeLeadershipStatus.FOLLOWER);
            if (coordinator.getSessionState() == SessionState.INVALID) {
                leadershipPublisher.nextStatus(NodeLeadershipStatus.UNDEFINED);
                leaderPublisher.nextLeader(new ClusterLeader(null, false));
                throw new IllegalStateException("Coordination session is not available");
            }
            leaderPublisher.nextLeader(new ClusterLeader(leader.orElse(null), true));
            leadershipPublisher.nextStatus(leadershipStatus);
            if (leadershipStatus != NodeLeadershipStatus.LEADER) {
                LOG.info("Not a leader, opening semaphore acquisition sequence");
                refreshBarrier.open();
            }
            if (coordinator.getSessionState() == SessionState.INVALID) {
                leadershipPublisher.nextStatus(NodeLeadershipStatus.UNDEFINED);
                leaderPublisher.nextLeader(new ClusterLeader(null, false));
                throw new IllegalStateException("Coordination session is not available");
            }
            return newSubscription.getDescription();
        }, executorService);
        LOG.info("Scheduled created YDB semaphore {} check", semaphoreName);
        return result;
    }

    private void shutdownExecutor() {
        LOG.info("Shutting down cluster manager leader election thread pool...");
        executorService.shutdown();
        try {
            executorService.awaitTermination(executorShutdownTimeout.toMillis(), TimeUnit.MILLISECONDS);
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        executorService.shutdownNow();
        LOG.info("Finished shutting down cluster manager leader election thread pool");
    }

    private NodeInfo readNodeInfo(byte[] bytes) {
        try {
            return nodeInfoReader.readValue(bytes);
        } catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    private byte[] writeNodeInfo(NodeInfo data) {
        try {
            return nodeInfoWriter.writeValueAsBytes(data);
        } catch (JsonProcessingException e) {
            throw new UncheckedIOException(e);
        }
    }

    private void subscribeToSemaphore(Mono<SemaphoreEvent> publisher) {
        LOG.info("Subscribing to semaphore {} events...", semaphoreName);
        publisher.subscribeOn(Schedulers.boundedElastic()).subscribe(event -> {
            LOG.info("Semaphore {} event received: {}", semaphoreName, event);
            onSubscription();
        }, e -> {
            if (e instanceof SubscriptionCancelledException) {
                LOG.info("Subscription to semaphore {} events is suppressed by newer subscription", semaphoreName);
                return;
            }
            if (stopped) {
                LOG.info("Leader election is stopped, semaphore event subscription failure is ignored");
                refreshBarrier.open();
                return;
            }
            LOG.info("Semaphore " + semaphoreName + " events subscription failure", e);
            onSubscription();
        });
        LOG.info("Subscribed to semaphore {} events", semaphoreName);
    }

    @SuppressWarnings("FutureReturnValueIgnored")
    private void onSubscription() {
        if (coordinator.getSessionState() == SessionState.INVALID) {
            LOG.info("Coordination session is not available, semaphore {} event is ignored", semaphoreName);
            leadershipPublisher.nextStatus(NodeLeadershipStatus.UNDEFINED);
            leaderPublisher.nextLeader(new ClusterLeader(null, false));
            refreshBarrier.open();
            return;
        }
        if (stopped) {
            LOG.info("Leader election is stopped, semaphore event is ignored");
            refreshBarrier.open();
            return;
        }
        LOG.info("Scheduling semaphore {} description update...", semaphoreName);
        try {
            CompletableFuture.runAsync(() -> {
                if (coordinator.getSessionState() == SessionState.INVALID) {
                    LOG.info("Coordination session is not available, semaphore {} event is ignored",
                            semaphoreName);
                    leadershipPublisher.nextStatus(NodeLeadershipStatus.UNDEFINED);
                    leaderPublisher.nextLeader(new ClusterLeader(null, false));
                    refreshBarrier.open();
                    return;
                }
                LOG.info("Getting updated semaphore {} description due to new event...", semaphoreName);
                ChangesSubscription subscription;
                try {
                    subscription = coordinator
                            .subscribeToChanges(semaphoreName, true, false, false, true)
                            .block(describeSemaphoreBlockTimeout);
                    LOG.info("Got updated semaphore {} description due to new event", semaphoreName);
                } catch (Exception e) {
                    if (e.getCause() instanceof InterruptedException) {
                        LOG.info("Cluster manager leader election is interrupted...");
                        refreshBarrier.open();
                        return;
                    }
                    if (stopped) {
                        LOG.info("Failed to get updated semaphore " + semaphoreName
                                + " due to new event, leader election is stopped");
                    } else if (coordinator.getSessionState() != SessionState.INVALID) {
                        LOG.info("Failed to get updated semaphore " + semaphoreName
                                + " due to new event, session is not available");
                        refreshBarrier.open();
                    } else {
                        LOG.info("Failed to get updated semaphore " + semaphoreName
                                + " due to new event, retrying", e);
                        onSubscription();
                    }
                    return;
                }
                if (subscription != null) {
                    subscribeToSemaphore(subscription.getSubscription());
                }
                Optional<NodeInfo> leader = Optional.ofNullable(subscription).flatMap(s -> s.getDescription()
                        .getOwners().stream().findFirst().map(v -> readNodeInfo(v.getData())));
                NodeLeadershipStatus leadershipStatus = leader.filter(n -> nodeId.equals(n.getUuid()))
                        .map(n -> NodeLeadershipStatus.LEADER).orElse(NodeLeadershipStatus.FOLLOWER);
                if (coordinator.getSessionState() == SessionState.INVALID) {
                    LOG.info("Coordination session is not available, new leadership {} state is ignored",
                            semaphoreName);
                    leadershipPublisher.nextStatus(NodeLeadershipStatus.UNDEFINED);
                    leaderPublisher.nextLeader(new ClusterLeader(null, false));
                    refreshBarrier.open();
                    return;
                }
                if (stopped) {
                    LOG.info("Leader election is stopped, semaphore event update is ignored");
                    refreshBarrier.open();
                    return;
                }
                leaderPublisher.nextLeader(new ClusterLeader(leader.orElse(null), true));
                leadershipPublisher.nextStatus(leadershipStatus);
                if (leadershipStatus != NodeLeadershipStatus.LEADER) {
                    LOG.info("Not a leader, opening semaphore acquisition sequence");
                    refreshBarrier.open();
                }
                if (coordinator.getSessionState() == SessionState.INVALID) {
                    LOG.info("Coordination session is not available, new leadership {} state is overwritten",
                            semaphoreName);
                    leadershipPublisher.nextStatus(NodeLeadershipStatus.UNDEFINED);
                    leaderPublisher.nextLeader(new ClusterLeader(null, false));
                    refreshBarrier.open();
                }
            }, executorService);
            LOG.info("Scheduled semaphore {} description update", semaphoreName);
        } catch (Exception e) {
            LOG.info("Failed to schedule semaphore " + semaphoreName + " description update", e);
            refreshBarrier.open();
        }
    }

    private static ExecutorService createExecutorService(int capacity) {
        ThreadFactory threadFactory = new ThreadFactoryBuilder()
                .setDaemon(true)
                .setNameFormat("cluster-manager-leader-election-pool-%d")
                .setUncaughtExceptionHandler((t, e) -> LOG
                        .error("Uncaught exception in cluster manager leader election pool thread " + t, e))
                .build();
        return new ThreadPoolExecutor(1, 1,
                0L, TimeUnit.MILLISECONDS,
                new LinkedBlockingQueue<>(capacity),
                threadFactory);
    }

}
