package ru.yandex.intranet.d.datasource.coordination.impl;

import java.time.Duration;
import java.time.Instant;
import java.util.HashSet;
import java.util.Random;
import java.util.Set;
import java.util.concurrent.CancellationException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;

import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.yandex.ydb.core.Status;
import com.yandex.ydb.core.StatusCode;
import com.yandex.ydb.core.UnexpectedResultException;
import com.yandex.ydb.core.rpc.OutStreamObserver;
import com.yandex.ydb.core.rpc.StreamObserver;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.core.Exceptions;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.datasource.coordination.CoordinationClient;
import ru.yandex.intranet.d.datasource.coordination.model.CreateCoordinationNodeRequest;
import ru.yandex.intranet.d.datasource.coordination.model.DescribeCoordinationNodeRequest;
import ru.yandex.intranet.d.datasource.coordination.model.NodeConsistencyMode;
import ru.yandex.intranet.d.datasource.coordination.model.OperationParameters;
import ru.yandex.intranet.d.datasource.coordination.model.TargetNodeConfig;
import ru.yandex.intranet.d.datasource.coordination.model.session.AcquireSemaphorePendingResponse;
import ru.yandex.intranet.d.datasource.coordination.model.session.AcquireSemaphoreRequest;
import ru.yandex.intranet.d.datasource.coordination.model.session.AcquireSemaphoreResultResponse;
import ru.yandex.intranet.d.datasource.coordination.model.session.ChangesSubscription;
import ru.yandex.intranet.d.datasource.coordination.model.session.CoordinationSemaphore;
import ru.yandex.intranet.d.datasource.coordination.model.session.CoordinationSemaphoreDescription;
import ru.yandex.intranet.d.datasource.coordination.model.session.CoordinationSessionRequest;
import ru.yandex.intranet.d.datasource.coordination.model.session.CoordinationSessionResponse;
import ru.yandex.intranet.d.datasource.coordination.model.session.CreateSemaphoreRequest;
import ru.yandex.intranet.d.datasource.coordination.model.session.CreateSemaphoreResultResponse;
import ru.yandex.intranet.d.datasource.coordination.model.session.DeleteSemaphoreRequest;
import ru.yandex.intranet.d.datasource.coordination.model.session.DeleteSemaphoreResultResponse;
import ru.yandex.intranet.d.datasource.coordination.model.session.DescribeSemaphoreChangedResponse;
import ru.yandex.intranet.d.datasource.coordination.model.session.DescribeSemaphoreRequest;
import ru.yandex.intranet.d.datasource.coordination.model.session.DescribeSemaphoreResultResponse;
import ru.yandex.intranet.d.datasource.coordination.model.session.FailureResponse;
import ru.yandex.intranet.d.datasource.coordination.model.session.PingPongRequest;
import ru.yandex.intranet.d.datasource.coordination.model.session.PingPongResponse;
import ru.yandex.intranet.d.datasource.coordination.model.session.ReleaseSemaphoreRequest;
import ru.yandex.intranet.d.datasource.coordination.model.session.ReleaseSemaphoreResultResponse;
import ru.yandex.intranet.d.datasource.coordination.model.session.SemaphoreEvent;
import ru.yandex.intranet.d.datasource.coordination.model.session.SessionOperationParameters;
import ru.yandex.intranet.d.datasource.coordination.model.session.SessionStartRequest;
import ru.yandex.intranet.d.datasource.coordination.model.session.SessionStartedResponse;
import ru.yandex.intranet.d.datasource.coordination.model.session.SessionState;
import ru.yandex.intranet.d.datasource.coordination.model.session.SessionStopRequest;
import ru.yandex.intranet.d.datasource.coordination.model.session.SessionStoppedResponse;
import ru.yandex.intranet.d.datasource.coordination.model.session.SubscriptionCancelledException;
import ru.yandex.intranet.d.datasource.coordination.model.session.UpdateSemaphoreRequest;
import ru.yandex.intranet.d.datasource.coordination.model.session.UpdateSemaphoreResultResponse;
import ru.yandex.intranet.d.util.Barrier;

/**
 * YDB coordination session initiator.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public class SessionInitiator implements Runnable {

    private static final Logger LOG = LoggerFactory.getLogger(SessionInitiator.class);

    private final CoordinationClient client;
    private final String nodePath;
    private final Duration operationTimeout;
    private final Duration cancelAfter;
    private final Duration deadlineAfter;
    private final Duration nodeSelfCheckPeriod;
    private final Duration nodeSessionGracePeriod;
    private final NodeConsistencyMode nodeReadConsistencyMode;
    private final NodeConsistencyMode nodeAttachConsistencyMode;
    private final Duration sessionTimeout;
    private final String sessionDescription;
    private final Duration reconnectionPause;
    private final Duration blockTimeout;
    private final Duration sessionStartTimeout;
    private final Duration sessionStopTimeout;
    private final Duration schedulerShutdownTimeout;
    private final Duration pingPeriod;
    private final Duration pingTimeout;
    private final long lostPingsThreshold;
    private final Duration sessionDeadlineAfter;
    private final Duration semaphoreManagementTimeout;
    private final Runnable onSessionStart;
    private final Runnable onSessionStop;
    private final Duration acquireSemaphoreTimeout;
    private final Duration releaseSemaphoreTimeout;
    private final Duration describeSemaphoreTimeout;
    private final Duration acquireEnqueuedSemaphoreTimeout;
    private final AtomicReference<SessionState> sessionState = new AtomicReference<>(SessionState.INVALID);
    private final ConcurrentLinkedQueue<Consumer<SessionState>> stateSubscribers;
    private final Barrier reconnectBarrier = new Barrier();
    private final AtomicReference<ConnectionHolder> currentConnection = new AtomicReference<>();
    private final byte[] sessionKey = generateSessionKey();
    private final AtomicLong sequenceNumber = new AtomicLong(0L);
    private final AtomicLong requestNumber = new AtomicLong(0L);
    private final AtomicLong pingNumber = new AtomicLong(0L);
    private final AtomicLong connectionNumber = new AtomicLong(0L);
    private final AtomicLong sessionIdRetriesCounter = new AtomicLong(0L);
    private final AtomicReference<Instant> lastSessionBreakAt = new AtomicReference<>();
    private final ScheduledExecutorService scheduler = prepareScheduler();
    private final ExecutorService sessionExecutorService;
    private final AtomicLong lostPings = new AtomicLong(0L);
    private final ConcurrentHashMap<Long, Long> pendingPings = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<Long, ScheduledFuture<?>> pendingPingTimeouts
            = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<Long, CompletableFuture<Void>> pendingCreateSemaphore = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<Long, ScheduledFuture<?>> pendingCreateSemaphoreTimeouts
            = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<Long, CompletableFuture<Void>> pendingDeleteSemaphore = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<Long, ScheduledFuture<?>> pendingDeleteSemaphoreTimeouts
            = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<Long, CompletableFuture<Void>> pendingUpdateSemaphore = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<Long, ScheduledFuture<?>> pendingUpdateSemaphoreTimeouts
            = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<Long, CompletableFuture<Boolean>> pendingAcquireSemaphore
            = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<Long, ScheduledFuture<?>> pendingAcquireSemaphoreTimeouts
            = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<Long, ScheduledFuture<?>> pendingAcquireEnqueuedSemaphoreTimeouts
            = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<Long, CompletableFuture<Boolean>> pendingReleaseSemaphore
            = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<Long, ScheduledFuture<?>> pendingReleaseSemaphoreTimeouts
            = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<Long, PendingDescription> pendingDescribeSemaphore
            = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<Long, ScheduledFuture<?>> pendingDescribeSemaphoreTimeouts
            = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<Long, PendingDescription> pendingSubscribeSemaphore
            = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<Long, ScheduledFuture<?>> pendingSubscribeSemaphoreTimeouts
            = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<Long, Subscription> eventSubscriptions
            = new ConcurrentHashMap<>();

    private volatile boolean stopped;
    private volatile long lastSessionId = 0L;
    private volatile boolean sessionStopPending = false;

    @SuppressWarnings("ParameterNumber")
    public SessionInitiator(CoordinationClient client, String nodePath, Duration operationTimeout,
                            Duration cancelAfter, Duration deadlineAfter, Duration nodeSelfCheckPeriod,
                            Duration nodeSessionGracePeriod, NodeConsistencyMode nodeReadConsistencyMode,
                            NodeConsistencyMode nodeAttachConsistencyMode, Duration sessionTimeout,
                            String sessionDescription, Duration reconnectionPause, Duration blockTimeout,
                            Duration sessionStartTimeout, Duration sessionStopTimeout,
                            Duration schedulerShutdownTimeout, Duration pingPeriod, Duration pingTimeout,
                            long lostPingsThreshold, Duration sessionDeadlineAfter,
                            Duration semaphoreManagementTimeout, Runnable onSessionStart, Runnable onSessionStop,
                            Duration acquireSemaphoreTimeout, Duration releaseSemaphoreTimeout,
                            Duration describeSemaphoreTimeout, Duration acquireEnqueuedSemaphoreTimeout,
                            ConcurrentLinkedQueue<Consumer<SessionState>> stateSubscribers, int executorPoolThreads) {
        this.client = client;
        this.nodePath = nodePath;
        this.operationTimeout = operationTimeout;
        this.cancelAfter = cancelAfter;
        this.deadlineAfter = deadlineAfter;
        this.nodeSelfCheckPeriod = nodeSelfCheckPeriod;
        this.nodeSessionGracePeriod = nodeSessionGracePeriod;
        this.nodeReadConsistencyMode = nodeReadConsistencyMode;
        this.nodeAttachConsistencyMode = nodeAttachConsistencyMode;
        this.sessionTimeout = sessionTimeout;
        this.sessionDescription = sessionDescription;
        this.reconnectionPause = reconnectionPause;
        this.blockTimeout = blockTimeout;
        this.sessionStartTimeout = sessionStartTimeout;
        this.sessionStopTimeout = sessionStopTimeout;
        this.schedulerShutdownTimeout = schedulerShutdownTimeout;
        this.pingPeriod = pingPeriod;
        this.pingTimeout = pingTimeout;
        this.lostPingsThreshold = lostPingsThreshold;
        this.sessionDeadlineAfter = sessionDeadlineAfter;
        this.semaphoreManagementTimeout = semaphoreManagementTimeout;
        this.onSessionStart = onSessionStart;
        this.onSessionStop = onSessionStop;
        this.acquireSemaphoreTimeout = acquireSemaphoreTimeout;
        this.releaseSemaphoreTimeout = releaseSemaphoreTimeout;
        this.describeSemaphoreTimeout = describeSemaphoreTimeout;
        this.acquireEnqueuedSemaphoreTimeout = acquireEnqueuedSemaphoreTimeout;
        this.stateSubscribers = stateSubscribers;
        this.sessionExecutorService = prepareSessionExecutor(executorPoolThreads);
    }

    @Override
    public void run() {
        LOG.info("Running YDB coordination session initiator thread...");
        while (!stopped) {
            try {
                reconnectionLoop();
            } catch (Throwable e) {
                LOG.error("Failed to establish YDB coordination session", e);
                Exceptions.throwIfJvmFatal(e);
                freeResources();
                if (!waitForNextReconnectionAttempt()) {
                    break;
                }
            }
        }
        shutdownScheduler();
        shutdownExecutor();
        LOG.info("YDB coordination session initiator thread finished");
    }

    public void stop() {
        this.stopped = true;
    }

    public void unblockReconnect() {
        reconnectBarrier.open();
    }

    public SessionState getSessionState() {
        return sessionState.get();
    }

    public Mono<CoordinationSemaphore> createSemaphore(String name, long limit, byte[] data) {
        try {
            ConnectionHolder connection = currentConnection.get();
            SessionState state = sessionState.get();
            if (connection != null && state == SessionState.VALID) {
                long requestId = requestNumber.incrementAndGet();
                LOG.info("Creating semaphore {} with request id {}:{}...", name, connection.getId(), requestId);
                CompletableFuture<Void> responseFuture = new CompletableFuture<>();
                ScheduledFuture<?> timeoutFuture = scheduleCreateSemaphoreTimeout(requestId);
                pendingCreateSemaphore.put(requestId, responseFuture);
                pendingCreateSemaphoreTimeouts.put(requestId, timeoutFuture);
                connection.getRequestObserver().onNext(CoordinationSessionRequest.createSemaphore(
                        new CreateSemaphoreRequest(requestId, name, limit, data)));
                return Mono.fromFuture(responseFuture).thenReturn(new CoordinationSemaphore(name));
            } else {
                return Mono.error(new IllegalStateException("No active YDB coordination session"));
            }
        } catch (Throwable e) {
            Exceptions.throwIfJvmFatal(e);
            return Mono.error(e);
        }
    }

    public Mono<Void> deleteSemaphore(String name, boolean force) {
        try {
            ConnectionHolder connection = currentConnection.get();
            SessionState state = sessionState.get();
            if (connection != null && state == SessionState.VALID) {
                long requestId = requestNumber.incrementAndGet();
                LOG.info("Deleting semaphore {} with request id {}:{}...", name, connection.getId(), requestId);
                CompletableFuture<Void> responseFuture = new CompletableFuture<>();
                ScheduledFuture<?> timeoutFuture = scheduleDeleteSemaphoreTimeout(requestId);
                pendingDeleteSemaphore.put(requestId, responseFuture);
                pendingDeleteSemaphoreTimeouts.put(requestId, timeoutFuture);
                connection.getRequestObserver().onNext(CoordinationSessionRequest.deleteSemaphore(
                        new DeleteSemaphoreRequest(requestId, name, force)));
                return Mono.fromFuture(responseFuture);
            } else {
                return Mono.error(new IllegalStateException("No active YDB coordination session"));
            }
        } catch (Throwable e) {
            Exceptions.throwIfJvmFatal(e);
            return Mono.error(e);
        }
    }

    public Mono<Void> updateSemaphore(String name, byte[] data) {
        try {
            ConnectionHolder connection = currentConnection.get();
            SessionState state = sessionState.get();
            if (connection != null && state == SessionState.VALID) {
                long requestId = requestNumber.incrementAndGet();
                LOG.info("Updating semaphore {} with request id {}:{}...", name, connection.getId(), requestId);
                CompletableFuture<Void> responseFuture = new CompletableFuture<>();
                ScheduledFuture<?> timeoutFuture = scheduleUpdateSemaphoreTimeout(requestId);
                pendingUpdateSemaphore.put(requestId, responseFuture);
                pendingUpdateSemaphoreTimeouts.put(requestId, timeoutFuture);
                connection.getRequestObserver().onNext(CoordinationSessionRequest.updateSemaphore(
                        new UpdateSemaphoreRequest(requestId, name, data)));
                return Mono.fromFuture(responseFuture);
            } else {
                return Mono.error(new IllegalStateException("No active YDB coordination session"));
            }
        } catch (Throwable e) {
            Exceptions.throwIfJvmFatal(e);
            return Mono.error(e);
        }
    }

    public Mono<Boolean> acquireSemaphore(String name, Duration timeout, long count, byte[] data, boolean ephemeral) {
        try {
            ConnectionHolder connection = currentConnection.get();
            SessionState state = sessionState.get();
            if (connection != null && state == SessionState.VALID) {
                long requestId = requestNumber.incrementAndGet();
                LOG.info("Acquiring semaphore {} with request id {}:{}...", name, connection.getId(), requestId);
                CompletableFuture<Boolean> responseFuture = new CompletableFuture<>();
                ScheduledFuture<?> timeoutFuture = scheduleAcquireSemaphoreTimeout(requestId);
                ScheduledFuture<?> timeoutEnqueuedFuture = scheduleAcquireSemaphoreEnqueuedTimeout(requestId);
                pendingAcquireSemaphore.put(requestId, responseFuture);
                pendingAcquireSemaphoreTimeouts.put(requestId, timeoutFuture);
                pendingAcquireEnqueuedSemaphoreTimeouts.put(requestId, timeoutEnqueuedFuture);
                connection.getRequestObserver().onNext(CoordinationSessionRequest.acquireSemaphore(
                        new AcquireSemaphoreRequest(requestId, name, timeout, count, data, ephemeral)));
                return Mono.fromFuture(responseFuture);
            } else {
                return Mono.error(new IllegalStateException("No active YDB coordination session"));
            }
        } catch (Throwable e) {
            Exceptions.throwIfJvmFatal(e);
            return Mono.error(e);
        }
    }

    public Mono<Boolean> releaseSemaphore(String name) {
        try {
            ConnectionHolder connection = currentConnection.get();
            SessionState state = sessionState.get();
            if (connection != null && state == SessionState.VALID) {
                long requestId = requestNumber.incrementAndGet();
                LOG.info("Releasing semaphore {} with request id {}:{}...", name, connection.getId(), requestId);
                CompletableFuture<Boolean> responseFuture = new CompletableFuture<>();
                ScheduledFuture<?> timeoutFuture = scheduleReleaseSemaphoreTimeout(requestId);
                pendingReleaseSemaphore.put(requestId, responseFuture);
                pendingReleaseSemaphoreTimeouts.put(requestId, timeoutFuture);
                connection.getRequestObserver().onNext(CoordinationSessionRequest.releaseSemaphore(
                        new ReleaseSemaphoreRequest(requestId, name)));
                return Mono.fromFuture(responseFuture);
            } else {
                return Mono.error(new IllegalStateException("No active YDB coordination session"));
            }
        } catch (Throwable e) {
            Exceptions.throwIfJvmFatal(e);
            return Mono.error(e);
        }
    }

    public Mono<CoordinationSemaphoreDescription> describeSemaphore(String name, boolean includeOwners,
                                                                    boolean includeWaiters) {
        try {
            ConnectionHolder connection = currentConnection.get();
            SessionState state = sessionState.get();
            if (connection != null && state == SessionState.VALID) {
                long requestId = requestNumber.incrementAndGet();
                LOG.info("Describing semaphore {} with request id {}:{}...", name, connection.getId(), requestId);
                CompletableFuture<Description> responseFuture = new CompletableFuture<>();
                ScheduledFuture<?> timeoutFuture = scheduleDescribeSemaphoreTimeout(requestId);
                pendingDescribeSemaphore.put(requestId, new PendingDescription(responseFuture, name));
                pendingDescribeSemaphoreTimeouts.put(requestId, timeoutFuture);
                connection.getRequestObserver().onNext(CoordinationSessionRequest.describeSemaphore(
                        new DescribeSemaphoreRequest(requestId, name, includeOwners, includeWaiters,
                                false, false)));
                return Mono.fromFuture(responseFuture).map(Description::getDescription);
            } else {
                return Mono.error(new IllegalStateException("No active YDB coordination session"));
            }
        } catch (Throwable e) {
            Exceptions.throwIfJvmFatal(e);
            return Mono.error(e);
        }
    }

    public Mono<ChangesSubscription> subscribeToChanges(String name, boolean includeOwners, boolean includeWaiters,
                                                        boolean watchData, boolean watchOwners) {
        try {
            ConnectionHolder connection = currentConnection.get();
            SessionState state = sessionState.get();
            if (connection != null && state == SessionState.VALID) {
                long requestId = requestNumber.incrementAndGet();
                LOG.info("Subscribing to semaphore {} with request id {}:{}...", name, connection.getId(), requestId);
                CompletableFuture<Description> responseFuture = new CompletableFuture<>();
                ScheduledFuture<?> timeoutFuture = scheduleSubscribeSemaphoreTimeout(requestId);
                pendingSubscribeSemaphore.put(requestId, new PendingDescription(responseFuture, name));
                pendingSubscribeSemaphoreTimeouts.put(requestId, timeoutFuture);
                CompletableFuture<SemaphoreEvent> eventsSubscription = new CompletableFuture<>();
                eventSubscriptions.put(requestId, new Subscription(eventsSubscription, name));
                connection.getRequestObserver().onNext(CoordinationSessionRequest.describeSemaphore(
                        new DescribeSemaphoreRequest(requestId, name, includeOwners, includeWaiters,
                                watchData, watchOwners)));
                return Mono.fromFuture(responseFuture).doOnError(e -> {
                    eventSubscriptions.remove(requestId);
                    eventsSubscription.completeExceptionally(e);
                }).flatMap(d -> {
                    if (d.isWatchAdded()) {
                        return Mono.just(new ChangesSubscription(Mono.fromFuture(eventsSubscription),
                                d.getDescription(), () -> {
                            eventSubscriptions.remove(requestId);
                            eventsSubscription.completeExceptionally(
                                    new CancellationException("Subscription was cancelled"));
                        }));
                    } else {
                        eventSubscriptions.remove(requestId);
                        eventsSubscription.completeExceptionally(new IllegalStateException("Watch was not added"));
                        return Mono.error(new IllegalStateException("Watch was not added"));
                    }
                });
            } else {
                return Mono.error(new IllegalStateException("No active YDB coordination session"));
            }
        } catch (Throwable e) {
            Exceptions.throwIfJvmFatal(e);
            return Mono.error(e);
        }
    }

    private void reconnectionLoop() {
        while (!stopped) {
            reconnectBarrier.close();
            if (stopped) {
                freeResources();
                LOG.info("Finishing YDB coordination session initiator thread...");
                return;
            }
            boolean nodeExists;
            try {
                nodeExists = checkNodeExists();
            } catch (Exception e) {
                if (e.getCause() instanceof InterruptedException) {
                    freeResources();
                    Thread.currentThread().interrupt();
                    LOG.info("Finishing YDB coordination session initiator thread...");
                    return;
                }
                LOG.info("Failed to check if YDB coordination node " + nodePath + " exists", e);
                freeResources();
                if (!waitForNextReconnectionAttempt()) {
                    return;
                }
                continue;
            }
            if (stopped) {
                freeResources();
                LOG.info("Finishing YDB coordination session initiator thread...");
                return;
            }
            try {
                createNodeIfNotExists(nodeExists);
            } catch (Exception e) {
                if (e.getCause() instanceof InterruptedException) {
                    freeResources();
                    Thread.currentThread().interrupt();
                    LOG.info("Finishing YDB coordination session initiator thread...");
                    return;
                }
                LOG.info("Failed to create YDB coordination node " + nodePath, e);
                freeResources();
                if (!waitForNextReconnectionAttempt()) {
                    return;
                }
                continue;
            }
            if (stopped) {
                freeResources();
                LOG.info("Finishing YDB coordination session initiator thread...");
                return;
            }
            try {
                createConnection();
            } catch (Exception e) {
                if (e.getCause() instanceof InterruptedException) {
                    freeResources();
                    Thread.currentThread().interrupt();
                    LOG.info("Finishing YDB coordination session initiator thread...");
                    return;
                }
                LOG.error("Failed to establish YDB coordination connection", e);
                freeResources();
                if (!waitForNextReconnectionAttempt()) {
                    return;
                }
                continue;
            }
            if (stopped) {
                freeResources();
                LOG.info("Finishing YDB coordination session initiator thread...");
                return;
            }
            try {
                startSession();
            } catch (InterruptedException e) {
                freeResources();
                Thread.currentThread().interrupt();
                LOG.info("Finishing YDB coordination session initiator thread...");
                return;
            } catch (Exception e) {
                if (e.getCause() instanceof InterruptedException) {
                    freeResources();
                    Thread.currentThread().interrupt();
                    LOG.info("Finishing YDB coordination session initiator thread...");
                    return;
                }
                LOG.error("Failed to establish YDB coordination session", e);
                freeResources();
                if (!waitForNextReconnectionAttempt()) {
                    return;
                }
                continue;
            }
            if (stopped) {
                freeResources();
                LOG.info("Finishing YDB coordination session initiator thread...");
                return;
            }
            LOG.info("Sleeping until reinitialization or shutdown...");
            try {
                reconnectBarrier.passThrough();
            } catch (InterruptedException e) {
                LOG.info("Waking up after shutdown was requested (through interruption)...");
                freeResources();
                Thread.currentThread().interrupt();
                LOG.info("Finishing YDB coordination session initiator thread...");
                return;
            }
            if (stopped) {
                LOG.info("Waking up after shutdown was requested (through unlock)...");
                freeResources();
                LOG.info("Finishing YDB coordination session initiator thread...");
                return;
            } else {
                LOG.info("Waking up after reconnection was requested...");
                freeResources();
                if (!waitForNextReconnectionAttempt()) {
                    return;
                }
            }
        }
    }

    private boolean waitForNextReconnectionAttempt() {
        if (stopped) {
            LOG.info("Finishing YDB coordination session initiator thread...");
            return false;
        }
        LOG.info("Sleeping until next YDB coordination session initiation attempt...");
        try {
            Thread.sleep(reconnectionPause.toMillis());
            LOG.info("Waking up for next YDB coordination session initiation attempt");
        } catch (InterruptedException ie) {
            LOG.info("Waking up after YDB coordination session initiator shutdown was requested...");
            Thread.currentThread().interrupt();
            if (stopped) {
                LOG.info("Finishing YDB coordination session initiator thread...");
                return false;
            }
        }
        if (stopped) {
            LOG.info("Finishing YDB coordination session initiator thread...");
            return false;
        }
        return true;
    }

    private void freeResources() {
        ConnectionHolder connection = currentConnection.get();
        if (connection != null) {
            stopSessionQuietly(connection);
            LOG.info("Completing YDB coordination connection...");
            connection.getRequestObserver().onCompleted();
            LOG.info("YDB coordination connection resources freed");
        } else {
            LOG.info("No active YDB coordination connection, nothing to free");
        }
        LOG.info("Invalidating session state...");
        SessionState oldState = sessionState.getAndSet(SessionState.INVALID);
        if (oldState != SessionState.INVALID) {
            onSessionStop.run();
            deliverSubscriptions(SessionState.INVALID);
        }
        currentConnection.set(null);
        LOG.info("Freeing session ping resources...");
        freePingResources();
        LOG.info("Freeing session requests resources...");
        freePendingRequests();
        LOG.info("Freeing session subscriptions resources...");
        freeSubscriptions();
        LOG.info("YDB coordination resources freed");
    }

    private boolean checkNodeExists() {
        try {
            LOG.info("Checking if YDB coordination node {} exists...", nodePath);
            client.describeNode(DescribeCoordinationNodeRequest.builder(nodePath)
                    .operationParameters(OperationParameters.builder()
                            .operationTimeout(operationTimeout)
                            .cancelAfter(cancelAfter)
                            .deadlineAfter(deadlineAfter)
                            .build())
                    .builder())
                    .block(blockTimeout);
            LOG.info("YDB coordination node {} already exists", nodePath);
            return true;
        } catch (UnexpectedResultException e) {
            if (e.getStatusCode() == StatusCode.SCHEME_ERROR) {
                LOG.info("YDB coordination node {} not yet exists", nodePath);
                return false;
            }
            throw e;
        }
    }

    private void createNodeIfNotExists(boolean nodeExists) {
        if (nodeExists) {
            LOG.info("Using already existing node {}", nodePath);
            return;
        }
        LOG.info("Creating YDB coordination node {}...", nodePath);
        client.createNode(CreateCoordinationNodeRequest.builder(nodePath)
                .configuration(TargetNodeConfig.builder()
                        .sessionGracePeriod(nodeSessionGracePeriod)
                        .selfCheckPeriod(nodeSelfCheckPeriod)
                        .attachConsistencyMode(nodeAttachConsistencyMode)
                        .readConsistencyMode(nodeReadConsistencyMode)
                        .build())
                .operationParameters(OperationParameters.builder()
                        .operationTimeout(operationTimeout)
                        .cancelAfter(cancelAfter)
                        .deadlineAfter(deadlineAfter)
                        .build())
                .build())
                .block(blockTimeout);
        LOG.info("YDB coordination node {} created", nodePath);
    }

    private void createConnection() {
        LOG.info("Creating YDB coordination connection...");
        SessionOperationParameters params = SessionOperationParameters.builder()
                .deadlineAfter(sessionDeadlineAfter)
                .build();
        CompletableFuture<SessionStartedResponse> sessionStartFuture = new CompletableFuture<>();
        CompletableFuture<SessionStoppedResponse> sessionStopFuture = new CompletableFuture<>();
        long connectionId = connectionNumber.incrementAndGet();
        SessionObserver sessionObserver = new SessionObserver(sessionStartFuture, sessionStopFuture,
                connectionId, this);
        OutStreamObserver<CoordinationSessionRequest> requestObserver = client.session(params, sessionObserver);
        sessionObserver.setRequestObserver(requestObserver);
        ConnectionHolder requestsSource = new ConnectionHolder(requestObserver, sessionStartFuture,
                sessionStopFuture, connectionId);
        currentConnection.set(requestsSource);
        LOG.info("YDB coordination connection created");
    }

    private void startSession() throws InterruptedException, ExecutionException, TimeoutException {
        LOG.info("Starting YDB coordination session...");
        ConnectionHolder connection = currentConnection.get();
        OutStreamObserver<CoordinationSessionRequest> requestObserver = connection.getRequestObserver();
        long sessionId = 0L;
        Instant previousSessionBreakAt = lastSessionBreakAt.get();
        Instant now = Instant.now();
        if (lastSessionId == 0L) {
            LOG.info("Starting with new YDB coordination session id");
        } else if (sessionIdRetriesCounter.incrementAndGet() <= 1L
                && previousSessionBreakAt != null && previousSessionBreakAt.isBefore(now)
                && Duration.between(previousSessionBreakAt, now).compareTo(nodeSessionGracePeriod) < 0) {
            sessionId = lastSessionId;
            LOG.info("Trying to reuse last YDB coordination session id: {}", sessionId);
        } else {
            sessionIdRetriesCounter.set(0L);
            lastSessionId = 0L;
            LOG.info("Falling back to new YDB coordination session id");
        }
        requestObserver.onNext(CoordinationSessionRequest.sessionStart(new SessionStartRequest(nodePath,
                sessionId, sessionTimeout, sessionDescription, sequenceNumber.incrementAndGet(), sessionKey)));
        LOG.info("YDB coordination session start message is scheduled, waiting for session start...");
        connection.sessionStartFuture.get(sessionStartTimeout.toMillis(), TimeUnit.MILLISECONDS);
        LOG.info("YDB coordination session was started");
    }

    private void stopSessionQuietly(ConnectionHolder connection) {
        try {
            if (sessionStopPending) {
                LOG.info("Sending YDB coordination session stop request...");
                connection.getRequestObserver().onNext(CoordinationSessionRequest
                        .sessionStop(new SessionStopRequest()));
                connection.getSessionStopFuture().get(sessionStopTimeout.toMillis(), TimeUnit.MILLISECONDS);
                LOG.info("YDB coordination session stopped");
            } else {
                LOG.info("YDB coordination session is not started yet, nothing to stop");
            }
        } catch (InterruptedException e) {
            LOG.info("YDB coordination session stop was interrupted");
            try {
                connection.getSessionStopFuture().get(sessionStopTimeout.toMillis(), TimeUnit.MILLISECONDS);
                LOG.info("YDB coordination session stopped");
            } catch (InterruptedException ex) {
                Thread.currentThread().interrupt();
                LOG.warn("YDB coordination session stop was interrupted again");
            } catch (TimeoutException ex) {
                LOG.warn("YDB coordination session was not stopped until timeout.");
            } catch (Exception ex) {
                LOG.error("Failed to stop YDB coordination session", e);
            }
        } catch (TimeoutException e) {
            LOG.warn("YDB coordination session was not stopped until timeout.");
        } catch (Exception e) {
            LOG.error("Failed to stop YDB coordination session", e);
        }
    }

    private void freePingResources() {
        lostPings.set(0L);
        pendingPings.clear();
        Set<Long> timeoutKeysToRemove = new HashSet<>();
        pendingPingTimeouts.forEach((k, v) -> {
            timeoutKeysToRemove.add(k);
            v.cancel(true);
        });
        timeoutKeysToRemove.forEach(pendingPingTimeouts::remove);
    }

    private void freePendingRequests() {
        freePendingRequestsType(pendingCreateSemaphore, pendingCreateSemaphoreTimeouts);
        freePendingRequestsType(pendingDeleteSemaphore, pendingDeleteSemaphoreTimeouts);
        freePendingRequestsType(pendingUpdateSemaphore, pendingUpdateSemaphoreTimeouts);
        freePendingRequestsType(pendingAcquireSemaphore, pendingAcquireSemaphoreTimeouts);
        Set<Long> acquireEnqueuedSemaphoreTimeoutKeysToRemove = new HashSet<>();
        pendingAcquireEnqueuedSemaphoreTimeouts.forEach((k, v) -> {
            acquireEnqueuedSemaphoreTimeoutKeysToRemove.add(k);
            v.cancel(true);
        });
        acquireEnqueuedSemaphoreTimeoutKeysToRemove.forEach(pendingAcquireEnqueuedSemaphoreTimeouts::remove);
        freePendingRequestsType(pendingReleaseSemaphore, pendingReleaseSemaphoreTimeouts);
        freePendingDescription(pendingDescribeSemaphore, pendingDescribeSemaphoreTimeouts);
        freePendingDescription(pendingSubscribeSemaphore, pendingSubscribeSemaphoreTimeouts);
    }

    private <V> void freePendingRequestsType(ConcurrentHashMap<Long, CompletableFuture<V>> pendingFutures,
                                             ConcurrentHashMap<Long, ScheduledFuture<?>> pendingScheduledTimeouts) {
        Set<Long> keysToRemove = new HashSet<>();
        pendingFutures.forEach((k, v) -> {
            keysToRemove.add(k);
            v.completeExceptionally(new CancellationException("YDB coordination session invalidated"));
        });
        keysToRemove.forEach(pendingFutures::remove);
        Set<Long> timeoutKeysToRemove = new HashSet<>();
        pendingScheduledTimeouts.forEach((k, v) -> {
            timeoutKeysToRemove.add(k);
            v.cancel(true);
        });
        timeoutKeysToRemove.forEach(pendingScheduledTimeouts::remove);
    }

    private void freePendingDescription(ConcurrentHashMap<Long, PendingDescription> pendingFutures,
                                        ConcurrentHashMap<Long, ScheduledFuture<?>> pendingScheduledTimeouts) {
        Set<Long> keysToRemove = new HashSet<>();
        pendingFutures.forEach((k, v) -> {
            keysToRemove.add(k);
            v.getFuture().completeExceptionally(new CancellationException("YDB coordination session invalidated"));
        });
        keysToRemove.forEach(pendingFutures::remove);
        Set<Long> timeoutKeysToRemove = new HashSet<>();
        pendingScheduledTimeouts.forEach((k, v) -> {
            timeoutKeysToRemove.add(k);
            v.cancel(true);
        });
        timeoutKeysToRemove.forEach(pendingScheduledTimeouts::remove);
    }

    private void freeSubscriptions() {
        Set<Long> keysToRemove = new HashSet<>();
        eventSubscriptions.forEach((k, v) -> {
            keysToRemove.add(k);
            v.getSubscription().completeExceptionally(
                    new CancellationException("YDB coordination session is not available"));
        });
        keysToRemove.forEach(eventSubscriptions::remove);
    }

    private void handleConnectionResponse(OutStreamObserver<CoordinationSessionRequest> requestObserver,
                                          CompletableFuture<SessionStartedResponse> sessionStartFuture,
                                          CompletableFuture<SessionStoppedResponse> sessionStopFuture,
                                          long connectionId,
                                          CoordinationSessionResponse response) {
        response.match(new CoordinationSessionResponse.Cases<Void>() {
            @Override
            public Void ping(PingPongResponse response) {
                return onPing(response, requestObserver, connectionId);
            }

            @Override
            public Void pong(PingPongResponse response) {
                return onPong(response, connectionId);
            }

            @Override
            public Void failure(FailureResponse response) {
                return onSessionFailure(response, sessionStartFuture, sessionStopFuture, connectionId);
            }

            @Override
            public Void sessionStarted(SessionStartedResponse response) {
                return onSessionStarted(response, sessionStartFuture, connectionId);
            }

            @Override
            public Void sessionStopped(SessionStoppedResponse response) {
                return onSessionStopped(response, sessionStartFuture, sessionStopFuture, connectionId);
            }

            @Override
            public Void acquireSemaphorePending(AcquireSemaphorePendingResponse response) {
                return onAcquireSemaphorePending(response, connectionId);
            }

            @Override
            public Void acquireSemaphoreResult(AcquireSemaphoreResultResponse response) {
                return onAcquireSemaphore(response, connectionId);
            }

            @Override
            public Void releaseSemaphoreResult(ReleaseSemaphoreResultResponse response) {
                return onReleaseSemaphore(response, connectionId);
            }

            @Override
            public Void describeSemaphoreResult(DescribeSemaphoreResultResponse response) {
                return onDescribeSemaphore(response, connectionId);
            }

            @Override
            public Void describeSemaphoreChanged(DescribeSemaphoreChangedResponse response) {
                return onDescribeSemaphoreChanged(response, connectionId);
            }

            @Override
            public Void createSemaphoreResult(CreateSemaphoreResultResponse response) {
                return onCreateSemaphore(response, connectionId);
            }

            @Override
            public Void updateSemaphoreResult(UpdateSemaphoreResultResponse response) {
                return onUpdateSemaphore(response, connectionId);
            }

            @Override
            public Void deleteSemaphoreResult(DeleteSemaphoreResultResponse response) {
                return onDeleteSemaphore(response, connectionId);
            }
        });
    }

    private boolean isCurrentConnection(long connectionId) {
        ConnectionHolder connection = currentConnection.get();
        if (connection != null) {
            boolean isCurrent = connection.getId() == connectionId;
            if (!isCurrent) {
                LOG.info("Received event is for {} connection but current connection is {}",
                        connectionId, connection.getId());
            }
            return isCurrent;
        }
        return false;
    }

    private Void onPing(PingPongResponse response, OutStreamObserver<CoordinationSessionRequest> requestObserver,
                        long connectionId) {
        LOG.info("Received ping from YDB coordination service with id = {}:{}, sending pong...",
                connectionId, response.getOpaque());
        if (!isCurrentConnection(connectionId)) {
            LOG.info("Received response is not from the currently active connection, ignoring");
            return null;
        }
        if (requestObserver != null) {
            requestObserver.onNext(CoordinationSessionRequest.pong(new PingPongRequest(response.getOpaque())));
        } else {
            LOG.warn("No request sink to send pong");
        }
        return null;
    }

    private Void onPong(PingPongResponse response, long connectionId) {
        LOG.info("Received pong from YDB coordination service {}:{}", connectionId, response.getOpaque());
        if (!isCurrentConnection(connectionId)) {
            LOG.info("Received response is not from the currently active connection, ignoring");
            return null;
        }
        if (pendingPings.containsKey(response.getOpaque())) {
            pendingPings.remove(response.getOpaque());
            lostPings.set(0L);
        } else {
            LOG.warn("Ping for pong {}:{} has already timed out", connectionId, response.getOpaque());
        }
        ScheduledFuture<?> scheduledPingTimeout = pendingPingTimeouts.get(response.getOpaque());
        if (scheduledPingTimeout != null) {
            LOG.info("Cancelling ping timeout for {}:{}", connectionId, response.getOpaque());
            scheduledPingTimeout.cancel(true);
            pendingPingTimeouts.remove(response.getOpaque());
        } else {
            LOG.info("No ping timeout to cancel for {}:{}", connectionId, response.getOpaque());
        }
        return null;
    }

    private Void onSessionFailure(FailureResponse response,
                                  CompletableFuture<SessionStartedResponse> sessionStartFuture,
                                  CompletableFuture<SessionStoppedResponse> sessionStopFuture,
                                  long connectionId) {
        LOG.error("Received failure from YDB coordination service: {}", response.getStatus());
        if (!isCurrentConnection(connectionId)) {
            LOG.info("Received response is not from the currently active connection, ignoring");
            return null;
        }
        boolean startWasCompleted = sessionStartFuture.completeExceptionally(
                new UnexpectedResultException("YDB coordination connection error received",
                        response.getStatus().getCode(), response.getStatus().getIssues()));
        if (startWasCompleted) {
            sessionStopFuture.completeExceptionally(new CancellationException("Session was not yet started"));
        }
        SessionState oldState = sessionState.getAndSet(SessionState.INVALID);
        if (oldState != SessionState.INVALID) {
            onSessionStop.run();
            deliverSubscriptions(SessionState.INVALID);
        }
        reconnectBarrier.open();
        return null;
    }

    private Void onSessionStarted(SessionStartedResponse response,
                                  CompletableFuture<SessionStartedResponse> sessionStartFuture, long connectionId) {
        LOG.info("YDB coordination session start received: {}", response);
        if (!isCurrentConnection(connectionId)) {
            LOG.info("Received response is not from the currently active connection, ignoring");
            return null;
        }
        lastSessionId = response.getSessionId();
        sessionStopPending = true;
        lastSessionBreakAt.set(null);
        sessionStartFuture.complete(response);
        SessionState oldState = sessionState.getAndSet(SessionState.VALID);
        if (oldState != SessionState.VALID) {
            onSessionStart.run();
            deliverSubscriptions(SessionState.VALID);
        }
        schedulePing(connectionId);
        return null;
    }

    private Void onSessionStopped(SessionStoppedResponse response,
                                  CompletableFuture<SessionStartedResponse> sessionStartFuture,
                                  CompletableFuture<SessionStoppedResponse> sessionStopFuture,
                                  long connectionId) {
        LOG.info("YDB coordination session stop received: {}", response);
        if (!isCurrentConnection(connectionId)) {
            LOG.info("Received response is not from the currently active connection, ignoring");
            return null;
        }
        long currentSessionId = lastSessionId;
        if (response.getSessionId() != currentSessionId) {
            LOG.warn("Stopped session {} is not a current session {}, message is ignored",
                    response.getSessionId(), currentSessionId);
            return null;
        }
        lastSessionId = 0L;
        sessionStopPending = false;
        reconnectBarrier.open();
        sessionStartFuture
                .completeExceptionally(new CancellationException(
                        "YDB coordination session stopped while not yet started"));
        sessionStopFuture.complete(response);
        SessionState oldState = sessionState.getAndSet(SessionState.INVALID);
        if (oldState != SessionState.INVALID) {
            onSessionStop.run();
            deliverSubscriptions(SessionState.INVALID);
        }
        return null;
    }

    private Void onAcquireSemaphorePending(AcquireSemaphorePendingResponse response, long connectionId) {
        LOG.info("Semaphore acquire is pending on server: {}", response);
        if (!isCurrentConnection(connectionId)) {
            LOG.info("Received response is not from the currently active connection, ignoring");
            return null;
        }
        ScheduledFuture<?> pendingEnqueuedTimeout = pendingAcquireEnqueuedSemaphoreTimeouts.get(response.getReqId());
        pendingAcquireEnqueuedSemaphoreTimeouts.remove(response.getReqId());
        if (pendingEnqueuedTimeout != null) {
            pendingEnqueuedTimeout.cancel(true);
        }
        if (!pendingAcquireSemaphore.containsKey(response.getReqId())) {
            LOG.info("Semaphore acquire {}:{} is already not pending on client",
                    connectionId, response.getReqId());
        }
        return null;
    }

    private Void onAcquireSemaphore(AcquireSemaphoreResultResponse response, long connectionId) {
        LOG.info("Semaphore acquire result received: {}", response);
        if (!isCurrentConnection(connectionId)) {
            LOG.info("Received response is not from the currently active connection, ignoring");
            return null;
        }
        CompletableFuture<Boolean> pendingFuture = pendingAcquireSemaphore.get(response.getReqId());
        ScheduledFuture<?> pendingEnqueuedTimeout = pendingAcquireEnqueuedSemaphoreTimeouts.get(response.getReqId());
        ScheduledFuture<?> pendingTimeout = pendingAcquireSemaphoreTimeouts.get(response.getReqId());
        if (pendingFuture == null) {
            LOG.info("Semaphore acquire {}:{} is already not pending, ignoring",
                    connectionId, response.getReqId());
            return null;
        }
        pendingAcquireSemaphore.remove(response.getReqId());
        pendingAcquireEnqueuedSemaphoreTimeouts.remove(response.getReqId());
        pendingAcquireSemaphoreTimeouts.remove(response.getReqId());
        if (pendingEnqueuedTimeout != null) {
            pendingEnqueuedTimeout.cancel(true);
        }
        if (pendingTimeout != null) {
            pendingTimeout.cancel(true);
        }
        if (response.getStatus().isSuccess()) {
            pendingFuture.complete(response.isAcquired());
        } else {
            pendingFuture.completeExceptionally(new UnexpectedResultException(
                    "Failed to acquire YDB coordination semaphore", response.getStatus().getCode(),
                    response.getStatus().getIssues()));
        }
        return null;
    }

    private Void onReleaseSemaphore(ReleaseSemaphoreResultResponse response, long connectionId) {
        LOG.info("Semaphore release result received: {}", response);
        if (!isCurrentConnection(connectionId)) {
            LOG.info("Received response is not from the currently active connection, ignoring");
            return null;
        }
        CompletableFuture<Boolean> pendingFuture = pendingReleaseSemaphore.get(response.getReqId());
        ScheduledFuture<?> pendingTimeout = pendingReleaseSemaphoreTimeouts.get(response.getReqId());
        if (pendingFuture == null) {
            LOG.info("Semaphore release {}:{} is already not pending, ignoring",
                    connectionId, response.getReqId());
            return null;
        }
        pendingReleaseSemaphore.remove(response.getReqId());
        pendingReleaseSemaphoreTimeouts.remove(response.getReqId());
        if (pendingTimeout != null) {
            pendingTimeout.cancel(true);
        }
        if (response.getStatus().isSuccess()) {
            pendingFuture.complete(response.isReleased());
        } else {
            pendingFuture.completeExceptionally(new UnexpectedResultException(
                    "Failed to release YDB coordination semaphore", response.getStatus().getCode(),
                    response.getStatus().getIssues()));
        }
        return null;
    }

    private Void onDescribeSemaphore(DescribeSemaphoreResultResponse response, long connectionId) {
        LOG.info("Semaphore describe result received: {}", response);
        if (!isCurrentConnection(connectionId)) {
            LOG.info("Received response is not from the currently active connection, ignoring");
            return null;
        }
        PendingDescription pendingDescribeFuture = pendingDescribeSemaphore
                .get(response.getReqId());
        PendingDescription pendingSubscribeFuture = pendingSubscribeSemaphore
                .get(response.getReqId());
        ScheduledFuture<?> pendingDescribeTimeout = pendingDescribeSemaphoreTimeouts.get(response.getReqId());
        ScheduledFuture<?> pendingSubscribeTimeout = pendingSubscribeSemaphoreTimeouts.get(response.getReqId());
        if (pendingDescribeFuture == null && pendingSubscribeFuture == null) {
            LOG.info("Semaphore describe/subscribe {}:{} is already not pending, ignoring",
                    connectionId, response.getReqId());
            return null;
        }
        pendingDescribeSemaphore.remove(response.getReqId());
        pendingDescribeSemaphoreTimeouts.remove(response.getReqId());
        pendingSubscribeSemaphore.remove(response.getReqId());
        pendingSubscribeSemaphoreTimeouts.remove(response.getReqId());
        if (pendingDescribeTimeout != null) {
            pendingDescribeTimeout.cancel(true);
        }
        if (pendingSubscribeTimeout != null) {
            pendingSubscribeTimeout.cancel(true);
        }
        Set<Long> keysToRemove = new HashSet<>();
        eventSubscriptions.forEach((k, v) -> {
            if (pendingDescribeFuture != null
                    && v.getSemaphoreName().equals(pendingDescribeFuture.getSemaphoreName())) {
                LOG.info("Cancelling stale semaphore {} subscription {}", v.getSemaphoreName(), k);
                keysToRemove.add(k);
                v.getSubscription()
                        .completeExceptionally(new SubscriptionCancelledException("Subscription is stale"));
            }
            if (pendingSubscribeFuture != null
                    && v.getSemaphoreName().equals(pendingSubscribeFuture.getSemaphoreName())
                    && !k.equals(response.getReqId())) {
                LOG.info("Cancelling stale semaphore {} subscription {}", v.getSemaphoreName(), k);
                keysToRemove.add(k);
                v.getSubscription()
                        .completeExceptionally(new SubscriptionCancelledException("Subscription is stale"));
            }
        });
        keysToRemove.forEach(eventSubscriptions::remove);
        if (pendingDescribeFuture != null) {
            if (response.getStatus().isSuccess()) {
                pendingDescribeFuture.getFuture().complete(new Description(response.getSemaphoreDescription(),
                        response.isWatchAdded()));
            } else {
                pendingDescribeFuture.getFuture().completeExceptionally(new UnexpectedResultException(
                        "Failed to describe YDB coordination semaphore", response.getStatus().getCode(),
                        response.getStatus().getIssues()));
            }
        } else {
            if (response.getStatus().isSuccess()) {
                pendingSubscribeFuture.getFuture().complete(new Description(response.getSemaphoreDescription(),
                        response.isWatchAdded()));
            } else {
                pendingSubscribeFuture.getFuture().completeExceptionally(new UnexpectedResultException(
                        "Failed to subscribe to YDB coordination semaphore", response.getStatus().getCode(),
                        response.getStatus().getIssues()));
            }
        }
        return null;
    }

    private Void onDescribeSemaphoreChanged(DescribeSemaphoreChangedResponse response, long connectionId) {
        LOG.info("Semaphore describe changed received: {}", response);
        if (!isCurrentConnection(connectionId)) {
            LOG.info("Received response is not from the currently active connection, ignoring");
            return null;
        }
        Subscription subscription = eventSubscriptions.get(response.getReqId());
        if (subscription == null) {
            LOG.info("No event subscription for {}:{}", connectionId, response.getReqId());
            return null;
        }
        eventSubscriptions.remove(response.getReqId());
        Set<Long> keysToRemove = new HashSet<>();
        eventSubscriptions.forEach((k, v) -> {
            if (v.getSemaphoreName().equals(subscription.getSemaphoreName())) {
                LOG.info("Cancelling stale semaphore {} subscription {}", v.getSemaphoreName(), k);
                keysToRemove.add(k);
                v.getSubscription()
                        .completeExceptionally(new SubscriptionCancelledException("Subscription is stale"));
            }
        });
        keysToRemove.forEach(eventSubscriptions::remove);
        subscription.getSubscription()
                .complete(new SemaphoreEvent(response.isDataChanged(), response.isOwnersChanged()));
        return null;
    }

    private Void onCreateSemaphore(CreateSemaphoreResultResponse response, long connectionId) {
        LOG.info("Semaphore creation result received: {}", response);
        if (!isCurrentConnection(connectionId)) {
            LOG.info("Received response is not from the currently active connection, ignoring");
            return null;
        }
        CompletableFuture<Void> pendingFuture = pendingCreateSemaphore.get(response.getReqId());
        ScheduledFuture<?> pendingTimeout = pendingCreateSemaphoreTimeouts.get(response.getReqId());
        if (pendingFuture == null) {
            LOG.info("Semaphore creation {}:{} is already not pending, ignoring",
                    connectionId, response.getReqId());
            return null;
        }
        pendingCreateSemaphore.remove(response.getReqId());
        pendingCreateSemaphoreTimeouts.remove(response.getReqId());
        if (pendingTimeout != null) {
            pendingTimeout.cancel(true);
        }
        if (response.getStatus().isSuccess()) {
            pendingFuture.complete(null);
        } else {
            pendingFuture.completeExceptionally(new UnexpectedResultException(
                    "Failed to create YDB coordination semaphore", response.getStatus().getCode(),
                    response.getStatus().getIssues()));
        }
        return null;
    }

    private Void onUpdateSemaphore(UpdateSemaphoreResultResponse response, long connectionId) {
        LOG.info("Semaphore update result received: {}", response);
        if (!isCurrentConnection(connectionId)) {
            LOG.info("Received response is not from the currently active connection, ignoring");
            return null;
        }
        CompletableFuture<Void> pendingFuture = pendingUpdateSemaphore.get(response.getReqId());
        ScheduledFuture<?> pendingTimeout = pendingUpdateSemaphoreTimeouts.get(response.getReqId());
        if (pendingFuture == null) {
            LOG.info("Semaphore update {}:{} is already not pending, ignoring",
                    connectionId, response.getReqId());
            return null;
        }
        pendingUpdateSemaphore.remove(response.getReqId());
        pendingUpdateSemaphoreTimeouts.remove(response.getReqId());
        if (pendingTimeout != null) {
            pendingTimeout.cancel(true);
        }
        if (response.getStatus().isSuccess()) {
            pendingFuture.complete(null);
        } else {
            pendingFuture.completeExceptionally(new UnexpectedResultException(
                    "Failed to update YDB coordination semaphore", response.getStatus().getCode(),
                    response.getStatus().getIssues()));
        }
        return null;
    }

    private Void onDeleteSemaphore(DeleteSemaphoreResultResponse response, long connectionId) {
        LOG.info("Semaphore deletion result received: {}", response);
        if (!isCurrentConnection(connectionId)) {
            LOG.info("Received response is not from the currently active connection, ignoring");
            return null;
        }
        CompletableFuture<Void> pendingFuture = pendingDeleteSemaphore.get(response.getReqId());
        ScheduledFuture<?> pendingTimeout = pendingDeleteSemaphoreTimeouts.get(response.getReqId());
        if (pendingFuture == null) {
            LOG.info("Semaphore deletion {}:{} is already not pending, ignoring",
                    connectionId, response.getReqId());
            return null;
        }
        pendingDeleteSemaphore.remove(response.getReqId());
        pendingDeleteSemaphoreTimeouts.remove(response.getReqId());
        if (pendingTimeout != null) {
            pendingTimeout.cancel(true);
        }
        if (response.getStatus().isSuccess()) {
            pendingFuture.complete(null);
        } else {
            pendingFuture.completeExceptionally(new UnexpectedResultException(
                    "Failed to delete YDB coordination semaphore", response.getStatus().getCode(),
                    response.getStatus().getIssues()));
        }
        return null;
    }

    private void handleConnectionError(CompletableFuture<SessionStartedResponse> sessionStartFuture,
                                       CompletableFuture<SessionStoppedResponse> sessionStopFuture, Throwable e,
                                       long connectionId) {
        LOG.error("YDB coordination connection failure", e);
        if (!isCurrentConnection(connectionId)) {
            LOG.info("Connection error is not from the currently active connection, ignoring");
            return;
        }
        sessionStopPending = false;
        lastSessionBreakAt.set(Instant.now());
        boolean startWasCompleted = sessionStartFuture.completeExceptionally(e);
        if (startWasCompleted) {
            sessionStopFuture.completeExceptionally(new CancellationException("Session was not yet started"));
        }
        SessionState oldState = sessionState.getAndSet(SessionState.INVALID);
        if (oldState != SessionState.INVALID) {
            onSessionStop.run();
            deliverSubscriptions(SessionState.INVALID);
        }
        reconnectBarrier.open();
    }

    private void handleConnectionCompletion(CompletableFuture<SessionStartedResponse> sessionStartFuture,
                                            CompletableFuture<SessionStoppedResponse> sessionStopFuture,
                                            long connectionId) {
        LOG.info("YDB coordination connection is complete");
        if (!isCurrentConnection(connectionId)) {
            LOG.info("Connection completion is not from the currently active connection, ignoring");
            return;
        }
        sessionStopPending = false;
        boolean startWasCompleted = sessionStartFuture.completeExceptionally(new CancellationException(
                "YDB coordination connection is complete while session is not yet started"));
        if (startWasCompleted) {
            sessionStopFuture.completeExceptionally(new CancellationException("Session was not yet started"));
        }
        SessionState oldState = sessionState.getAndSet(SessionState.INVALID);
        if (oldState != SessionState.INVALID) {
            onSessionStop.run();
            deliverSubscriptions(SessionState.INVALID);
        }
        reconnectBarrier.open();
    }

    private void shutdownScheduler() {
        LOG.info("Shutting down YDB coordination scheduled thread pool...");
        scheduler.shutdown();
        try {
            scheduler.awaitTermination(schedulerShutdownTimeout.toMillis(), TimeUnit.MILLISECONDS);
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        scheduler.shutdownNow();
        LOG.info("Finished shutting down YDB coordination scheduled thread pool");
    }

    private void shutdownExecutor() {
        LOG.info("Shutting down YDB coordination session executor thread pool...");
        sessionExecutorService.shutdown();
        try {
            sessionExecutorService.awaitTermination(schedulerShutdownTimeout.toMillis(), TimeUnit.MILLISECONDS);
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        sessionExecutorService.shutdownNow();
        LOG.info("Finished shutting down YDB coordination session executor thread pool");
    }

    @SuppressWarnings("FutureReturnValueIgnored")
    private void schedulePing(long connectionId) {
        scheduler.schedule(() -> {
            try {
                doPing(connectionId);
            } catch (Throwable e) {
                LOG.error("Error while processing scheduled YDB coordination session periodic ping");
                Exceptions.throwIfJvmFatal(e);
            }
        }, pingPeriod.toMillis(), TimeUnit.MILLISECONDS);
    }

    private ScheduledFuture<?> schedulePingTimeout(long connectionId, long currentPingNumber) {
        return scheduler.schedule(() -> {
            try {
                doPingTimeout(connectionId, currentPingNumber);
            } catch (Throwable e) {
                LOG.error("Error while processing scheduled YDB coordination session ping timeout");
                Exceptions.throwIfJvmFatal(e);
            }
        }, pingTimeout.toMillis(), TimeUnit.MILLISECONDS);
    }

    private void doPing(long connectionId) {
        LOG.info("Processing scheduled YDB coordination session periodic ping");
        ConnectionHolder connection = currentConnection.get();
        if (connection != null) {
            if (connection.getId() != connectionId) {
                LOG.info("Scheduled ping is not for current connection, skipping, expected {}, actual {}",
                        connectionId, connection.getId());
                return;
            }
            long currentPingNumber = pingNumber.incrementAndGet();
            LOG.info("Scheduling current YDB coordination session ping timeout...");
            ScheduledFuture<?> scheduledTimeout = schedulePingTimeout(connectionId, currentPingNumber);
            pendingPingTimeouts.put(currentPingNumber, scheduledTimeout);
            LOG.info("Sending YDB coordination session ping {}:{}...", connectionId, currentPingNumber);
            pendingPings.put(currentPingNumber, currentPingNumber);
            connection.getRequestObserver().onNext(CoordinationSessionRequest
                    .ping(new PingPongRequest(currentPingNumber)));
            LOG.info("Scheduling next YDB coordination session ping...");
            schedulePing(connectionId);
            LOG.info("Done with current YDB coordination session ping...");
        } else {
            LOG.info("No YDB coordination connection to send scheduled ping");
        }
    }

    private void doPingTimeout(long connectionId, long currentPingNumber) {
        LOG.info("Processing YDB coordination session ping timeout for {}:{}", connectionId, currentPingNumber);
        pendingPingTimeouts.remove(currentPingNumber);
        ConnectionHolder connection = currentConnection.get();
        if (connection != null) {
            if (connection.getId() != connectionId) {
                LOG.info("Scheduled ping timeout is not for current connection, skipping, expected {}, actual {}",
                        connectionId, connection.getId());
                return;
            }
            if (!pendingPings.containsKey(currentPingNumber)) {
                LOG.info("YDB coordination session ping {}:{} is not pending already",
                        connectionId, currentPingNumber);
                return;
            }
            LOG.info("Registering ping {}:{} as lost", connectionId, currentPingNumber);
            pendingPings.remove(currentPingNumber);
            long lostPingsCount = lostPings.incrementAndGet();
            if (lostPingsCount > lostPingsThreshold) {
                LOG.error("YDB coordination lost pings threshold exceeded");
                sessionStopPending = false;
                lastSessionBreakAt.set(Instant.now());
                boolean startWasCompleted = connection.getSessionStartFuture()
                        .completeExceptionally(new CancellationException(
                                "YDB coordination lost pings threshold exceeded"));
                if (startWasCompleted) {
                    connection.getSessionStopFuture()
                            .completeExceptionally(new CancellationException("Session was not yet started"));
                }
                SessionState oldState = sessionState.getAndSet(SessionState.INVALID);
                if (oldState != SessionState.INVALID) {
                    onSessionStop.run();
                    deliverSubscriptions(SessionState.INVALID);
                }
                reconnectBarrier.open();
            }
        } else {
            LOG.info("No YDB coordination connection to process ping {}:{} timeout", connectionId, currentPingNumber);
        }
    }

    private ScheduledFuture<?> scheduleCreateSemaphoreTimeout(long requestId) {
        return scheduler.schedule(() -> {
            try {
                doSemaphoreCreationTimeout(requestId);
            } catch (Throwable e) {
                LOG.error("Error while processing scheduled YDB coordination semaphore creation timeout");
                Exceptions.throwIfJvmFatal(e);
            }
        }, semaphoreManagementTimeout.toMillis(), TimeUnit.MILLISECONDS);
    }

    private void doSemaphoreCreationTimeout(long requestId) {
        LOG.info("Processing YDB coordination semaphore creation timeout for {}", requestId);
        pendingCreateSemaphoreTimeouts.remove(requestId);
        CompletableFuture<Void> pendingFuture = pendingCreateSemaphore.get(requestId);
        if (pendingFuture == null) {
            LOG.info("YDB coordination semaphore creation {} is not pending already", requestId);
            return;
        }
        LOG.info("Registering semaphore creation request {} as lost", requestId);
        pendingCreateSemaphore.remove(requestId);
        pendingFuture.completeExceptionally(new TimeoutException("Semaphore creation request timed out"));
    }

    private ScheduledFuture<?> scheduleDeleteSemaphoreTimeout(long requestId) {
        return scheduler.schedule(() -> {
            try {
                doSemaphoreDeletionTimeout(requestId);
            } catch (Throwable e) {
                LOG.error("Error while processing scheduled YDB coordination semaphore deletion timeout");
                Exceptions.throwIfJvmFatal(e);
            }
        }, semaphoreManagementTimeout.toMillis(), TimeUnit.MILLISECONDS);
    }

    private void doSemaphoreDeletionTimeout(long requestId) {
        LOG.info("Processing YDB coordination semaphore deletion timeout for {}", requestId);
        pendingDeleteSemaphoreTimeouts.remove(requestId);
        CompletableFuture<Void> pendingFuture = pendingDeleteSemaphore.get(requestId);
        if (pendingFuture == null) {
            LOG.info("YDB coordination semaphore deletion {} is not pending already", requestId);
            return;
        }
        LOG.info("Registering semaphore deletion request {} as lost", requestId);
        pendingDeleteSemaphore.remove(requestId);
        pendingFuture.completeExceptionally(new TimeoutException("Semaphore deletion request timed out"));
    }

    private ScheduledFuture<?> scheduleUpdateSemaphoreTimeout(long requestId) {
        return scheduler.schedule(() -> {
            try {
                doSemaphoreUpdateTimeout(requestId);
            } catch (Throwable e) {
                LOG.error("Error while processing scheduled YDB coordination semaphore update timeout");
                Exceptions.throwIfJvmFatal(e);
            }
        }, semaphoreManagementTimeout.toMillis(), TimeUnit.MILLISECONDS);
    }

    private void doSemaphoreUpdateTimeout(long requestId) {
        LOG.info("Processing YDB coordination semaphore update timeout for {}", requestId);
        pendingUpdateSemaphoreTimeouts.remove(requestId);
        CompletableFuture<Void> pendingFuture = pendingUpdateSemaphore.get(requestId);
        if (pendingFuture == null) {
            LOG.info("YDB coordination semaphore update {} is not pending already", requestId);
            return;
        }
        LOG.info("Registering semaphore update request {} as lost", requestId);
        pendingUpdateSemaphore.remove(requestId);
        pendingFuture.completeExceptionally(new TimeoutException("Semaphore update request timed out"));
    }

    private ScheduledFuture<?> scheduleAcquireSemaphoreTimeout(long requestId) {
        return scheduler.schedule(() -> {
            try {
                doSemaphoreAcquireTimeout(requestId);
            } catch (Throwable e) {
                LOG.error("Error while processing scheduled YDB coordination semaphore acquire timeout");
                Exceptions.throwIfJvmFatal(e);
            }
        }, acquireSemaphoreTimeout.toMillis(), TimeUnit.MILLISECONDS);
    }

    private void doSemaphoreAcquireTimeout(long requestId) {
        LOG.info("Processing YDB coordination semaphore acquire timeout for {}", requestId);
        pendingAcquireSemaphoreTimeouts.remove(requestId);
        CompletableFuture<Boolean> pendingFuture = pendingAcquireSemaphore.get(requestId);
        if (pendingFuture == null) {
            LOG.info("YDB coordination semaphore acquire {} is not pending already", requestId);
            return;
        }
        LOG.info("Registering semaphore acquire request {} as lost", requestId);
        pendingAcquireSemaphore.remove(requestId);
        pendingFuture.completeExceptionally(new TimeoutException("Semaphore acquire request timed out"));
        ScheduledFuture<?> pendingTimeout = pendingAcquireEnqueuedSemaphoreTimeouts.get(requestId);
        if (pendingTimeout != null) {
            pendingTimeout.cancel(true);
        }
        pendingAcquireEnqueuedSemaphoreTimeouts.remove(requestId);
    }

    private ScheduledFuture<?> scheduleAcquireSemaphoreEnqueuedTimeout(long requestId) {
        return scheduler.schedule(() -> {
            try {
                doSemaphoreAcquireEnqueuedTimeout(requestId);
            } catch (Throwable e) {
                LOG.error("Error while processing scheduled YDB coordination semaphore acquire enqueued timeout");
                Exceptions.throwIfJvmFatal(e);
            }
        }, acquireEnqueuedSemaphoreTimeout.toMillis(), TimeUnit.MILLISECONDS);
    }

    private void doSemaphoreAcquireEnqueuedTimeout(long requestId) {
        LOG.info("Processing YDB coordination semaphore acquire enqueued timeout for {}", requestId);
        ScheduledFuture<?> pendingEnqueuedTimeout = pendingAcquireEnqueuedSemaphoreTimeouts.get(requestId);
        if (pendingEnqueuedTimeout == null) {
            LOG.info("YDB coordination semaphore acquire enqueue {} is not pending already", requestId);
            return;
        }
        pendingAcquireEnqueuedSemaphoreTimeouts.remove(requestId);
        CompletableFuture<Boolean> pendingFuture = pendingAcquireSemaphore.get(requestId);
        if (pendingFuture == null) {
            LOG.info("YDB coordination semaphore acquire {} is not pending already", requestId);
            return;
        }
        LOG.info("Registering semaphore acquire request {} as lost due to missing enqueue confirmation", requestId);
        pendingAcquireSemaphore.remove(requestId);
        pendingFuture.completeExceptionally(new TimeoutException("Semaphore acquire request timed out"));
        ScheduledFuture<?> pendingTimeout = pendingAcquireSemaphoreTimeouts.get(requestId);
        if (pendingTimeout != null) {
            pendingTimeout.cancel(true);
        }
        pendingAcquireSemaphoreTimeouts.remove(requestId);
    }

    private ScheduledFuture<?> scheduleReleaseSemaphoreTimeout(long requestId) {
        return scheduler.schedule(() -> {
            try {
                doSemaphoreReleaseTimeout(requestId);
            } catch (Throwable e) {
                LOG.error("Error while processing scheduled YDB coordination semaphore release timeout");
                Exceptions.throwIfJvmFatal(e);
            }
        }, releaseSemaphoreTimeout.toMillis(), TimeUnit.MILLISECONDS);
    }

    private void doSemaphoreReleaseTimeout(long requestId) {
        LOG.info("Processing YDB coordination semaphore release timeout for {}", requestId);
        pendingReleaseSemaphoreTimeouts.remove(requestId);
        CompletableFuture<Boolean> pendingFuture = pendingReleaseSemaphore.get(requestId);
        if (pendingFuture == null) {
            LOG.info("YDB coordination semaphore release {} is not pending already", requestId);
            return;
        }
        LOG.info("Registering semaphore release request {} as lost", requestId);
        pendingReleaseSemaphore.remove(requestId);
        pendingFuture.completeExceptionally(new TimeoutException("Semaphore release request timed out"));
    }

    private ScheduledFuture<?> scheduleDescribeSemaphoreTimeout(long requestId) {
        return scheduler.schedule(() -> {
            try {
                doSemaphoreDescribeTimeout(requestId);
            } catch (Throwable e) {
                LOG.error("Error while processing scheduled YDB coordination semaphore describe timeout");
                Exceptions.throwIfJvmFatal(e);
            }
        }, describeSemaphoreTimeout.toMillis(), TimeUnit.MILLISECONDS);
    }

    private void doSemaphoreDescribeTimeout(long requestId) {
        LOG.info("Processing YDB coordination semaphore describe timeout for {}", requestId);
        pendingDescribeSemaphoreTimeouts.remove(requestId);
        PendingDescription pendingFuture = pendingDescribeSemaphore.get(requestId);
        if (pendingFuture == null) {
            LOG.info("YDB coordination semaphore describe {} is not pending already", requestId);
            return;
        }
        LOG.info("Registering semaphore describe request {} as lost", requestId);
        pendingDescribeSemaphore.remove(requestId);
        pendingFuture.getFuture()
                .completeExceptionally(new TimeoutException("Semaphore describe request timed out"));
    }

    private ScheduledFuture<?> scheduleSubscribeSemaphoreTimeout(long requestId) {
        return scheduler.schedule(() -> {
            try {
                doSemaphoreSubscribeTimeout(requestId);
            } catch (Throwable e) {
                LOG.error("Error while processing scheduled YDB coordination semaphore subscribe timeout");
                Exceptions.throwIfJvmFatal(e);
            }
        }, describeSemaphoreTimeout.toMillis(), TimeUnit.MILLISECONDS);
    }

    private void doSemaphoreSubscribeTimeout(long requestId) {
        LOG.info("Processing YDB coordination semaphore subscribe timeout for {}", requestId);
        pendingSubscribeSemaphoreTimeouts.remove(requestId);
        PendingDescription pendingFuture = pendingSubscribeSemaphore.get(requestId);
        if (pendingFuture == null) {
            LOG.info("YDB coordination semaphore subscribe {} is not pending already", requestId);
            return;
        }
        LOG.info("Registering semaphore subscribe request {} as lost", requestId);
        pendingSubscribeSemaphore.remove(requestId);
        pendingFuture.getFuture()
                .completeExceptionally(new TimeoutException("Semaphore subscribe request timed out"));
    }

    private void deliverSubscriptions(SessionState sessionState) {
        stateSubscribers.forEach(s -> s.accept(sessionState));
    }

    private static byte[] generateSessionKey() {
        byte[] result = new byte[16];
        new Random().nextBytes(result);
        return result;
    }

    private static ScheduledExecutorService prepareScheduler() {
        ThreadFactory threadFactory = new ThreadFactoryBuilder()
                .setDaemon(true)
                .setNameFormat("ydb-coordination-scheduling-pool-%d")
                .setUncaughtExceptionHandler((t, e) -> LOG
                        .error("Uncaught exception in ydb coordination scheduler thread " + t, e))
                .build();
        ScheduledThreadPoolExecutor scheduledThreadPoolExecutor = new ScheduledThreadPoolExecutor(2,
                threadFactory);
        scheduledThreadPoolExecutor.setRemoveOnCancelPolicy(true);
        return scheduledThreadPoolExecutor;
    }

    private static ExecutorService prepareSessionExecutor(int executorPoolThreads) {
        ThreadFactory threadFactory = new ThreadFactoryBuilder()
                .setDaemon(true)
                .setNameFormat("ydb-coordination-session-executor-pool-%d")
                .setUncaughtExceptionHandler((t, e) -> LOG
                        .error("Uncaught exception in ydb coordination session executor thread " + t, e))
                .build();
        return Executors.newFixedThreadPool(executorPoolThreads, threadFactory);
    }

    private static class PendingDescription {

        private final CompletableFuture<Description> future;
        private final String semaphoreName;

        private PendingDescription(CompletableFuture<Description> future, String semaphoreName) {
            this.future = future;
            this.semaphoreName = semaphoreName;
        }

        public CompletableFuture<Description> getFuture() {
            return future;
        }

        public String getSemaphoreName() {
            return semaphoreName;
        }

    }

    private static class Subscription {

        private final CompletableFuture<SemaphoreEvent> subscription;
        private final String semaphoreName;

        private Subscription(CompletableFuture<SemaphoreEvent> subscription, String semaphoreName) {
            this.subscription = subscription;
            this.semaphoreName = semaphoreName;
        }

        public CompletableFuture<SemaphoreEvent> getSubscription() {
            return subscription;
        }

        public String getSemaphoreName() {
            return semaphoreName;
        }

    }

    private static class Description {

        private final CoordinationSemaphoreDescription description;
        private final boolean watchAdded;

        private Description(CoordinationSemaphoreDescription description, boolean watchAdded) {
            this.description = description;
            this.watchAdded = watchAdded;
        }

        public CoordinationSemaphoreDescription getDescription() {
            return description;
        }

        public boolean isWatchAdded() {
            return watchAdded;
        }

    }

    private static class ConnectionHolder {

        private final OutStreamObserver<CoordinationSessionRequest> requestObserver;
        private final CompletableFuture<SessionStartedResponse> sessionStartFuture;
        private final CompletableFuture<SessionStoppedResponse> sessionStopFuture;
        private final long id;

        private ConnectionHolder(OutStreamObserver<CoordinationSessionRequest> requestObserver,
                                 CompletableFuture<SessionStartedResponse> sessionStartFuture,
                                 CompletableFuture<SessionStoppedResponse> sessionStopFuture,
                                 long id) {
            this.requestObserver = requestObserver;
            this.sessionStartFuture = sessionStartFuture;
            this.sessionStopFuture = sessionStopFuture;
            this.id = id;
        }

        public OutStreamObserver<CoordinationSessionRequest> getRequestObserver() {
            return requestObserver;
        }

        public CompletableFuture<SessionStartedResponse> getSessionStartFuture() {
            return sessionStartFuture;
        }

        public CompletableFuture<SessionStoppedResponse> getSessionStopFuture() {
            return sessionStopFuture;
        }

        public long getId() {
            return id;
        }

    }

    private static class SessionObserver implements StreamObserver<CoordinationSessionResponse> {

        private final AtomicReference<OutStreamObserver<CoordinationSessionRequest>> requestObserver
                = new AtomicReference<>();
        private final CompletableFuture<SessionStartedResponse> sessionStartFuture;
        private final CompletableFuture<SessionStoppedResponse> sessionStopFuture;
        private final long connectionId;
        private final SessionInitiator sessionInitiator;

        private SessionObserver(CompletableFuture<SessionStartedResponse> sessionStartFuture,
                                CompletableFuture<SessionStoppedResponse> sessionStopFuture,
                                long connectionId, SessionInitiator sessionInitiator) {
            this.sessionStartFuture = sessionStartFuture;
            this.sessionStopFuture = sessionStopFuture;
            this.connectionId = connectionId;
            this.sessionInitiator = sessionInitiator;
        }

        @Override
        @SuppressWarnings("FutureReturnValueIgnored")
        public void onNext(CoordinationSessionResponse value) {
            sessionInitiator.sessionExecutorService.submit(() ->
                    sessionInitiator.handleConnectionResponse(requestObserver.get(), sessionStartFuture,
                            sessionStopFuture, connectionId, value));
        }

        @Override
        @SuppressWarnings("FutureReturnValueIgnored")
        public void onError(Status status) {
            sessionInitiator.sessionExecutorService.submit(() -> {
                UnexpectedResultException exception = new UnexpectedResultException(
                        "Failure in YDB coordination session", status.getCode(), status.getIssues());
                sessionInitiator.handleConnectionError(sessionStartFuture, sessionStopFuture,
                        exception, connectionId);
            });
        }

        @Override
        @SuppressWarnings("FutureReturnValueIgnored")
        public void onCompleted() {
            sessionInitiator.sessionExecutorService.submit(() ->
                    sessionInitiator.handleConnectionCompletion(sessionStartFuture, sessionStopFuture, connectionId));
        }

        public void setRequestObserver(OutStreamObserver<CoordinationSessionRequest> requestObserver) {
            this.requestObserver.set(requestObserver);
        }

    }

}
