package ru.yandex.intranet.d.datasource.coordination.model;

import java.time.Duration;
import java.util.Objects;

/**
 * YDB coordination node configuration.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class NodeConfig {

    private final String path;
    private final Duration selfCheckPeriod;
    private final Duration sessionGracePeriod;
    private final NodeConsistencyMode readConsistencyMode;
    private final NodeConsistencyMode attachConsistencyMode;
    private final NodeRateLimiterCountersMode rateLimiterCountersMode;

    public NodeConfig(String path, Duration selfCheckPeriod, Duration sessionGracePeriod,
                      NodeConsistencyMode readConsistencyMode, NodeConsistencyMode attachConsistencyMode,
                      NodeRateLimiterCountersMode rateLimiterCountersMode) {
        this.path = path;
        this.selfCheckPeriod = selfCheckPeriod;
        this.sessionGracePeriod = sessionGracePeriod;
        this.readConsistencyMode = readConsistencyMode;
        this.attachConsistencyMode = attachConsistencyMode;
        this.rateLimiterCountersMode = rateLimiterCountersMode;
    }

    /**
     * Get node path.
     * @return Node path.
     */
    public String getPath() {
        return path;
    }

    /**
     * Get period for self-checks.
     * @return Period for self-checks.
     */
    public Duration getSelfCheckPeriod() {
        return selfCheckPeriod;
    }

    /**
     * Get grace period for sessions on master change.
     * @return Grace period for sessions on master change.
     */
    public Duration getSessionGracePeriod() {
        return sessionGracePeriod;
    }

    /**
     * Get consistency mode for read operations.
     * @return Consistency mode for read operations.
     */
    public NodeConsistencyMode getReadConsistencyMode() {
        return readConsistencyMode;
    }

    /**
     * Get consistency mode for attach operations.
     * @return Consistency mode for attach operations.
     */
    public NodeConsistencyMode getAttachConsistencyMode() {
        return attachConsistencyMode;
    }

    /**
     * Get rate limiter counters mode.
     * @return Rate limiter counters mode.
     */
    public NodeRateLimiterCountersMode getRateLimiterCountersMode() {
        return rateLimiterCountersMode;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        NodeConfig that = (NodeConfig) o;
        return Objects.equals(path, that.path) &&
                Objects.equals(selfCheckPeriod, that.selfCheckPeriod) &&
                Objects.equals(sessionGracePeriod, that.sessionGracePeriod) &&
                readConsistencyMode == that.readConsistencyMode &&
                attachConsistencyMode == that.attachConsistencyMode &&
                rateLimiterCountersMode == that.rateLimiterCountersMode;
    }

    @Override
    public int hashCode() {
        return Objects.hash(path, selfCheckPeriod, sessionGracePeriod, readConsistencyMode, attachConsistencyMode,
                rateLimiterCountersMode);
    }

    @Override
    public String toString() {
        return "NodeConfig{" +
                "path='" + path + '\'' +
                ", selfCheckPeriod=" + selfCheckPeriod +
                ", sessionGracePeriod=" + sessionGracePeriod +
                ", readConsistencyMode=" + readConsistencyMode +
                ", attachConsistencyMode=" + attachConsistencyMode +
                ", rateLimiterCountersMode=" + rateLimiterCountersMode +
                '}';
    }

}
