package ru.yandex.intranet.d.datasource.coordination.model;

import java.time.Duration;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

/**
 * Operation parameters.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class OperationParameters {

    private final OperationMode operationMode;
    private final Duration operationTimeout;
    private final Duration cancelAfter;
    private final Map<String, String> labels;
    private final long deadlineAfterNanos;

    private OperationParameters(OperationMode operationMode, Duration operationTimeout, Duration cancelAfter,
                                Map<String, String> labels, long deadlineAfterNanos) {
        this.operationMode = operationMode;
        this.operationTimeout = operationTimeout;
        this.cancelAfter = cancelAfter;
        this.labels = labels;
        this.deadlineAfterNanos = deadlineAfterNanos;
    }

    public static Builder builder() {
        return new Builder();
    }

    public OperationMode getOperationMode() {
        return operationMode;
    }

    /**
     * Indicates that client is no longer interested in the result of operation after the specified duration
     * starting from the time operation arrives at the server. Server will try to stop the execution of operation
     * and if no result is currently available the operation will receive TIMEOUT status code, which will be sent
     * back to client if it was waiting for the operation result. Timeout of operation does not tell anything about
     * its result, it might be completed successfully or cancelled on server.
     * @return Operation timeout.
     */
    public Optional<Duration> getOperationTimeout() {
        return Optional.ofNullable(operationTimeout);
    }

    /**
     * Server will try to cancel the operation after the specified duration starting from the time the operation
     * arrives at server. In case of successful cancellation operation will receive CANCELLED status code, which
     * will be sent back to client if it was waiting for the operation result. In case when cancellation isn't
     * possible, no action will be performed.
     * @return Cancel after timeout.
     */
    public Optional<Duration> getCancelAfter() {
        return Optional.ofNullable(cancelAfter);
    }

    /**
     * Get user-defined labels of operation.
     * @return User-defined labels of operation.
     */
    public Map<String, String> getLabels() {
        return labels;
    }

    public long getDeadlineAfterNanos() {
        return deadlineAfterNanos;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        OperationParameters that = (OperationParameters) o;
        return deadlineAfterNanos == that.deadlineAfterNanos &&
                operationMode == that.operationMode &&
                Objects.equals(operationTimeout, that.operationTimeout) &&
                Objects.equals(cancelAfter, that.cancelAfter) &&
                Objects.equals(labels, that.labels);
    }

    @Override
    public int hashCode() {
        return Objects.hash(operationMode, operationTimeout, cancelAfter, labels, deadlineAfterNanos);
    }

    @Override
    public String toString() {
        return "OperationParameters{" +
                "operationMode=" + operationMode +
                ", operationTimeout=" + operationTimeout +
                ", cancelAfter=" + cancelAfter +
                ", labels=" + labels +
                ", deadlineAfterNanos=" + deadlineAfterNanos +
                '}';
    }

    public static final class Builder {

        private final Map<String, String> labels = new HashMap<>();
        private final OperationMode operationMode = OperationMode.SYNC;

        private Duration operationTimeout;
        private Duration cancelAfter;
        private long deadlineAfterNanos = 0;

        private Builder() {
        }

        public Builder operationTimeout(Duration operationTimeout) {
            this.operationTimeout = operationTimeout;
            return this;
        }

        public Builder cancelAfter(Duration cancelAfter) {
            this.cancelAfter = cancelAfter;
            return this;
        }

        public Builder deadlineAfter(Duration deadlineAfter) {
            this.deadlineAfterNanos = System.nanoTime() + deadlineAfter.toNanos();
            return this;
        }

        public OperationParameters build() {
            return new OperationParameters(operationMode, operationTimeout, cancelAfter, labels, deadlineAfterNanos);
        }

    }

}
