package ru.yandex.intranet.d.datasource.coordination.model;

import java.time.Duration;
import java.util.Objects;

/**
 * YDB coordination node configuration.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class TargetNodeConfig {

    private final Duration selfCheckPeriod;
    private final Duration sessionGracePeriod;
    private final NodeConsistencyMode readConsistencyMode;
    private final NodeConsistencyMode attachConsistencyMode;
    private final NodeRateLimiterCountersMode rateLimiterCountersMode;

    private TargetNodeConfig(Duration selfCheckPeriod,
                             Duration sessionGracePeriod,
                             NodeConsistencyMode readConsistencyMode,
                             NodeConsistencyMode attachConsistencyMode,
                             NodeRateLimiterCountersMode rateLimiterCountersMode) {
        this.selfCheckPeriod = selfCheckPeriod;
        this.sessionGracePeriod = sessionGracePeriod;
        this.readConsistencyMode = readConsistencyMode;
        this.attachConsistencyMode = attachConsistencyMode;
        this.rateLimiterCountersMode = rateLimiterCountersMode;
    }

    public static Builder builder() {
        return new Builder();
    }

    /**
     * Get period for self-checks.
     * @return Period for self-checks.
     */
    public Duration getSelfCheckPeriod() {
        return selfCheckPeriod;
    }

    /**
     * Get grace period for sessions on master change.
     * @return Grace period for sessions on master change.
     */
    public Duration getSessionGracePeriod() {
        return sessionGracePeriod;
    }

    /**
     * Get consistency mode for read operations.
     * @return Consistency mode for read operations.
     */
    public NodeConsistencyMode getReadConsistencyMode() {
        return readConsistencyMode;
    }

    /**
     * Get consistency mode for attach operations.
     * @return Consistency mode for attach operations.
     */
    public NodeConsistencyMode getAttachConsistencyMode() {
        return attachConsistencyMode;
    }

    /**
     * Get rate limiter counters mode.
     * @return Rate limiter counters mode.
     */
    public NodeRateLimiterCountersMode getRateLimiterCountersMode() {
        return rateLimiterCountersMode;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        TargetNodeConfig that = (TargetNodeConfig) o;
        return Objects.equals(selfCheckPeriod, that.selfCheckPeriod) &&
                Objects.equals(sessionGracePeriod, that.sessionGracePeriod) &&
                readConsistencyMode == that.readConsistencyMode &&
                attachConsistencyMode == that.attachConsistencyMode &&
                rateLimiterCountersMode == that.rateLimiterCountersMode;
    }

    @Override
    public int hashCode() {
        return Objects.hash(selfCheckPeriod, sessionGracePeriod, readConsistencyMode,
                attachConsistencyMode, rateLimiterCountersMode);
    }

    @Override
    public String toString() {
        return "TargetNodeConfig{" +
                "selfCheckPeriod=" + selfCheckPeriod +
                ", sessionGracePeriod=" + sessionGracePeriod +
                ", readConsistencyMode=" + readConsistencyMode +
                ", attachConsistencyMode=" + attachConsistencyMode +
                ", rateLimiterCountersMode=" + rateLimiterCountersMode +
                '}';
    }

    public static final class Builder {

        private Duration selfCheckPeriod = Duration.ofSeconds(1);
        private Duration sessionGracePeriod = Duration.ofSeconds(10);
        private NodeConsistencyMode readConsistencyMode = NodeConsistencyMode.UNSET;
        private NodeConsistencyMode attachConsistencyMode = NodeConsistencyMode.UNSET;
        private NodeRateLimiterCountersMode rateLimiterCountersMode = NodeRateLimiterCountersMode.UNSET;

        private Builder() {
        }

        public Builder selfCheckPeriod(Duration selfCheckPeriod) {
            this.selfCheckPeriod = selfCheckPeriod;
            return this;
        }

        public Builder sessionGracePeriod(Duration sessionGracePeriod) {
            this.sessionGracePeriod = sessionGracePeriod;
            return this;
        }

        public Builder readConsistencyMode(NodeConsistencyMode readConsistencyMode) {
            this.readConsistencyMode = readConsistencyMode;
            return this;
        }

        public Builder attachConsistencyMode(NodeConsistencyMode attachConsistencyMode) {
            this.attachConsistencyMode = attachConsistencyMode;
            return this;
        }

        public Builder rateLimiterCountersMode(NodeRateLimiterCountersMode rateLimiterCountersMode) {
            this.rateLimiterCountersMode = rateLimiterCountersMode;
            return this;
        }

        public TargetNodeConfig build() {
            return new TargetNodeConfig(selfCheckPeriod, sessionGracePeriod, readConsistencyMode,
                    attachConsistencyMode, rateLimiterCountersMode);
        }

    }

}
