package ru.yandex.intranet.d.datasource.coordination.model.session;

import java.time.Duration;
import java.util.Arrays;
import java.util.Objects;

/**
 * Acquire semaphore request. Used to acquire a semaphore. A single session cannot acquire the same semaphore multiple
 * times. Later requests override previous operations with the same semaphore, e.g. to reduce acquired count, change
 * timeout or attached data.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class AcquireSemaphoreRequest {

    private final long reqId;
    private final String name;
    private final Duration timeout;
    private final long count;
    private final byte[] data;
    private final boolean ephemeral;

    public AcquireSemaphoreRequest(long reqId, String name, Duration timeout, long count, byte[] data,
                                   boolean ephemeral) {
        this.reqId = reqId;
        this.name = name;
        this.timeout = timeout;
        this.count = count;
        this.data = data;
        this.ephemeral = ephemeral;
    }

    /**
     * Client-defined request id, echoed in the response.
     * @return Client-defined request id.
     */
    public long getReqId() {
        return reqId;
    }

    /**
     * Get name of the semaphore to acquire.
     * @return Name of the semaphore to acquire.
     */
    public String getName() {
        return name;
    }

    /**
     * Timeout after which operation will fail if it's still waiting in the waiters queue.
     * @return Timeout.
     */
    public Duration getTimeout() {
        return timeout;
    }

    /**
     * Get number of tokens to acquire on the semaphore.
     * @return Number of tokens to acquire on the semaphore.
     */
    public long getCount() {
        return count;
    }

    /**
     * Get user-defined binary data that may be attached to the operation.
     * @return User-defined binary data that may be attached to the operation.
     */
    public byte[] getData() {
        return data;
    }

    /**
     * Ephemeral semaphores are created with the first acquire operation and automatically deleted with the last
     * release operation.
     * @return Ephemeral flag.
     */
    public boolean isEphemeral() {
        return ephemeral;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        AcquireSemaphoreRequest that = (AcquireSemaphoreRequest) o;
        return reqId == that.reqId &&
                count == that.count &&
                ephemeral == that.ephemeral &&
                Objects.equals(name, that.name) &&
                Objects.equals(timeout, that.timeout) &&
                Arrays.equals(data, that.data);
    }

    @Override
    public int hashCode() {
        int result = Objects.hash(reqId, name, timeout, count, ephemeral);
        result = 31 * result + Arrays.hashCode(data);
        return result;
    }

    @Override
    public String toString() {
        return "AcquireSemaphoreRequest{" +
                "reqId=" + reqId +
                ", name='" + name + '\'' +
                ", timeout=" + timeout +
                ", count=" + count +
                ", data=" + Arrays.toString(data) +
                ", ephemeral=" + ephemeral +
                '}';
    }

}
