package ru.yandex.intranet.d.datasource.coordination.model.session;

import java.util.Arrays;
import java.util.List;
import java.util.Objects;

/**
 * Semaphore description. Describes the state of a semaphore.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class CoordinationSemaphoreDescription {

    private final String name;
    private final byte[] data;
    private final long count;
    private final long limit;
    private final boolean ephemeral;
    private final List<CoordinationSemaphoreSession> owners;
    private final List<CoordinationSemaphoreSession> waiters;

    public CoordinationSemaphoreDescription(String name, byte[] data, long count, long limit, boolean ephemeral,
                                            List<CoordinationSemaphoreSession> owners,
                                            List<CoordinationSemaphoreSession> waiters) {
        this.name = name;
        this.data = data;
        this.count = count;
        this.limit = limit;
        this.ephemeral = ephemeral;
        this.owners = owners;
        this.waiters = waiters;
    }

    /**
     * Get name of the semaphore.
     * @return Name of the semaphore.
     */
    public String getName() {
        return name;
    }

    /**
     * Get user-defined data attached to the semaphore.
     * @return User-defined data attached to the semaphore.
     */
    public byte[] getData() {
        return data;
    }

    /**
     * Get number of tokens currently acquired by owners.
     * @return Number of tokens currently acquired by owners.
     */
    public long getCount() {
        return count;
    }

    /**
     * Get maximum number of tokens that may acquired.
     * @return Maximum number of tokens that may acquired.
     */
    public long getLimit() {
        return limit;
    }

    /**
     * Ephemeral semaphores are deleted when released by all owners and waiters.
     * @return Ephemeral flag.
     */
    public boolean isEphemeral() {
        return ephemeral;
    }

    /**
     * Get a list of current owners of the semaphore.
     * @return A list of current owners of the semaphore.
     */
    public List<CoordinationSemaphoreSession> getOwners() {
        return owners;
    }

    /**
     * Get a list of current waiters on the semaphore.
     * @return A list of current waiters on the semaphore.
     */
    public List<CoordinationSemaphoreSession> getWaiters() {
        return waiters;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        CoordinationSemaphoreDescription that = (CoordinationSemaphoreDescription) o;
        return count == that.count &&
                limit == that.limit &&
                ephemeral == that.ephemeral &&
                Objects.equals(name, that.name) &&
                Arrays.equals(data, that.data) &&
                Objects.equals(owners, that.owners) &&
                Objects.equals(waiters, that.waiters);
    }

    @Override
    public int hashCode() {
        int result = Objects.hash(name, count, limit, ephemeral, owners, waiters);
        result = 31 * result + Arrays.hashCode(data);
        return result;
    }

    @Override
    public String toString() {
        return "CoordinationSemaphoreDescription{" +
                "name='" + name + '\'' +
                ", data=" + Arrays.toString(data) +
                ", count=" + count +
                ", limit=" + limit +
                ", ephemeral=" + ephemeral +
                ", owners=" + owners +
                ", waiters=" + waiters +
                '}';
    }

}
