package ru.yandex.intranet.d.datasource.coordination.model.session;

import java.time.Duration;
import java.util.Arrays;
import java.util.Objects;

/**
 * Semaphore session. Describes an owner or a waiter of this semaphore.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class CoordinationSemaphoreSession {

    private final long orderId;
    private final long sessionId;
    private final Duration timeout;
    private final long count;
    private final byte[] data;

    public CoordinationSemaphoreSession(long orderId, long sessionId, Duration timeout, long count,
                                        byte[] data) {
        this.orderId = orderId;
        this.sessionId = sessionId;
        this.timeout = timeout;
        this.count = count;
        this.data = data;
    }

    /**
     * Get a monotonically increasing id which determines locking order.
     * @return A monotonically increasing id which determines locking order.
     */
    public long getOrderId() {
        return orderId;
    }

    /**
     * Get an id of the session which tried to acquire the semaphore.
     * @return An id of the session which tried to acquire the semaphore.
     */
    public long getSessionId() {
        return sessionId;
    }

    /**
     * Get a timeout for operation in waiters queue.
     * @return A timeout in milliseconds for operation in waiters queue.
     */
    public Duration getTimeout() {
        return timeout;
    }

    /**
     * Get number of tokens for an acquire operation.
     * @return Number of tokens for an acquire operation.
     */
    public long getCount() {
        return count;
    }

    /**
     * Get user-defined data attached to the acquire operation.
     * @return User-defined data attached to the acquire operation.
     */
    public byte[] getData() {
        return data;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        CoordinationSemaphoreSession that = (CoordinationSemaphoreSession) o;
        return orderId == that.orderId &&
                sessionId == that.sessionId &&
                count == that.count &&
                Objects.equals(timeout, that.timeout) &&
                Arrays.equals(data, that.data);
    }

    @Override
    public int hashCode() {
        int result = Objects.hash(orderId, sessionId, timeout, count);
        result = 31 * result + Arrays.hashCode(data);
        return result;
    }

    @Override
    public String toString() {
        return "CoordinationSemaphoreSession{" +
                "orderId=" + orderId +
                ", sessionId=" + sessionId +
                ", timeout=" + timeout +
                ", count=" + count +
                ", data=" + Arrays.toString(data) +
                '}';
    }

}
