package ru.yandex.intranet.d.datasource.coordination.model.session;

import java.util.Objects;

/**
 * Describe semaphore request. Used to describe semaphores and watch them for changes.
 * A describe operation will cancel previous watches on the same semaphore.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class DescribeSemaphoreRequest {

    private final long reqId;
    private final String name;
    private final boolean includeOwners;
    private final boolean includeWaiters;
    private final boolean watchData;
    private final boolean watchOwners;

    public DescribeSemaphoreRequest(long reqId, String name, boolean includeOwners, boolean includeWaiters,
                                    boolean watchData, boolean watchOwners) {
        this.reqId = reqId;
        this.name = name;
        this.includeOwners = includeOwners;
        this.includeWaiters = includeWaiters;
        this.watchData = watchData;
        this.watchOwners = watchOwners;
    }

    /**
     * Get client-defined request id, echoed in the response.
     * @return Client-defined request id.
     */
    public long getReqId() {
        return reqId;
    }

    /**
     * Get name of the semaphore to describe.
     * @return Name of the semaphore to describe.
     */
    public String getName() {
        return name;
    }

    /**
     * Response will include owners list if true.
     * @return Include owners flag.
     */
    public boolean isIncludeOwners() {
        return includeOwners;
    }

    /**
     * Response will include waiters list if true.
     * @return Include waiters flag.
     */
    public boolean isIncludeWaiters() {
        return includeWaiters;
    }

    /**
     * Watch for changes in semaphore data.
     * @return Watch data flag.
     */
    public boolean isWatchData() {
        return watchData;
    }

    /**
     * Watch for changes in semaphore owners (including owners data)
     * @return Watch owners flag.
     */
    public boolean isWatchOwners() {
        return watchOwners;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        DescribeSemaphoreRequest that = (DescribeSemaphoreRequest) o;
        return reqId == that.reqId &&
                includeOwners == that.includeOwners &&
                includeWaiters == that.includeWaiters &&
                watchData == that.watchData &&
                watchOwners == that.watchOwners &&
                Objects.equals(name, that.name);
    }

    @Override
    public int hashCode() {
        return Objects.hash(reqId, name, includeOwners, includeWaiters, watchData, watchOwners);
    }

    @Override
    public String toString() {
        return "DescribeSemaphoreRequest{" +
                "reqId=" + reqId +
                ", name='" + name + '\'' +
                ", includeOwners=" + includeOwners +
                ", includeWaiters=" + includeWaiters +
                ", watchData=" + watchData +
                ", watchOwners=" + watchOwners +
                '}';
    }

}
