package ru.yandex.intranet.d.datasource.coordination.model.session;

import java.time.Duration;
import java.util.Arrays;
import java.util.Objects;

/**
 * Session start request. First message used to start/restore a session.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class SessionStartRequest {

    private final String path;
    private final long sessionId;
    private final Duration timeout;
    private final String description;
    private final long seqNo;
    private final byte[] protectionKey;

    public SessionStartRequest(String path, long sessionId, Duration timeout, String description, long seqNo,
                               byte[] protectionKey) {
        this.path = path;
        this.sessionId = sessionId;
        this.timeout = timeout;
        this.description = description;
        this.seqNo = seqNo;
        this.protectionKey = protectionKey;
    }

    /**
     * Get a path to a coordination node.
     * @return Path to a coordination node.
     */
    public String getPath() {
        return path;
    }

    /**
     * Non-zero when restoring a session, 0 when creating a new session.
     * @return Session id.
     */
    public long getSessionId() {
        return sessionId;
    }

    /**
     * Get the timeout during which client may restore a detached session.
     * @return Timeout in milliseconds during which client may restore a detached session.
     */
    public Duration getTimeout() {
        return timeout;
    }

    /**
     * Get a user-defined description that may be used to describe the client.
     * @return User-defined description that may be used to describe the client.
     */
    public String getDescription() {
        return description;
    }

    /**
     * Monotonically increasing sequence number generated by the client. When concurrent SessionStart requests are
     * detected the one with the biggest sequence number will succeed.
     * @return Sequence number.
     */
    public long getSeqNo() {
        return seqNo;
    }

    /**
     * Random bytes used to protect session from restore by other clients (max. 16 bytes)
     * @return Protection key.
     */
    public byte[] getProtectionKey() {
        return protectionKey;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        SessionStartRequest that = (SessionStartRequest) o;
        return sessionId == that.sessionId &&
                seqNo == that.seqNo &&
                Objects.equals(path, that.path) &&
                Objects.equals(timeout, that.timeout) &&
                Objects.equals(description, that.description) &&
                Arrays.equals(protectionKey, that.protectionKey);
    }

    @Override
    public int hashCode() {
        int result = Objects.hash(path, sessionId, timeout, description, seqNo);
        result = 31 * result + Arrays.hashCode(protectionKey);
        return result;
    }

    @Override
    public String toString() {
        return "SessionStartRequest{" +
                "path='" + path + '\'' +
                ", sessionId=" + sessionId +
                ", timeout=" + timeout +
                ", description='" + description + '\'' +
                ", seqNo=" + seqNo +
                ", protectionKey=" + Arrays.toString(protectionKey) +
                '}';
    }

}
