package ru.yandex.intranet.d.datasource.impl;

import java.util.Arrays;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.TimeUnit;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Component;

/**
 * YDB query source.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class YdbQuerySource {
    private static final String INCLUDE_PREFIX = "--{";
    private static final String INCLUDE_POSTFIX = "}";

    private final String tablePathPrefix;
    private final String abcTablePathPrefix;
    private final LoadingCache<String, String> queryCache;
    private final LoadingCache<String, String> abcQueryCache;

    public YdbQuerySource(@Qualifier("yqlQueries") Properties properties,
                          @Value("${ydb.tablePathPrefix}") String tablePathPrefix,
                          @Value("${ydb.abcTablePathPrefix}") String abcTablePathPrefix) {
        this.tablePathPrefix = tablePathPrefix;
        this.abcTablePathPrefix = abcTablePathPrefix;
        this.queryCache = CacheBuilder.newBuilder()
                .maximumSize(10000)
                .expireAfterAccess(1, TimeUnit.HOURS)
                .build(new CacheLoader<>() {
                    @Override
                    public String load(@NonNull String key) {
                        return prepareQuery(key, tablePathPrefix, properties);
                    }
                });
        this.abcQueryCache = CacheBuilder.newBuilder()
                .maximumSize(10000)
                .expireAfterAccess(1, TimeUnit.HOURS)
                .build(new CacheLoader<>() {
                    @Override
                    public String load(@NonNull String key) {
                        return prepareQuery(key, abcTablePathPrefix, properties);
                    }
                });
    }

    public String getQuery(String key) {
        return queryCache.getUnchecked(key);
    }

    public String getAbcQuery(String key) {
        return abcQueryCache.getUnchecked(key);
    }

    public String preprocessRawQuery(String query) {
        return preprocessQuery(query, tablePathPrefix, null);
    }

    public String preprocessTableName(String tableName) {
        return tablePathPrefix + "/" + tableName;
    }

    public String preprocessAbcTableName(String tableName) {
        return abcTablePathPrefix + "/" + tableName;
    }

    private static String preprocessQuery(String text, String tablePathPrefix, String key) {
        String yqlVersion = "--!syntax_v1\n";
        String pragma = tablePathPrefix != null ? "PRAGMA TablePathPrefix = \"" + tablePathPrefix + "\";\n" : "";
        String debug = key != null ? "-- " + key + "\n" : "";
        return yqlVersion + pragma + debug + text;
    }

    private static Set<String> addKeyToPath(String key, Set<String> path) {
        if (path.contains(key)) {
            throw new IllegalStateException("Recursive includes found for key '" + key + "', path: " + path);
        }
        int pathSize = path.size();
        String[] newPath = new String[pathSize + 1];
        newPath = path.toArray(newPath);
        newPath[pathSize] = key;
        return Collections.unmodifiableSet(new LinkedHashSet<>(Arrays.asList(newPath)));
    }

    private static String getQueryWithIncludes(String key, Properties properties) {
        return getQueryWithIncludes(key, properties, Set.of());
    }

    private static String getQueryWithIncludes(String key, Properties properties, Set<String> path) {
        path = addKeyToPath(key, path);
        String query = (String) properties.get(key);
        if (query == null) {
            throw new IllegalArgumentException("Query key not found: '" + key + "'");
        }

        StringBuilder result = new StringBuilder();
        int lastIndex = 0;
        int indexOfPrefix;
        while ((indexOfPrefix = query.indexOf(INCLUDE_PREFIX, lastIndex)) >= 0) {
            int indexOfPostfix = query.indexOf(INCLUDE_POSTFIX, indexOfPrefix);
            if (indexOfPostfix < 0) {
                throw new IllegalArgumentException(
                        "Include, starting in position " + indexOfPrefix + ", is not closed");
            }
            result.append(query, lastIndex, indexOfPrefix);
            String includeKey = query.substring(indexOfPrefix + INCLUDE_PREFIX.length(), indexOfPostfix);
            String include = getQueryWithIncludes(includeKey, properties, path);
            result.append(include);

            lastIndex = indexOfPostfix + INCLUDE_POSTFIX.length();
        }
        result.append(query, lastIndex, query.length());

        return result.toString();
    }

    private static String prepareQuery(String key, String tablePathPrefix, Properties properties) {
        return preprocessQuery(getQueryWithIncludes(key, properties), tablePathPrefix, key);
    }

}
