package ru.yandex.intranet.d.datasource.impl;

import java.util.Set;
import java.util.concurrent.TimeoutException;

import com.yandex.ydb.core.StatusCode;
import com.yandex.ydb.core.UnexpectedResultException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.util.retry.Retry;
import ru.yandex.intranet.d.metrics.YdbMetrics;

import static com.yandex.ydb.core.StatusCode.ABORTED;
import static com.yandex.ydb.core.StatusCode.BAD_SESSION;
import static com.yandex.ydb.core.StatusCode.CLIENT_CANCELLED;
import static com.yandex.ydb.core.StatusCode.CLIENT_DEADLINE_EXCEEDED;
import static com.yandex.ydb.core.StatusCode.CLIENT_RESOURCE_EXHAUSTED;
import static com.yandex.ydb.core.StatusCode.OVERLOADED;
import static com.yandex.ydb.core.StatusCode.SESSION_BUSY;
import static com.yandex.ydb.core.StatusCode.SESSION_EXPIRED;
import static com.yandex.ydb.core.StatusCode.TIMEOUT;
import static com.yandex.ydb.core.StatusCode.TRANSPORT_UNAVAILABLE;
import static com.yandex.ydb.core.StatusCode.UNAVAILABLE;
import static com.yandex.ydb.core.StatusCode.UNDETERMINED;

/**
 * YDB retry policies.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class YdbRetry {

    private static final Logger LOG = LoggerFactory.getLogger(YdbRetry.class);

    private static final Set<StatusCode> REQUEST_RETRYABLE_CODES = Set.of(
            UNAVAILABLE,
            OVERLOADED,
            TRANSPORT_UNAVAILABLE,
            CLIENT_RESOURCE_EXHAUSTED
    );
    private static final Set<StatusCode> IDEMPOTENT_REQUEST_RETRYABLE_CODES = Set.of(
            UNAVAILABLE,
            OVERLOADED,
            TRANSPORT_UNAVAILABLE,
            CLIENT_RESOURCE_EXHAUSTED
    );
    private static final Set<StatusCode> IDEMPOTENT_TRANSACTION_RETRYABLE_CODES = Set.of(
            ABORTED,
            UNDETERMINED
    );
    private static final Set<StatusCode> IDEMPOTENT_SESSION_RETRYABLE_CODES = Set.of(
            SESSION_BUSY,
            SESSION_EXPIRED,
            BAD_SESSION,
            CLIENT_DEADLINE_EXCEEDED,
            CLIENT_CANCELLED,
            TIMEOUT
    );

    private YdbRetry() {
    }

    public static Retry retryRequest(long max, YdbMetrics ydbMetrics) {
        return Retry.max(max)
                .filter(e -> YdbRetry.isRequestRetryable(e, false))
                .onRetryExhaustedThrow((builder, rs) -> rs.failure())
                .doBeforeRetry(YdbRetry::logRequestRetry)
                .doAfterRetry(r -> ydbMetrics.afterRequestRetry());
    }

    public static Retry retryIdempotentRequest(long max, YdbMetrics ydbMetrics) {
        return Retry.max(max)
                .filter(e -> YdbRetry.isRequestRetryable(e, true))
                .onRetryExhaustedThrow((builder, rs) -> rs.failure())
                .doBeforeRetry(YdbRetry::logRequestRetry)
                .doAfterRetry(r -> ydbMetrics.afterRequestRetry());
    }

    public static Retry retryIdempotentSession(long max, YdbMetrics ydbMetrics) {
        return Retry.max(max)
                .filter(e -> YdbRetry.isSessionRetryable(e, true))
                .onRetryExhaustedThrow((builder, rs) -> rs.failure())
                .doBeforeRetry(YdbRetry::logSessionRetry)
                .doAfterRetry(r -> ydbMetrics.afterSessionRetry());
    }

    public static Retry retryTransaction(long max, YdbMetrics ydbMetrics) {
        return Retry.max(max)
                .filter(e -> YdbRetry.isTransactionRetryable(e, false))
                .onRetryExhaustedThrow((builder, rs) -> rs.failure())
                .doBeforeRetry(YdbRetry::logTransactionRetry)
                .doAfterRetry(r -> ydbMetrics.afterTransactionRetry());
    }

    public static Retry retryIdempotentTransaction(long max, YdbMetrics ydbMetrics) {
        return Retry.max(max)
                .filter(e -> YdbRetry.isTransactionRetryable(e, true))
                .onRetryExhaustedThrow((builder, rs) -> rs.failure())
                .doBeforeRetry(YdbRetry::logTransactionRetry)
                .doAfterRetry(r -> ydbMetrics.afterTransactionRetry());
    }

    public static Retry retryTimeout(long max, YdbMetrics ydbMetrics) {
        return Retry.max(max)
                .filter(YdbRetry::isTimeout)
                .onRetryExhaustedThrow((builder, rs) -> rs.failure())
                .doBeforeRetry(YdbRetry::logTimeoutRetry)
                .doAfterRetry(r -> ydbMetrics.afterTimeoutRetry());
    }

    private static boolean isRequestRetryable(Throwable e, boolean idempotentRequest) {
        if (!(e instanceof UnexpectedResultException unexpectedResultException)) {
            return false;
        }
        StatusCode statusCode = unexpectedResultException.getStatusCode();
        if (idempotentRequest) {
            return IDEMPOTENT_REQUEST_RETRYABLE_CODES.contains(statusCode);
        } else {
            return REQUEST_RETRYABLE_CODES.contains(statusCode);
        }
    }

    private static boolean isSessionRetryable(Throwable e, boolean idempotentSession) {
        if (idempotentSession && e instanceof TimeoutException) {
            return true;
        }
        if (!(e instanceof UnexpectedResultException unexpectedResultException)) {
            return false;
        }
        StatusCode statusCode = unexpectedResultException.getStatusCode();
        if (idempotentSession) {
            return IDEMPOTENT_SESSION_RETRYABLE_CODES.contains(statusCode);
        } else {
            return false;
        }
    }

    private static boolean isTransactionRetryable(Throwable e, boolean idempotentTransaction) {
        if (!(e instanceof UnexpectedResultException unexpectedResultException)) {
            return false;
        }
        StatusCode statusCode = unexpectedResultException.getStatusCode();
        if (idempotentTransaction) {
            return IDEMPOTENT_TRANSACTION_RETRYABLE_CODES.contains(statusCode)
                    || YdbStatusUtils.isTransactionNotFoundError(unexpectedResultException);
        } else {
            return statusCode == ABORTED;
        }
    }

    private static boolean isTimeout(Throwable e) {
        if (e instanceof TimeoutException) {
            return true;
        }
        if (!(e instanceof UnexpectedResultException unexpectedResultException)) {
            return false;
        }
        StatusCode statusCode = unexpectedResultException.getStatusCode();
        return statusCode == TIMEOUT;
    }

    private static void logRequestRetry(Retry.RetrySignal retrySignal) {
        LOG.info("Retry YDB request attempt #{}", retrySignal.totalRetries());
    }

    private static void logSessionRetry(Retry.RetrySignal retrySignal) {
        LOG.info("Retry YDB session attempt #{}", retrySignal.totalRetries());
    }

    private static void logTransactionRetry(Retry.RetrySignal retrySignal) {
        LOG.info("Retry YDB transaction attempt #{}", retrySignal.totalRetries());
    }

    private static void logTimeoutRetry(Retry.RetrySignal retrySignal) {
        LOG.info("Retry YDB timeout attempt #{}", retrySignal.totalRetries());
    }

}
