package ru.yandex.intranet.d.datasource.impl;

import java.time.Duration;

import com.yandex.ydb.table.SchemeClient;
import com.yandex.ydb.table.description.DescribePathResult;
import com.yandex.ydb.table.description.ListDirectoryResult;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.datasource.model.YdbSchemeClient;
import ru.yandex.intranet.d.metrics.YdbMetrics;
import ru.yandex.intranet.d.util.AsyncMetrics;

/**
 * YDB scheme client.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public class YdbSchemeClientImpl implements YdbSchemeClient {

    private final SchemeClient schemeClient;
    private final long queryRetries;
    private final long queryTimeoutMillis;
    private final YdbMetrics ydbMetrics;
    private final Duration queryClientTimeout;

    public YdbSchemeClientImpl(SchemeClient schemeClient, long queryRetries, long queryTimeoutMillis,
                               YdbMetrics ydbMetrics, Duration queryClientTimeout) {
        this.schemeClient = schemeClient;
        this.queryRetries = queryRetries;
        this.queryTimeoutMillis = queryTimeoutMillis;
        this.ydbMetrics = ydbMetrics;
        this.queryClientTimeout = queryClientTimeout;
    }

    @Override
    public void close() {
        schemeClient.close();
    }

    @Override
    public Mono<Void> makeDirectory(String path) {
        return AsyncMetrics.metric(appendRetry(appendTimeout(Mono.fromFuture(() -> schemeClient.makeDirectory(path))
                .doOnSuccess(s -> YdbStatusUtils.checkAnyStatus(s, "Failed to make directory")).then())),
                ydbMetrics::afterMakeDirectory);
    }

    @Override
    public Mono<Void> makeDirectories(String path) {
        return AsyncMetrics.metric(appendRetry(appendTimeout(Mono.fromFuture(() -> schemeClient.makeDirectories(path))
                .doOnSuccess(s -> YdbStatusUtils.checkAnyStatus(s, "Failed to make directories")).then())),
                ydbMetrics::afterMakeDirectories);
    }

    @Override
    public Mono<Void> removeDirectory(String path) {
        return AsyncMetrics.metric(appendRetry(appendTimeout(Mono.fromFuture(() -> schemeClient.removeDirectory(path))
                .doOnSuccess(s -> YdbStatusUtils.checkAnyStatus(s, "Failed to remove directory")).then())),
                ydbMetrics::afterRemoveDirectory);
    }

    @Override
    public Mono<DescribePathResult> describePath(String path) {
        return AsyncMetrics.metric(appendRetry(appendTimeout(Mono.fromFuture(() -> schemeClient.describePath(path))
                .map(r -> YdbStatusUtils.checkAnyResult(r, "Failed to describe path")))),
                ydbMetrics::afterDescribePath);
    }

    @Override
    public Mono<ListDirectoryResult> listDirectory(String path) {
        return AsyncMetrics.metric(appendRetry(appendTimeout(Mono.fromFuture(() -> schemeClient.listDirectory(path))
                .map(r -> YdbStatusUtils.checkAnyResult(r, "Failed to list directory")))),
                ydbMetrics::afterListDirectory);
    }

    private <T> Mono<T> appendRetry(Mono<T> publisher) {
        if (queryRetries > 0) {
            return publisher.retryWhen(YdbRetry.retryRequest(queryRetries, ydbMetrics));
        } else {
            return publisher;
        }
    }

    private <T> Mono<T> appendTimeout(Mono<T> publisher) {
        if (!queryClientTimeout.isNegative() && !queryClientTimeout.isZero()) {
            return publisher.timeout(queryClientTimeout);
        } else {
            return publisher;
        }
    }

}
