package ru.yandex.intranet.d.datasource.impl;

import java.time.Duration;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.function.BiFunction;
import java.util.function.Function;

import com.yandex.ydb.table.Session;
import com.yandex.ydb.table.SessionStatus;
import com.yandex.ydb.table.YdbTable;
import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.query.DataQueryResult;
import com.yandex.ydb.table.query.ExplainDataQueryResult;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.settings.AlterTableSettings;
import com.yandex.ydb.table.settings.BeginTxSettings;
import com.yandex.ydb.table.settings.CloseSessionSettings;
import com.yandex.ydb.table.settings.CommitTxSettings;
import com.yandex.ydb.table.settings.CopyTableSettings;
import com.yandex.ydb.table.settings.CreateTableSettings;
import com.yandex.ydb.table.settings.DescribeTableSettings;
import com.yandex.ydb.table.settings.DropTableSettings;
import com.yandex.ydb.table.settings.ExecuteDataQuerySettings;
import com.yandex.ydb.table.settings.ExecuteScanQuerySettings;
import com.yandex.ydb.table.settings.ExecuteSchemeQuerySettings;
import com.yandex.ydb.table.settings.ExplainDataQuerySettings;
import com.yandex.ydb.table.settings.KeepAliveSessionSettings;
import com.yandex.ydb.table.settings.PrepareDataQuerySettings;
import com.yandex.ydb.table.settings.ReadTableSettings;
import com.yandex.ydb.table.settings.RollbackTxSettings;
import com.yandex.ydb.table.transaction.Transaction;
import com.yandex.ydb.table.transaction.TransactionMode;
import com.yandex.ydb.table.transaction.TxControl;
import org.reactivestreams.Publisher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;

import ru.yandex.intranet.d.datasource.model.WithTxId;
import ru.yandex.intranet.d.datasource.model.YdbAlterTableSettings;
import ru.yandex.intranet.d.datasource.model.YdbCreateTableSettings;
import ru.yandex.intranet.d.datasource.model.YdbDataQuery;
import ru.yandex.intranet.d.datasource.model.YdbExecuteDataQuerySettings;
import ru.yandex.intranet.d.datasource.model.YdbExecuteScanQuerySettings;
import ru.yandex.intranet.d.datasource.model.YdbPrepareDataQuerySettings;
import ru.yandex.intranet.d.datasource.model.YdbQueryStatsCollectionMode;
import ru.yandex.intranet.d.datasource.model.YdbReadTableSettings;
import ru.yandex.intranet.d.datasource.model.YdbScanQueryRequestMode;
import ru.yandex.intranet.d.datasource.model.YdbSession;
import ru.yandex.intranet.d.datasource.model.YdbTransaction;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.metrics.YdbMetrics;
import ru.yandex.intranet.d.util.AsyncMetrics;
import ru.yandex.intranet.d.util.MdcTaskDecorator;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.result.ResultTx;

/**
 * YDB session.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public class YdbSessionImpl implements YdbSession {

    private static final Logger LOG = LoggerFactory.getLogger(YdbSessionImpl.class);

    private final Session session;
    private final long queryTimeoutMillis;
    private final long queryRetries;
    private final long transactionRetries;
    private final YdbMetrics ydbMetrics;
    private final Duration queryClientTimeout;

    public YdbSessionImpl(Session session, long queryTimeoutMillis, long queryRetries, long transactionRetries,
                          YdbMetrics ydbMetrics, Duration queryClientTimeout) {
        this.session = session;
        this.queryTimeoutMillis = queryTimeoutMillis;
        this.queryRetries = queryRetries;
        this.transactionRetries = transactionRetries;
        this.ydbMetrics = ydbMetrics;
        this.queryClientTimeout = queryClientTimeout;
    }

    @Override
    public String getId() {
        return session.getId();
    }

    @Override
    public Mono<Void> createTable(String path, TableDescription tableDescriptions, YdbCreateTableSettings settings) {
        return AsyncMetrics.metric(appendRetry(appendTimeout(Mono.fromFuture(() -> session
                .createTable(path, tableDescriptions, prepareCreateTableSettings(settings)))
                .doOnSuccess(s -> YdbStatusUtils.checkAnyStatus(s, "Failed to create YDB table")).then())),
                ydbMetrics::afterCreateTable);
    }

    @Override
    public Mono<Void> createTable(String path, TableDescription tableDescriptions) {
        return AsyncMetrics.metric(appendRetry(appendTimeout(Mono.fromFuture(() -> session
                .createTable(path, tableDescriptions, prepareCreateTableSettings()))
                .doOnSuccess(s -> YdbStatusUtils.checkAnyStatus(s, "Failed to create YDB table")).then())),
                ydbMetrics::afterCreateTable);
    }

    @Override
    public Mono<Void> dropTable(String path) {
        return AsyncMetrics.metric(appendRetry(appendTimeout(Mono.fromFuture(() -> session
                .dropTable(path, prepareDropTableSettings()))
                .doOnSuccess(s -> YdbStatusUtils.checkAnyStatus(s, "Failed to drop YDB table")).then())),
                ydbMetrics::afterDropTable);
    }

    @Override
    public Mono<Void> alterTable(String path, YdbAlterTableSettings settings) {
        return AsyncMetrics.metric(appendRetry(appendTimeout(Mono.fromFuture(() -> session
                .alterTable(path, prepareAlterTableSettings(settings)))
                .doOnSuccess(s -> YdbStatusUtils.checkAnyStatus(s, "Failed to alter YDB table")).then())),
                ydbMetrics::afterAlterTable);
    }

    @Override
    public Mono<Void> alterTable(String path) {
        return AsyncMetrics.metric(appendRetry(appendTimeout(Mono.fromFuture(() -> session
                .alterTable(path, prepareAlterTableSettings()))
                .doOnSuccess(s -> YdbStatusUtils.checkAnyStatus(s, "Failed to alter YDB table")).then())),
                ydbMetrics::afterAlterTable);
    }

    @Override
    public Mono<Void> copyTable(String src, String dst) {
        return AsyncMetrics.metric(appendRetry(appendTimeout(Mono.fromFuture(() -> session
                .copyTable(src, dst, prepareCopyTableSettings()))
                .doOnSuccess(s -> YdbStatusUtils.checkAnyStatus(s, "Failed to copy YDB table")).then())),
                ydbMetrics::afterCopyTable);
    }

    @Override
    public Mono<TableDescription> describeTable(String path) {
        return AsyncMetrics.metric(appendIdempotentRetry(appendTimeout(Mono.fromFuture(() -> session
                .describeTable(path, prepareDescribeTableSettings()))
                .map(s -> YdbStatusUtils.checkResult(s, "Failed to describe table")))),
                ydbMetrics::afterDescribeTable);
    }

    @Override
    public Mono<DataQueryResult> executeDataQuery(String query, TxControl txControl, Params params,
                                                  YdbExecuteDataQuerySettings settings) {
        return AsyncMetrics.metric(appendRetry(appendTimeout(Mono.fromFuture(() -> session
                .executeDataQuery(query, txControl, params, prepareExecuteDataQuerySettings(settings)))
                .map(s -> YdbStatusUtils.checkResult(s, "Failed to execute YDB data query")))),
                ydbMetrics::afterExecuteDataQuery);
    }

    @Override
    public Mono<DataQueryResult> executeDataQuery(String query, TxControl txControl, Params params) {
        return AsyncMetrics.metric(appendRetry(appendTimeout(Mono.fromFuture(() -> session
                .executeDataQuery(query, txControl, params, prepareExecuteDataQuerySettings()))
                .map(s -> YdbStatusUtils.checkResult(s, "Failed to execute YDB data query")))),
                ydbMetrics::afterExecuteDataQuery);
    }

    @Override
    public Mono<DataQueryResult> executeDataQuery(String query, TxControl txControl) {
        return AsyncMetrics.metric(appendRetry(appendTimeout(Mono.fromFuture(() -> session
                .executeDataQuery(query, txControl, Params.empty(), prepareExecuteDataQuerySettings()))
                .map(s -> YdbStatusUtils.checkResult(s, "Failed to execute YDB data query")))),
                ydbMetrics::afterExecuteDataQuery);
    }

    @Override
    public Mono<DataQueryResult> executeDataQueryRetryable(String query, TxControl txControl, Params params,
                                                           YdbExecuteDataQuerySettings settings) {
        return AsyncMetrics.metric(appendIdempotentRetry(appendTimeout(Mono.fromFuture(() -> session
                                .executeDataQuery(query, txControl, params, prepareExecuteDataQuerySettings(settings)))
                        .map(s -> YdbStatusUtils.checkResult(s, "Failed to execute YDB data query")))),
                ydbMetrics::afterExecuteDataQuery);
    }

    @Override
    public Mono<DataQueryResult> executeDataQueryRetryable(String query, TxControl txControl, Params params) {
        return AsyncMetrics.metric(appendIdempotentRetry(appendTimeout(Mono.fromFuture(() -> session
                                .executeDataQuery(query, txControl, params, prepareExecuteDataQuerySettings()))
                        .map(s -> YdbStatusUtils.checkResult(s, "Failed to execute YDB data query")))),
                ydbMetrics::afterExecuteDataQuery);
    }

    @Override
    public Mono<DataQueryResult> executeDataQueryRetryable(String query, TxControl txControl) {
        return AsyncMetrics.metric(appendIdempotentRetry(appendTimeout(Mono.fromFuture(() -> session
                                .executeDataQuery(query, txControl, Params.empty(), prepareExecuteDataQuerySettings()))
                        .map(s -> YdbStatusUtils.checkResult(s, "Failed to execute YDB data query")))),
                ydbMetrics::afterExecuteDataQuery);
    }

    @Override
    public Mono<DataQueryResult> executeDataQueryCommit(String query, TransactionMode txMode, Params params,
                                                        YdbExecuteDataQuerySettings settings) {
        TxControl txControl = txModeToTxControl(txMode, true);
        return AsyncMetrics.metric(appendTxRetry(txControl, appendRetry(appendTimeout(Mono.fromFuture(() -> session
                .executeDataQuery(query, txControl, params,
                        prepareExecuteDataQuerySettings(settings)))
                .map(s -> YdbStatusUtils.checkResult(s, "Failed to execute YDB data query"))))),
                ydbMetrics::afterExecuteDataQuery);
    }

    @Override
    public Mono<DataQueryResult> executeDataQueryCommit(String query, TransactionMode txMode, Params params) {
        TxControl txControl = txModeToTxControl(txMode, true);
        return AsyncMetrics.metric(appendTxRetry(txControl, appendRetry(appendTimeout(Mono.fromFuture(() -> session
                .executeDataQuery(query, txControl, params,
                        prepareExecuteDataQuerySettings()))
                .map(s -> YdbStatusUtils.checkResult(s, "Failed to execute YDB data query"))))),
                ydbMetrics::afterExecuteDataQuery);
    }

    @Override
    public Mono<DataQueryResult> executeDataQueryCommit(String query, TransactionMode txMode) {
        TxControl txControl = txModeToTxControl(txMode, true);
        return AsyncMetrics.metric(appendTxRetry(txControl, appendRetry(appendTimeout(Mono.fromFuture(() -> session
                .executeDataQuery(query, txControl, Params.empty(),
                        prepareExecuteDataQuerySettings()))
                .map(s -> YdbStatusUtils.checkResult(s, "Failed to execute YDB data query"))))),
                ydbMetrics::afterExecuteDataQuery);
    }

    @Override
    public Mono<DataQueryResult> executeDataQueryCommitRetryable(
            String query, TransactionMode txMode, Params params, YdbExecuteDataQuerySettings settings) {
        TxControl txControl = txModeToTxControl(txMode, true);
        return AsyncMetrics.metric(appendIdempotentTxRetry(txControl, appendIdempotentRetry(appendTimeout(
                Mono.fromFuture(() -> session.executeDataQuery(query, txControl, params,
                        prepareExecuteDataQuerySettings(settings)))
                .map(s -> YdbStatusUtils.checkResult(s, "Failed to execute YDB data query"))))),
                ydbMetrics::afterExecuteDataQuery);
    }

    @Override
    public Mono<DataQueryResult> executeDataQueryCommitRetryable(String query, TransactionMode txMode,
                                                                 Params params) {
        TxControl txControl = txModeToTxControl(txMode, true);
        return AsyncMetrics.metric(appendIdempotentTxRetry(txControl, appendIdempotentRetry(appendTimeout(
                Mono.fromFuture(() -> session.executeDataQuery(query, txControl, params,
                        prepareExecuteDataQuerySettings()))
                .map(s -> YdbStatusUtils.checkResult(s, "Failed to execute YDB data query"))))),
                ydbMetrics::afterExecuteDataQuery);
    }

    @Override
    public Mono<DataQueryResult> executeDataQueryCommitRetryable(String query, TransactionMode txMode) {
        TxControl txControl = txModeToTxControl(txMode, true);
        return AsyncMetrics.metric(appendIdempotentTxRetry(txControl, appendIdempotentRetry(appendTimeout(
                Mono.fromFuture(() -> session.executeDataQuery(query, txControl, Params.empty(),
                        prepareExecuteDataQuerySettings()))
                .map(s -> YdbStatusUtils.checkResult(s, "Failed to execute YDB data query"))))),
                ydbMetrics::afterExecuteDataQuery);
    }

    @Override
    public Mono<YdbDataQuery> prepareDataQuery(String query, YdbPrepareDataQuerySettings settings) {
        return AsyncMetrics.metric(appendIdempotentRetry(appendTimeout(Mono.fromFuture(() -> session
                .prepareDataQuery(query, preparePrepareDataQuerySettings(settings)))
                .map(s -> YdbStatusUtils.checkAnyResult(s.map(dataQuery -> new YdbDataQueryImpl(dataQuery,
                        queryTimeoutMillis, queryRetries, transactionRetries, ydbMetrics, queryClientTimeout)),
                        "Failed to prepare YDB data query")))),
                ydbMetrics::afterPrepareDataQuery);
    }

    @Override
    public Mono<YdbDataQuery> prepareDataQuery(String query) {
        return AsyncMetrics.metric(appendIdempotentRetry(appendTimeout(Mono.fromFuture(() -> session
                .prepareDataQuery(query, preparePrepareDataQuerySettings()))
                .map(s -> YdbStatusUtils.checkAnyResult(s.map(dataQuery -> new YdbDataQueryImpl(dataQuery,
                        queryTimeoutMillis, queryRetries, transactionRetries, ydbMetrics, queryClientTimeout)),
                        "Failed to prepare YDB data query")))),
                ydbMetrics::afterPrepareDataQuery);
    }

    @Override
    public Mono<Void> executeSchemeQuery(String query) {
        return AsyncMetrics.metric(appendRetry(appendTimeout(Mono.fromFuture(() -> session
                .executeSchemeQuery(query, prepareExecuteSchemeQuerySettings()))
                .doOnSuccess(s -> YdbStatusUtils.checkAnyStatus(s, "Failed to execute YDB scheme query")).then())),
                ydbMetrics::afterExecuteSchemeQuery);
    }

    @Override
    public Mono<ExplainDataQueryResult> explainDataQuery(String query) {
        return AsyncMetrics.metric(appendIdempotentRetry(appendTimeout(Mono.fromFuture(() -> session
                .explainDataQuery(query, prepareExplainDataQuerySettings()))
                .map(r -> YdbStatusUtils.checkAnyResult(r, "Failed to explain YDB data query")))),
                ydbMetrics::afterExplainDataQuery);
    }

    @Override
    public Mono<YdbTransaction> beginTransaction(TransactionMode transactionMode) {
        return AsyncMetrics.metric(appendRetry(appendTimeout(Mono.fromFuture(() -> session
                .beginTransaction(transactionMode, prepareBeginTxSettings()))
                .doOnDiscard(com.yandex.ydb.core.Result.class, this::onDiscardedTransaction)
                .map(r -> YdbStatusUtils.checkAnyResult(r.map(transaction -> new YdbTransactionImpl(transaction,
                        queryTimeoutMillis, queryRetries, ydbMetrics, queryClientTimeout)),
                        "Failed to begin YDB transaction")))),
                ydbMetrics::afterBeginTransaction);
    }

    @Override
    public Mono<Void> commitTransaction(String txId) {
        return AsyncMetrics.metric(appendRetry(appendTimeout(Mono.fromFuture(() -> session
                .commitTransaction(txId, prepareCommitTxSettings()))
                .doOnSuccess(r -> YdbStatusUtils.checkStatus(r, "Failed to commit YDB transaction")).then())),
                ydbMetrics::afterCommitTransaction);
    }

    @Override
    public Mono<Void> rollbackTransaction(String txId) {
        return AsyncMetrics.metric(appendRetry(appendTimeout(Mono.fromFuture(() -> session
                .rollbackTransaction(txId, prepareRollbackTxSettings()))
                .doOnSuccess(r -> YdbStatusUtils.checkAnyStatus(r, "Failed to rollback YDB transaction")).then())),
                ydbMetrics::afterRollbackTransaction);
    }

    @Override
    @SuppressWarnings("FutureReturnValueIgnored")
    public Flux<ResultSetReader> readTable(String tablePath, YdbReadTableSettings settings) {
        return AsyncMetrics.metric(appendIdempotentRetry(appendTimeout(settings, Flux.defer(() ->
                Flux.create(sink -> session.readTable(tablePath, prepareReadTableSettings(settings), sink::next)
                .whenComplete((status, ex) -> {
                    if (ex != null) {
                        sink.error(YdbStatusUtils.checkException(ex));
                    } else {
                        if (status.isSuccess()) {
                            sink.complete();
                        } else {
                            sink.error(YdbStatusUtils.checkAnyStatusReturn(status, "Failed to read YDB table"));
                        }
                    }
                }))))), ydbMetrics::afterReadTable);
    }

    @Override
    @SuppressWarnings("FutureReturnValueIgnored")
    public Flux<ResultSetReader> executeScanQuery(String query, Params params, YdbExecuteScanQuerySettings settings) {
        return AsyncMetrics.metric(appendIdempotentRetry(appendTimeout(settings, Flux.defer(() -> Flux.create(sink ->
                session.executeScanQuery(query, params, prepareExecuteScanQuerySettings(settings), sink::next)
                .whenComplete((status, ex) -> {
                    if (ex != null) {
                        sink.error(YdbStatusUtils.checkException(ex));
                    } else {
                        if (status.isSuccess()) {
                            sink.complete();
                        } else {
                            sink.error(YdbStatusUtils.checkAnyStatusReturn(status, "Failed to execute YDB scan query"));
                        }
                    }
                }))))), ydbMetrics::afterExecuteScanQuery);
    }

    @Override
    public Flux<ResultSetReader> executeScanQuery(String query, Params params) {
        return executeScanQuery(query, params, YdbExecuteScanQuerySettings.builder()
                .requestMode(YdbScanQueryRequestMode.EXEC)
                .statsMode(YdbQueryStatsCollectionMode.NONE)
                .build());
    }

    @Override
    public Mono<SessionStatus> keepAlive() {
        return AsyncMetrics.metric(appendIdempotentRetry(appendTimeout(Mono
                .fromFuture(() -> session.keepAlive(prepareKeepAliveSessionSettings()))
                .map(r -> YdbStatusUtils.checkAnyResult(r, "Failed YDB session keep-alive")))),
                ydbMetrics::afterSessionKeepAlive);
    }

    @Override
    public void invalidateQueryCache() {
        session.invalidateQueryCache();
    }

    @Override
    public boolean release() {
        return session.release();
    }

    @Override
    public Mono<Void> close() {
        return AsyncMetrics.metric(appendRetry(appendTimeout(Mono
                .fromFuture(() -> session.close(prepareCloseSessionSettings()))
                .doOnSuccess(s -> YdbStatusUtils.checkAnyStatus(s, "Failed to close YDB session")).then())),
                ydbMetrics::afterCloseSession);
    }

    @Override
    public <T> Mono<T> usingTxMono(TransactionMode transactionMode,
                                   Function<YdbTxSession, Mono<T>> body) {
        Mono<T> result = Mono.usingWhen(beginTransactionImpl(transactionMode),
                t -> body.apply(initTxSession(TxControl.id(t).setCommitTx(false), false)), this::commitImpl,
                (t, e) -> rollbackImpl(t), this::rollbackImpl);
        if (transactionRetries > 0) {
            return result
                    .retryWhen(YdbRetry.retryTransaction(transactionRetries, ydbMetrics));
        } else {
            return result;
        }
    }

    @Override
    public <T> Flux<T> usingTxFlux(TransactionMode transactionMode,
                                   Function<YdbTxSession, Flux<T>> body) {
        Flux<T> result = Flux.usingWhen(beginTransactionImpl(transactionMode),
                t -> body.apply(initTxSession(TxControl.id(t).setCommitTx(false), false)), this::commitImpl,
                (t, e) -> rollbackImpl(t), this::rollbackImpl);
        if (transactionRetries > 0) {
            return result
                    .retryWhen(YdbRetry.retryTransaction(transactionRetries, ydbMetrics));
        } else {
            return result;
        }
    }

    @Override
    public <T> Mono<T> usingTxMonoRetryable(TransactionMode transactionMode,
                                            Function<YdbTxSession, Mono<T>> body) {
        Mono<T> result = Mono.usingWhen(beginTransactionImpl(transactionMode),
                t -> body.apply(initTxSession(TxControl.id(t).setCommitTx(false), false)), this::commitImpl,
                (t, e) -> rollbackImpl(t), this::rollbackImpl);
        if (transactionRetries > 0) {
            return result
                    .retryWhen(YdbRetry.retryIdempotentTransaction(transactionRetries, ydbMetrics));
        } else {
            return result;
        }
    }

    @Override
    public <T> Flux<T> usingTxFluxRetryable(TransactionMode transactionMode,
                                            Function<YdbTxSession, Flux<T>> body) {
        Flux<T> result = Flux.usingWhen(beginTransactionImpl(transactionMode),
                t -> body.apply(initTxSession(TxControl.id(t).setCommitTx(false), false)), this::commitImpl,
                (t, e) -> rollbackImpl(t), this::rollbackImpl);
        if (transactionRetries > 0) {
            return result
                    .retryWhen(YdbRetry.retryIdempotentTransaction(transactionRetries, ydbMetrics));
        } else {
            return result;
        }
    }

    @Override
    public YdbTxSession asTx(TxControl txControl) {
        return new YdbTxSessionImpl(session, txControl, queryTimeoutMillis, queryRetries, transactionRetries, false,
                ydbMetrics, queryClientTimeout);
    }

    @Override
    public YdbTxSession asTxCommit(TransactionMode txMode) {
        TxControl txControl = txModeToTxControl(txMode, true);
        return new YdbTxSessionImpl(session, txControl, queryTimeoutMillis, queryRetries, transactionRetries, false,
                ydbMetrics, queryClientTimeout);
    }

    @Override
    public YdbTxSession asTxCommitRetryable(TransactionMode txMode) {
        TxControl txControl = txModeToTxControl(txMode, true);
        return new YdbTxSessionImpl(session, txControl, queryTimeoutMillis, queryRetries, transactionRetries, true,
                ydbMetrics, queryClientTimeout);
    }

    @Override
    public <U, V, W> Mono<W> usingCompTxMono(TransactionMode txMode,
                                             Function<YdbTxSession, Mono<Tuple2<U, String>>> preamble,
                                             BiFunction<YdbTxSession, U, Mono<V>> body,
                                             BiFunction<YdbTxSession, V, Mono<W>> trailer) {
        Mono<W> result = Mono.usingWhen(Mono.defer(() -> preamble.apply(initTxSession(txMode, false, false))),
                t -> body.apply(initTxSessionBody(t.getT2(), false), t.getT1())
                        .flatMap(v -> trailer.apply(initTxSessionTrailer(t.getT2(), false), v)),
                t -> Mono.empty(), (t, e) -> rollbackImpl(t.getT2()), t -> rollbackImpl(t.getT2()));
        if (transactionRetries > 0) {
            return result
                    .retryWhen(YdbRetry.retryTransaction(transactionRetries, ydbMetrics));
        } else {
            return result;
        }
    }

    @Override
    public <U, V, W> Flux<W> usingCompTxFlux(TransactionMode txMode,
                                             Function<YdbTxSession, Flux<Tuple2<U, String>>> preamble,
                                             BiFunction<YdbTxSession, U, Flux<V>> body,
                                             BiFunction<YdbTxSession, V, Flux<W>> trailer) {
        Flux<W> result = Flux.usingWhen(Flux.defer(() -> preamble.apply(initTxSession(txMode, false, false))),
                t -> body.apply(initTxSessionBody(t.getT2(), false), t.getT1())
                        .flatMap(v -> trailer.apply(initTxSessionTrailer(t.getT2(), false), v)),
                t -> Mono.empty(), (t, e) -> rollbackImpl(t.getT2()), t -> rollbackImpl(t.getT2()));
        if (transactionRetries > 0) {
            return result
                    .retryWhen(YdbRetry.retryTransaction(transactionRetries, ydbMetrics));
        } else {
            return result;
        }
    }

    @Override
    public <U, V, W> Mono<W> usingCompTxMonoRetryable(TransactionMode txMode,
                                                      Function<YdbTxSession, Mono<Tuple2<U, String>>> preamble,
                                                      BiFunction<YdbTxSession, U, Mono<V>> body,
                                                      BiFunction<YdbTxSession, V, Mono<W>> trailer) {
        Mono<W> result = Mono.usingWhen(Mono.defer(() -> preamble.apply(initTxSession(txMode, false, false))),
                t -> body.apply(initTxSessionBody(t.getT2(), false), t.getT1())
                        .flatMap(v -> trailer.apply(initTxSessionTrailer(t.getT2(), false), v)),
                t -> Mono.empty(), (t, e) -> rollbackImpl(t.getT2()), t -> rollbackImpl(t.getT2()));
        if (transactionRetries > 0) {
            return result
                    .retryWhen(YdbRetry.retryIdempotentTransaction(transactionRetries, ydbMetrics));
        } else {
            return result;
        }
    }

    @Override
    public <U, V, W> Mono<W> usingCompTxRetryable(
            Function<YdbTxSession, Mono<WithTxId<U>>> preamble,
            BiFunction<YdbTxSession, U, Mono<V>> body,
            BiFunction<YdbTxSession, V, Mono<W>> trailer
    ) {
        Mono<W> result = Mono.usingWhen(
                Mono.defer(() -> preamble.apply(initTxSession(TransactionMode.SERIALIZABLE_READ_WRITE, false, false))),
                t -> body.apply(initTxSessionBody(t.getTransactionId(), false), t.get())
                        .flatMap(v -> trailer.apply(initTxSessionTrailer(t.getTransactionId(), false), v)),
                t -> Mono.empty(),
                (t, e) -> rollbackImpl(t.getTransactionId()),
                t -> rollbackImpl(t.getTransactionId())
        );
        if (transactionRetries > 0) {
            return result
                    .retryWhen(YdbRetry.retryIdempotentTransaction(transactionRetries, ydbMetrics));
        } else {
            return result;
        }
    }

    @Override
    public <U, V, W> Mono<Result<W>> usingCompResultTxRetryable(
            TransactionMode txMode,
            Function<YdbTxSession, Mono<ResultTx<U>>> preamble,
            BiFunction<YdbTxSession, U, Mono<WithTxId<V>>> body,
            BiFunction<YdbTxSession, V, Mono<WithTxId<W>>> trailer
    ) {
        Publisher<ResultTx<U>> defer = Mono.defer(() -> preamble.apply(this.initTxSession(txMode, false, false)));
        Mono<Result<W>> result = Mono.<Result<W>, ResultTx<U>>usingWhen(
                defer,
                r -> r.applyMono(
                        t -> body
                                .apply(this.initTxSessionBody(r.getTransactionId(), false), t)
                                .flatMap(v -> trailer
                                        .apply(this.initTxSessionTrailer(r.getTransactionId(), false), v.get())
                                        .map(WithTxId::getClosed)
                                ),
                        errors -> (r.getTransactionId() == null ? Mono.empty() :
                                this.rollbackTransaction(r.getTransactionId()))
                                .thenReturn(ResultTx.failure(errors, null))
                ).map(ResultTx::toResult),
                t -> Mono.empty(),
                (t, e) -> this.rollbackImpl(t.getTransactionId()),
                t -> this.rollbackImpl(t.getTransactionId())
        );
        if (transactionRetries > 0) {
            return result
                    .retryWhen(YdbRetry.retryIdempotentTransaction(transactionRetries, ydbMetrics));
        } else {
            return result;
        }
    }

    @Override
    public <U, V, W> Flux<W> usingCompTxFluxRetryable(TransactionMode txMode,
                                                      Function<YdbTxSession, Flux<Tuple2<U, String>>> preamble,
                                                      BiFunction<YdbTxSession, U, Flux<V>> body,
                                                      BiFunction<YdbTxSession, V, Flux<W>> trailer) {
        Flux<W> result = Flux.usingWhen(Flux.defer(() -> preamble.apply(initTxSession(txMode, false, false))),
                t -> body.apply(initTxSessionBody(t.getT2(), false), t.getT1())
                        .flatMap(v -> trailer.apply(initTxSessionTrailer(t.getT2(), false), v)),
                t -> Mono.empty(), (t, e) -> rollbackImpl(t.getT2()), t -> rollbackImpl(t.getT2()));
        if (transactionRetries > 0) {
            return result
                    .retryWhen(YdbRetry.retryIdempotentTransaction(transactionRetries, ydbMetrics));
        } else {
            return result;
        }
    }

    @Override
    public String toString() {
        return session.toString();
    }

    private TxControl txModeToTxControl(TransactionMode txMode, boolean commit) {
        switch (txMode) {
            case SERIALIZABLE_READ_WRITE:
                return TxControl.serializableRw().setCommitTx(commit);
            case ONLINE_READ_ONLY:
                return TxControl.onlineRo().setCommitTx(commit);
            case STALE_READ_ONLY:
                return TxControl.staleRo().setCommitTx(commit);
            default:
                throw new IllegalArgumentException("Unexpected transaction mode: " + txMode);
        }
    }

    private YdbTxSession initTxSession(TransactionMode txMode, boolean commit, boolean retryCommit) {
        return initTxSession(txModeToTxControl(txMode, commit), retryCommit);
    }

    private YdbTxSession initTxSession(TxControl txControl, boolean retryCommit) {
        return new YdbTxSessionImpl(session, txControl, queryTimeoutMillis, queryRetries, transactionRetries,
                retryCommit, ydbMetrics, queryClientTimeout);
    }

    private YdbTxSession initTxSessionBody(String txId, boolean retryCommit) {
        return initTxSession(TxControl.id(txId).setCommitTx(false), retryCommit);
    }

    private YdbTxSession initTxSessionTrailer(String txId, boolean retryCommit) {
        return initTxSession(TxControl.id(txId).setCommitTx(true), retryCommit);
    }

    private Mono<Transaction> beginTransactionImpl(TransactionMode transactionMode) {
        return appendRetry(appendTimeout(Mono.fromFuture(() -> session.beginTransaction(transactionMode))
                .doOnDiscard(com.yandex.ydb.core.Result.class, this::onDiscardedTransaction)
                .map(r -> YdbStatusUtils.checkAnyResult(r, "Failed to begin YDB transaction"))));
    }

    private Mono<Void> commitImpl(Transaction transaction) {
        return appendRetry(appendTimeout(Mono.fromFuture(transaction::commit)
                .doOnSuccess(s -> YdbStatusUtils.checkStatus(s, "Failed to commit YDB transaction")).then()));
    }

    private Mono<Void> rollbackImpl(Transaction transaction) {
        return appendRetryTimeout(appendRetry(appendTimeout(Mono.fromFuture(transaction::rollback)
                .doOnSuccess(s -> YdbStatusUtils.checkAnyStatus(s, "Failed to rollback YDB transaction")))))
                .doOnError(e -> LOG.warn("Failed to rollback YDB transaction", e))
                .then()
                .onErrorResume(e -> Mono.empty());
    }

    private Mono<Void> rollbackImpl(String txId) {
        return appendRetryTimeout(appendRetry(appendTimeout(Mono.fromFuture(() -> session
                .rollbackTransaction(txId, prepareRollbackTxSettings()))
                .doOnSuccess(s -> YdbStatusUtils.checkAnyStatus(s, "Failed to rollback YDB transaction")))))
                .doOnError(e -> LOG.warn("Failed to rollback YDB transaction", e))
                .then()
                .onErrorResume(e -> Mono.empty());
    }

    private <T> Mono<T> appendRetry(Mono<T> publisher) {
        if (queryRetries > 0) {
            return publisher.retryWhen(YdbRetry.retryRequest(queryRetries, ydbMetrics));
        } else {
            return publisher;
        }
    }

    private <T> Mono<T> appendIdempotentRetry(Mono<T> publisher) {
        if (queryRetries > 0) {
            return publisher.retryWhen(YdbRetry.retryIdempotentRequest(queryRetries, ydbMetrics));
        } else {
            return publisher;
        }
    }

    private <T> Mono<T> appendIdempotentTxRetry(TxControl txControl, Mono<T> publisher) {
        if (transactionRetries > 0 && txControl.isCommitTx() && txControl.toPb().hasBeginTx()) {
            return publisher.retryWhen(YdbRetry.retryIdempotentTransaction(transactionRetries, ydbMetrics));
        } else {
            return publisher;
        }
    }

    private <T> Mono<T> appendTxRetry(TxControl txControl, Mono<T> publisher) {
        if (transactionRetries > 0 && txControl.isCommitTx() && txControl.toPb().hasBeginTx()) {
            return publisher.retryWhen(YdbRetry.retryTransaction(transactionRetries, ydbMetrics));
        } else {
            return publisher;
        }
    }

    private <T> Flux<T> appendRetry(Flux<T> publisher) {
        if (queryRetries > 0) {
            return publisher.retryWhen(YdbRetry.retryRequest(queryRetries, ydbMetrics));
        } else {
            return publisher;
        }
    }

    private <T> Flux<T> appendIdempotentRetry(Flux<T> publisher) {
        if (queryRetries > 0) {
            return publisher.retryWhen(YdbRetry.retryIdempotentRequest(queryRetries, ydbMetrics));
        } else {
            return publisher;
        }
    }

    private <T> Mono<T> appendRetryTimeout(Mono<T> publisher) {
        if (queryRetries > 0) {
            return publisher.retryWhen(YdbRetry.retryTimeout(queryRetries, ydbMetrics));
        } else {
            return publisher;
        }
    }

    private <T> Mono<T> appendTimeout(Mono<T> publisher) {
        if (!queryClientTimeout.isNegative() && !queryClientTimeout.isZero()) {
            return publisher.timeout(queryClientTimeout);
        } else {
            return publisher;
        }
    }

    private <T> Flux<T> appendTimeout(YdbReadTableSettings ydbReadTableSettings, Flux<T> publisher) {
        if (ydbReadTableSettings.getTimeout().isPresent()
                && !ydbReadTableSettings.getTimeout().get().isNegative()
                && !ydbReadTableSettings.getTimeout().get().isZero()) {
            return publisher.timeout(ydbReadTableSettings.getTimeout().get());
        } else if (!queryClientTimeout.isNegative() && !queryClientTimeout.isZero()) {
            return publisher.timeout(queryClientTimeout);
        } else {
            return publisher;
        }
    }

    private <T> Flux<T> appendTimeout(YdbExecuteScanQuerySettings ydbExecuteScanQuerySettings, Flux<T> publisher) {
        if (ydbExecuteScanQuerySettings.getTimeout().isPresent()
                && !ydbExecuteScanQuerySettings.getTimeout().get().isNegative()
                && !ydbExecuteScanQuerySettings.getTimeout().get().isZero()) {
            return publisher.timeout(ydbExecuteScanQuerySettings.getTimeout().get());
        } else if (!queryClientTimeout.isNegative() && !queryClientTimeout.isZero()) {
            return publisher.timeout(queryClientTimeout);
        } else {
            return publisher;
        }
    }

    private CreateTableSettings prepareCreateTableSettings() {
        return prepareCreateTableSettings(YdbCreateTableSettings.builder().build());
    }

    private CreateTableSettings prepareCreateTableSettings(YdbCreateTableSettings ydbCreateTableSettings) {
        CreateTableSettings settings = new CreateTableSettings();
        ydbCreateTableSettings.getCompactionPolicy().ifPresent(settings::setCompactionPolicy);
        ydbCreateTableSettings.getExecutionPolicy().ifPresent(settings::setExecutionPolicy);
        ydbCreateTableSettings.getPartitioningPolicy().ifPresent(settings::setPartitioningPolicy);
        ydbCreateTableSettings.getPresetName().ifPresent(settings::setPresetName);
        ydbCreateTableSettings.getReplicationPolicy().ifPresent(settings::setReplicationPolicy);
        ydbCreateTableSettings.getStoragePolicy().ifPresent(settings::setStoragePolicy);
        String logId = MDC.get(MdcTaskDecorator.LOG_ID_MDC_KEY);
        if (logId != null) {
            settings.setTraceId(logId);
        }
        if (queryTimeoutMillis > 0) {
            settings.setTimeout(queryTimeoutMillis, TimeUnit.MILLISECONDS);
        }
        return settings;
    }

    private DropTableSettings prepareDropTableSettings() {
        DropTableSettings settings = new DropTableSettings();
        String logId = MDC.get(MdcTaskDecorator.LOG_ID_MDC_KEY);
        if (logId != null) {
            settings.setTraceId(logId);
        }
        if (queryTimeoutMillis > 0) {
            settings.setTimeout(queryTimeoutMillis, TimeUnit.MILLISECONDS);
        }
        return settings;
    }

    private AlterTableSettings prepareAlterTableSettings() {
        return prepareAlterTableSettings(YdbAlterTableSettings.builder().build());
    }

    private AlterTableSettings prepareAlterTableSettings(YdbAlterTableSettings ydbAlterTableSettings) {
        AlterTableSettings settings = new AlterTableSettings();
        ydbAlterTableSettings.getAddColumns().forEach(settings::addColumn);
        ydbAlterTableSettings.getDropColumns().forEach(settings::dropColumn);
        String logId = MDC.get(MdcTaskDecorator.LOG_ID_MDC_KEY);
        if (logId != null) {
            settings.setTraceId(logId);
        }
        if (queryTimeoutMillis > 0) {
            settings.setTimeout(queryTimeoutMillis, TimeUnit.MILLISECONDS);
        }
        return settings;
    }

    private CopyTableSettings prepareCopyTableSettings() {
        CopyTableSettings settings = new CopyTableSettings();
        String logId = MDC.get(MdcTaskDecorator.LOG_ID_MDC_KEY);
        if (logId != null) {
            settings.setTraceId(logId);
        }
        if (queryTimeoutMillis > 0) {
            settings.setTimeout(queryTimeoutMillis, TimeUnit.MILLISECONDS);
        }
        return settings;
    }

    private DescribeTableSettings prepareDescribeTableSettings() {
        DescribeTableSettings settings = new DescribeTableSettings();
        String logId = MDC.get(MdcTaskDecorator.LOG_ID_MDC_KEY);
        if (logId != null) {
            settings.setTraceId(logId);
        }
        if (queryTimeoutMillis > 0) {
            settings.setTimeout(queryTimeoutMillis, TimeUnit.MILLISECONDS);
        }
        return settings;
    }

    private ExecuteDataQuerySettings prepareExecuteDataQuerySettings() {
        return prepareExecuteDataQuerySettings(YdbExecuteDataQuerySettings.builder().build());
    }

    private ExecuteDataQuerySettings prepareExecuteDataQuerySettings(
            YdbExecuteDataQuerySettings ydbExecuteDataQuerySettings) {
        ExecuteDataQuerySettings settings = new ExecuteDataQuerySettings();
        if (ydbExecuteDataQuerySettings.isKeepInQueryCache()) {
            settings.keepInQueryCache();
        }
        String logId = MDC.get(MdcTaskDecorator.LOG_ID_MDC_KEY);
        if (logId != null) {
            settings.setTraceId(logId);
        }
        if (queryTimeoutMillis > 0) {
            settings.setTimeout(queryTimeoutMillis, TimeUnit.MILLISECONDS);
        }
        return settings;
    }

    private PrepareDataQuerySettings preparePrepareDataQuerySettings() {
        return preparePrepareDataQuerySettings(YdbPrepareDataQuerySettings.builder().build());
    }

    private PrepareDataQuerySettings preparePrepareDataQuerySettings(
            YdbPrepareDataQuerySettings ydbPrepareDataQuerySettings) {
        PrepareDataQuerySettings settings = new PrepareDataQuerySettings();
        if (ydbPrepareDataQuerySettings.isKeepInQueryCache()) {
            settings.keepInQueryCache();
        }
        String logId = MDC.get(MdcTaskDecorator.LOG_ID_MDC_KEY);
        if (logId != null) {
            settings.setTraceId(logId);
        }
        if (queryTimeoutMillis > 0) {
            settings.setTimeout(queryTimeoutMillis, TimeUnit.MILLISECONDS);
        }
        return settings;
    }

    private ExecuteSchemeQuerySettings prepareExecuteSchemeQuerySettings() {
        ExecuteSchemeQuerySettings settings = new ExecuteSchemeQuerySettings();
        String logId = MDC.get(MdcTaskDecorator.LOG_ID_MDC_KEY);
        if (logId != null) {
            settings.setTraceId(logId);
        }
        if (queryTimeoutMillis > 0) {
            settings.setTimeout(queryTimeoutMillis, TimeUnit.MILLISECONDS);
        }
        return settings;
    }

    private ExplainDataQuerySettings prepareExplainDataQuerySettings() {
        ExplainDataQuerySettings settings = new ExplainDataQuerySettings();
        String logId = MDC.get(MdcTaskDecorator.LOG_ID_MDC_KEY);
        if (logId != null) {
            settings.setTraceId(logId);
        }
        if (queryTimeoutMillis > 0) {
            settings.setTimeout(queryTimeoutMillis, TimeUnit.MILLISECONDS);
        }
        return settings;
    }

    private BeginTxSettings prepareBeginTxSettings() {
        BeginTxSettings settings = new BeginTxSettings();
        String logId = MDC.get(MdcTaskDecorator.LOG_ID_MDC_KEY);
        if (logId != null) {
            settings.setTraceId(logId);
        }
        if (queryTimeoutMillis > 0) {
            settings.setTimeout(queryTimeoutMillis, TimeUnit.MILLISECONDS);
        }
        return settings;
    }

    private CommitTxSettings prepareCommitTxSettings() {
        CommitTxSettings settings = new CommitTxSettings();
        String logId = MDC.get(MdcTaskDecorator.LOG_ID_MDC_KEY);
        if (logId != null) {
            settings.setTraceId(logId);
        }
        if (queryTimeoutMillis > 0) {
            settings.setTimeout(queryTimeoutMillis, TimeUnit.MILLISECONDS);
        }
        return settings;
    }

    private RollbackTxSettings prepareRollbackTxSettings() {
        RollbackTxSettings settings = new RollbackTxSettings();
        String logId = MDC.get(MdcTaskDecorator.LOG_ID_MDC_KEY);
        if (logId != null) {
            settings.setTraceId(logId);
        }
        if (queryTimeoutMillis > 0) {
            settings.setTimeout(queryTimeoutMillis, TimeUnit.MILLISECONDS);
        }
        return settings;
    }

    private ReadTableSettings prepareReadTableSettings(YdbReadTableSettings ydbReadTableSettings) {
        ReadTableSettings.Builder builder = ReadTableSettings.newBuilder();
        if (ydbReadTableSettings.getTimeout().isPresent()) {
            builder.timeout(ydbReadTableSettings.getTimeout().get());
        } else {
            if (queryTimeoutMillis > 0) {
                builder.timeout(queryTimeoutMillis, TimeUnit.MILLISECONDS);
            }
        }
        if (!ydbReadTableSettings.getColumns().isEmpty()) {
            builder.columns(ydbReadTableSettings.getColumns());
        }
        builder.orderedRead(ydbReadTableSettings.isOrdered());
        ydbReadTableSettings.getFromKey().ifPresent(k -> builder.fromKey(k, ydbReadTableSettings.isFromInclusive()));
        ydbReadTableSettings.getToKey().ifPresent(k -> builder.toKey(k, ydbReadTableSettings.isToInclusive()));
        builder.rowLimit(ydbReadTableSettings.getRowLimit());
        return builder.build();
    }

    private ExecuteScanQuerySettings prepareExecuteScanQuerySettings(
            YdbExecuteScanQuerySettings ydbExecuteScanQuerySettings) {
        ExecuteScanQuerySettings.Builder builder = ExecuteScanQuerySettings.newBuilder();
        if (ydbExecuteScanQuerySettings.getTimeout().isPresent()) {
            builder.timeout(ydbExecuteScanQuerySettings.getTimeout().get());
        } else {
            if (queryTimeoutMillis > 0) {
                builder.timeout(queryTimeoutMillis, TimeUnit.MILLISECONDS);
            }
        }
        if (ydbExecuteScanQuerySettings.getRequestMode().isPresent()) {
            switch (ydbExecuteScanQuerySettings.getRequestMode().get()) {
                case UNSPECIFIED:
                    builder.mode(YdbTable.ExecuteScanQueryRequest.Mode.MODE_UNSPECIFIED);
                    break;
                case EXPLAIN:
                    builder.mode(YdbTable.ExecuteScanQueryRequest.Mode.MODE_EXPLAIN);
                    break;
                case EXEC:
                    builder.mode(YdbTable.ExecuteScanQueryRequest.Mode.MODE_EXEC);
                    break;
                default:
                    throw new IllegalArgumentException("Unexpected mode: " +
                            ydbExecuteScanQuerySettings.getRequestMode().get());
            }
        }
        if (ydbExecuteScanQuerySettings.getStatsMode().isPresent()) {
            switch (ydbExecuteScanQuerySettings.getStatsMode().get()) {
                case UNSPECIFIED:
                    builder.collectStats(YdbTable.QueryStatsCollection.Mode.STATS_COLLECTION_UNSPECIFIED);
                    break;
                case NONE:
                    builder.collectStats(YdbTable.QueryStatsCollection.Mode.STATS_COLLECTION_NONE);
                    break;
                case BASIC:
                    builder.collectStats(YdbTable.QueryStatsCollection.Mode.STATS_COLLECTION_BASIC);
                    break;
                case FULL:
                    builder.collectStats(YdbTable.QueryStatsCollection.Mode.STATS_COLLECTION_FULL);
                    break;
                default:
                    throw new IllegalArgumentException("Unexpected stats mode: " +
                            ydbExecuteScanQuerySettings.getStatsMode().get());
            }
        }
        return builder.build();
    }

    private KeepAliveSessionSettings prepareKeepAliveSessionSettings() {
        KeepAliveSessionSettings settings = new KeepAliveSessionSettings();
        String logId = MDC.get(MdcTaskDecorator.LOG_ID_MDC_KEY);
        if (logId != null) {
            settings.setTraceId(logId);
        }
        if (queryTimeoutMillis > 0) {
            settings.setTimeout(queryTimeoutMillis, TimeUnit.MILLISECONDS);
        }
        return settings;
    }

    private CloseSessionSettings prepareCloseSessionSettings() {
        CloseSessionSettings settings = new CloseSessionSettings();
        String logId = MDC.get(MdcTaskDecorator.LOG_ID_MDC_KEY);
        if (logId != null) {
            settings.setTraceId(logId);
        }
        if (queryTimeoutMillis > 0) {
            settings.setTimeout(queryTimeoutMillis, TimeUnit.MILLISECONDS);
        }
        return settings;
    }

    @SuppressWarnings("rawtypes")
    private void onDiscardedTransaction(com.yandex.ydb.core.Result r) {
        if (r == null) {
            return;
        }
        if (!r.isSuccess()) {
            return;
        }
        Optional valueO = r.ok();
        if (valueO.isEmpty()) {
            return;
        }
        Object value = valueO.get();
        if (!(value instanceof Transaction)) {
            return;
        }
        LOG.info("YDB transaction subscription was discarded");
        try {
            ((Transaction) value).rollback().whenComplete((s, e) -> {
                if (s != null && !s.isSuccess()) {
                    LOG.warn("Failed to rollback discarded YDB transaction: {}", s);
                }
                if (e != null) {
                    LOG.warn("Failed to rollback discarded YDB transaction", e);
                }
            });
        } catch (Exception e) {
            LOG.warn("Failed to rollback discarded YDB transaction", e);
        }
    }

}
