package ru.yandex.intranet.d.datasource.impl;

import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletionException;
import java.util.concurrent.TimeoutException;
import java.util.stream.Stream;

import com.yandex.ydb.core.Issue;
import com.yandex.ydb.core.Result;
import com.yandex.ydb.core.Status;
import com.yandex.ydb.core.StatusCode;
import com.yandex.ydb.core.UnexpectedResultException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.intranet.d.datasource.model.BadSessionException;
import ru.yandex.intranet.d.datasource.model.SessionPoolDepletedException;
import ru.yandex.intranet.d.datasource.model.TransactionLocksInvalidatedException;
import ru.yandex.intranet.d.datasource.model.TransportUnavailableException;
import ru.yandex.yql.proto.IssueId;

/**
 * YDB status utils.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class YdbStatusUtils {

    private static final Logger LOG = LoggerFactory.getLogger(YdbStatusUtils.class);

    private YdbStatusUtils() {
    }

    public static void checkStatus(Status status, String message) {
        if (!status.isSuccess() && status.getCode().equals(StatusCode.ABORTED)) {
            boolean issueMatch = List.of(status.getIssues()).stream().anyMatch(issue -> {
                boolean severityMatch = Issue.Severity.ERROR.equals(issue.getSeverity());
                boolean codeMatch = issue.getCode() == IssueId.TIssuesIds.EIssueCode.CORE_EXEC_VALUE;
                boolean innerIssueMatch = List.of(issue.getIssues()).stream().anyMatch(innerIssue -> {
                    boolean innerSeverityMatch = Issue.Severity.ERROR
                            .equals(innerIssue.getSeverity());
                    boolean innerCodeMatch = innerIssue.getCode() == IssueId.TIssuesIds.EIssueCode
                            .KIKIMR_LOCKS_INVALIDATED_VALUE;
                    return innerSeverityMatch && innerCodeMatch;
                });
                return severityMatch && codeMatch && innerIssueMatch;
            });
            if (issueMatch) {
                LOG.warn("YDB transaction locks invalidated: {}", status);
                throw new TransactionLocksInvalidatedException(message,
                        status.getCode(), status.getIssues());
            }
        }
        if (!status.isSuccess() && status.getCode().equals(StatusCode.TRANSPORT_UNAVAILABLE)) {
            LOG.warn("YDB transport unavailable: {}", status);
            throw new TransportUnavailableException(message, status.getCode(), status.getIssues());
        }
        if (!status.isSuccess() && status.getCode().equals(StatusCode.BAD_SESSION)) {
            LOG.warn("Bad YDB session: {}", status);
            throw new BadSessionException(message, status.getCode(), status.getIssues());
        }
        status.expect(message);
    }

    public static <T> T checkResult(Result<T> result, String message) {
        if (!result.isSuccess() && result.getCode().equals(StatusCode.ABORTED)) {
            boolean issueMatch = List.of(result.getIssues()).stream().anyMatch(issue -> {
                boolean severityMatch = Issue.Severity.ERROR.equals(issue.getSeverity());
                boolean codeMatch = issue.getCode() == IssueId.TIssuesIds.EIssueCode.CORE_EXEC_VALUE;
                boolean innerIssueMatch = List.of(issue.getIssues()).stream().anyMatch(innerIssue -> {
                    boolean innerSeverityMatch = Issue.Severity.ERROR
                            .equals(innerIssue.getSeverity());
                    boolean innerCodeMatch = innerIssue.getCode() == IssueId.TIssuesIds.EIssueCode
                            .KIKIMR_LOCKS_INVALIDATED_VALUE;
                    return innerSeverityMatch && innerCodeMatch;
                });
                return severityMatch && codeMatch && innerIssueMatch;
            });
            if (issueMatch) {
                LOG.warn("YDB transaction locks invalidated: {}", result);
                throw new TransactionLocksInvalidatedException(message,
                        result.getCode(), result.getIssues());
            }
        }
        if (!result.isSuccess() && result.getCode().equals(StatusCode.TRANSPORT_UNAVAILABLE)) {
            LOG.warn("YDB transport unavailable: {}", result);
            throw new TransportUnavailableException(message, result.getCode(), result.getIssues());
        }
        if (!result.isSuccess() && result.getCode().equals(StatusCode.BAD_SESSION)) {
            LOG.warn("Bad YDB session: {}", result);
            throw new BadSessionException(message, result.getCode(), result.getIssues());
        }
        return result.expect(message);
    }

    public static <T> T checkSessionResult(Result<T> result, String message) {
        if (!result.isSuccess() && result.getCode().equals(StatusCode.CLIENT_INTERNAL_ERROR)) {
            Optional<Throwable> errorO = result.error();
            if (errorO.isPresent()) {
                Throwable error = errorO.get();
                if (isDepletedSessionPoolUnwrapped(error) || isDepletedSessionPoolWrapped(error)) {
                    LOG.warn("YDB session pool is depleted: {}", result);
                    throw new SessionPoolDepletedException(message, result.getCode(), result.getIssues());
                }
                Optional<UnexpectedResultException> unexpectedO = unwrapUnexpectedResult(error);
                if (unexpectedO.isPresent()) {
                    UnexpectedResultException unexpected = unexpectedO.get();
                    if (unexpected.getStatusCode().equals(StatusCode.TRANSPORT_UNAVAILABLE)) {
                        LOG.warn("YDB transport unavailable: {}", result);
                        throw new TransportUnavailableException(message,
                                unexpected.getStatusCode(), unexpected.getIssues());
                    } else {
                        LOG.warn("Failed to get a session: {}", result);
                        throw new UnexpectedResultException(message,
                                unexpected.getStatusCode(), unexpected.getIssues());
                    }
                }
            }
        }
        if (!result.isSuccess() && result.getCode().equals(StatusCode.TRANSPORT_UNAVAILABLE)) {
            LOG.warn("YDB transport unavailable: {}", result);
            throw new TransportUnavailableException(message, result.getCode(), result.getIssues());
        }
        return result.expect(message);
    }

    public static void checkAnyStatus(Status status, String message) {
        if (!status.isSuccess() && status.getCode().equals(StatusCode.TRANSPORT_UNAVAILABLE)) {
            LOG.warn("YDB transport unavailable: {}", status);
            throw new TransportUnavailableException(message, status.getCode(), status.getIssues());
        }
        if (!status.isSuccess() && status.getCode().equals(StatusCode.BAD_SESSION)) {
            LOG.warn("Bad YDB session: {}", status);
            throw new BadSessionException(message, status.getCode(), status.getIssues());
        }
        status.expect(message);
    }

    public static Throwable checkAnyStatusReturn(Status status, String message) {
        if (!status.isSuccess() && status.getCode().equals(StatusCode.TRANSPORT_UNAVAILABLE)) {
            LOG.warn("YDB transport unavailable: {}", status);
            return new TransportUnavailableException(message, status.getCode(), status.getIssues());
        }
        if (!status.isSuccess() && status.getCode().equals(StatusCode.BAD_SESSION)) {
            LOG.warn("Bad YDB session: {}", status);
            throw new BadSessionException(message, status.getCode(), status.getIssues());
        }
        return new UnexpectedResultException(message, status.getCode(), status.getIssues());
    }

    public static <T> T checkAnyResult(Result<T> result, String message) {
        if (!result.isSuccess() && result.getCode().equals(StatusCode.TRANSPORT_UNAVAILABLE)) {
            LOG.warn("YDB transport unavailable: {}", result);
            throw new TransportUnavailableException(message, result.getCode(), result.getIssues());
        }
        if (!result.isSuccess() && result.getCode().equals(StatusCode.BAD_SESSION)) {
            LOG.warn("Bad YDB session: {}", result);
            throw new BadSessionException(message, result.getCode(), result.getIssues());
        }
        return result.expect(message);
    }

    public static Throwable checkException(Throwable t) {
        if (t instanceof UnexpectedResultException) {
            UnexpectedResultException unexpected = (UnexpectedResultException) t;
            if (unexpected.getStatusCode().equals(StatusCode.TRANSPORT_UNAVAILABLE)) {
                LOG.warn("YDB transport unavailable", t);
                return new TransportUnavailableException(unexpected.getMessage(), unexpected.getStatusCode(),
                        unexpected.getIssues());
            }
            if (unexpected.getStatusCode().equals(StatusCode.BAD_SESSION)) {
                LOG.warn("Bad YDB session", t);
                throw new BadSessionException(unexpected.getMessage(), unexpected.getStatusCode(),
                        unexpected.getIssues());
            }
        }
        return t;
    }

    public static boolean isTransactionNotFoundError(UnexpectedResultException ex) {
        return StatusCode.NOT_FOUND.equals(ex.getStatusCode()) && Stream.of(ex.getIssues()).anyMatch(issue -> {
            boolean severityMatch = Issue.Severity.ERROR.equals(issue.getSeverity());
            boolean codeMatch = issue.getCode() == IssueId.TIssuesIds.EIssueCode.KIKIMR_TRANSACTION_NOT_FOUND_VALUE;
            return severityMatch && codeMatch;
        });
    }

    private static boolean isDepletedSessionPoolUnwrapped(Throwable error) {
        return error instanceof UnexpectedResultException
                && (error.getCause() instanceof IllegalStateException
                || error.getCause() instanceof TimeoutException);
    }

    private static boolean isDepletedSessionPoolWrapped(Throwable error) {
        return error instanceof UnexpectedResultException && error.getCause() instanceof CompletionException
                && (error.getCause().getCause() instanceof IllegalStateException
                || error.getCause().getCause() instanceof TimeoutException);
    }

    private static Optional<UnexpectedResultException> unwrapUnexpectedResult(Throwable error) {
        if (error instanceof UnexpectedResultException && error.getCause() instanceof CompletionException
                && error.getCause().getCause() instanceof UnexpectedResultException) {
            return Optional.of((UnexpectedResultException) error.getCause().getCause());
        } else if (error instanceof UnexpectedResultException) {
            return Optional.of((UnexpectedResultException) error);
        } else {
            return Optional.empty();
        }
    }

}
