package ru.yandex.intranet.d.datasource.impl;

import java.time.Duration;
import java.util.Optional;
import java.util.function.Function;

import com.yandex.ydb.core.Result;
import com.yandex.ydb.table.Session;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.stats.SessionPoolStats;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import reactor.core.scheduler.Schedulers;

import ru.yandex.intranet.d.datasource.model.YdbSession;
import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.metrics.YdbMetrics;
import ru.yandex.intranet.d.util.AsyncMetrics;

/**
 * YDB table client.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public class YdbTableClientImpl implements YdbTableClient {

    private static final Logger LOG = LoggerFactory.getLogger(YdbTableClientImpl.class);

    private final TableClient tableClient;
    private final Duration sessionWaitTimeout;
    private final long sessionWaitRetries;
    private final long queryTimeoutMillis;
    private final long queryRetries;
    private final long sessionRetries;
    private final long transactionRetries;
    private final YdbMetrics ydbMetrics;
    private final Duration sessionClientWaitTimeout;
    private final Duration queryClientTimeout;

    @SuppressWarnings("ParameterNumber")
    public YdbTableClientImpl(TableClient tableClient, Duration sessionWaitTimeout, long sessionWaitRetries,
                              long queryTimeoutMillis, long queryRetries, long sessionRetries,
                              long transactionRetries, YdbMetrics ydbMetrics, Duration sessionClientWaitTimeout,
                              Duration queryClientTimeout) {
        this.tableClient = tableClient;
        this.sessionWaitTimeout = sessionWaitTimeout;
        this.sessionWaitRetries = sessionWaitRetries;
        this.queryTimeoutMillis = queryTimeoutMillis;
        this.queryRetries = queryRetries;
        this.sessionRetries = sessionRetries;
        this.transactionRetries = transactionRetries;
        this.ydbMetrics = ydbMetrics;
        this.sessionClientWaitTimeout = sessionClientWaitTimeout;
        this.queryClientTimeout = queryClientTimeout;
    }

    @Override
    public void close() {
        tableClient.close();
    }

    @Override
    public Mono<YdbSession> getSession() {
        return getSessionImpl().map(session -> new YdbSessionImpl(session, queryTimeoutMillis, queryRetries,
                transactionRetries, ydbMetrics, queryClientTimeout));
    }

    @Override
    public <T> Mono<T> usingSessionMono(Function<YdbSession, Mono<T>> body) {
        return Mono.usingWhen(getSessionImpl(),
                session -> body.apply(new YdbSessionImpl(session, queryTimeoutMillis,
                        queryRetries, transactionRetries, ydbMetrics, queryClientTimeout)),
                this::releaseSession);
    }

    @Override
    public <T> Flux<T> usingSessionFlux(Function<YdbSession, Flux<T>> body) {
        return Flux.usingWhen(getSessionImpl(),
                session -> body.apply(new YdbSessionImpl(session, queryTimeoutMillis,
                        queryRetries, transactionRetries, ydbMetrics, queryClientTimeout)),
                this::releaseSession);
    }

    @Override
    public <T> Mono<T> usingSessionMonoRetryable(Function<YdbSession, Mono<T>> body) {
        Mono<T> result = Mono.usingWhen(getSessionImpl(),
                session -> body.apply(new YdbSessionImpl(session, queryTimeoutMillis,
                        queryRetries, transactionRetries, ydbMetrics, queryClientTimeout)),
                this::releaseSession);
        if (sessionRetries > 0) {
            return result.retryWhen(YdbRetry.retryIdempotentSession(sessionRetries, ydbMetrics));
        } else {
            return result;
        }
    }

    @Override
    public <T> Flux<T> usingSessionFluxRetryable(Function<YdbSession, Flux<T>> body) {
        Flux<T> result = Flux.usingWhen(getSessionImpl(),
                session -> body.apply(new YdbSessionImpl(session, queryTimeoutMillis,
                        queryRetries, transactionRetries, ydbMetrics, queryClientTimeout)),
                this::releaseSession);
        if (sessionRetries > 0) {
            return result.retryWhen(YdbRetry.retryIdempotentSession(sessionRetries, ydbMetrics));
        } else {
            return result;
        }
    }

    @Override
    public SessionPoolStats getSessionPoolStats() {
        return tableClient.getSessionPoolStats();
    }

    private Mono<Session> getSessionImpl() {
        Mono<Session> result = appendTimeout(Mono.fromFuture(() -> tableClient
                .getOrCreateSession(sessionWaitTimeout))
                .doOnDiscard(Result.class, this::onSessionDiscarded)
                .map(s -> YdbStatusUtils.checkSessionResult(s, "Failed to obtain YDB session"))
        );
        if (sessionWaitRetries > 0) {
            return AsyncMetrics.metric(result
                    .retryWhen(YdbRetry.retryRequest(sessionWaitRetries, ydbMetrics)), ydbMetrics::afterGetSession);
        } else {
            return AsyncMetrics.metric(result, ydbMetrics::afterGetSession);
        }
    }

    private Mono<Void> releaseSession(Session session) {
        return AsyncMetrics.metric(appendTimeout(Mono.just(session)
                .publishOn(Schedulers.boundedElastic())
                .map(Session::release)), ydbMetrics::afterReleaseSession)
                .doOnError(e -> LOG.warn("Failed to release session", e))
                .then()
                .onErrorResume(e -> Mono.empty());
    }

    private <T> Mono<T> appendTimeout(Mono<T> publisher) {
        if (!sessionClientWaitTimeout.isZero() && !sessionClientWaitTimeout.isNegative()) {
            return publisher.timeout(sessionClientWaitTimeout);
        } else {
            return publisher;
        }
    }

    @SuppressWarnings("rawtypes")
    private void onSessionDiscarded(Result r) {
        if (r == null) {
            return;
        }
        if (!r.isSuccess()) {
            return;
        }
        Optional valueO = r.ok();
        if (valueO.isEmpty()) {
            return;
        }
        Object value = valueO.get();
        if (!(value instanceof Session)) {
            return;
        }
        LOG.info("YDB session subscription was discarded");
        try {
            ((Session) value).release();
        } catch (Exception e) {
            LOG.warn("Failed to release discarded YDB session", e);
        }
    }

}
