package ru.yandex.intranet.d.datasource.impl;

import java.time.Duration;
import java.util.concurrent.TimeUnit;

import com.yandex.ydb.table.settings.CommitTxSettings;
import com.yandex.ydb.table.settings.RollbackTxSettings;
import com.yandex.ydb.table.transaction.Transaction;
import org.slf4j.MDC;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.datasource.model.YdbTransaction;
import ru.yandex.intranet.d.metrics.YdbMetrics;
import ru.yandex.intranet.d.util.AsyncMetrics;
import ru.yandex.intranet.d.util.MdcTaskDecorator;

/**
 * YDB transaction.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public class YdbTransactionImpl implements YdbTransaction {

    private final Transaction transaction;
    private final long queryTimeoutMillis;
    private final long queryRetries;
    private final YdbMetrics ydbMetrics;
    private final Duration queryClientTimeout;

    public YdbTransactionImpl(Transaction transaction, long queryTimeoutMillis, long queryRetries,
                              YdbMetrics ydbMetrics, Duration queryClientTimeout) {
        this.transaction = transaction;
        this.queryTimeoutMillis = queryTimeoutMillis;
        this.queryRetries = queryRetries;
        this.ydbMetrics = ydbMetrics;
        this.queryClientTimeout = queryClientTimeout;
    }

    @Override
    public String getId() {
        return transaction.getId();
    }

    @Override
    public boolean isActive() {
        return transaction.isActive();
    }

    @Override
    public Mono<Void> commit() {
        return AsyncMetrics.metric(appendRetry(appendTimeout(Mono
                .fromFuture(() -> transaction.commit(prepareCommitTxSettings()))
                .doOnSuccess(s -> YdbStatusUtils.checkStatus(s, "Failed to commit YDB transaction")).then())),
                ydbMetrics::afterCommitTransaction);
    }

    @Override
    public Mono<Void> rollback() {
        return AsyncMetrics.metric(appendRetry(appendTimeout(Mono
                .fromFuture(() -> transaction.rollback(prepareRollbackTxSettings()))
                .doOnSuccess(s -> YdbStatusUtils.checkAnyStatus(s, "Failed to rollback YDB transaction")).then())),
                ydbMetrics::afterRollbackTransaction);
    }

    private <T> Mono<T> appendRetry(Mono<T> publisher) {
        if (queryRetries > 0) {
            return publisher.retryWhen(YdbRetry.retryRequest(queryRetries, ydbMetrics));
        } else {
            return publisher;
        }
    }

    private <T> Mono<T> appendTimeout(Mono<T> publisher) {
        if (!queryClientTimeout.isNegative() && !queryClientTimeout.isZero()) {
            return publisher.timeout(queryClientTimeout);
        } else {
            return publisher;
        }
    }

    private CommitTxSettings prepareCommitTxSettings() {
        CommitTxSettings settings = new CommitTxSettings();
        String logId = MDC.get(MdcTaskDecorator.LOG_ID_MDC_KEY);
        if (logId != null) {
            settings.setTraceId(logId);
        }
        if (queryTimeoutMillis > 0) {
            settings.setTimeout(queryTimeoutMillis, TimeUnit.MILLISECONDS);
        }
        return settings;
    }

    private RollbackTxSettings prepareRollbackTxSettings() {
        RollbackTxSettings settings = new RollbackTxSettings();
        String logId = MDC.get(MdcTaskDecorator.LOG_ID_MDC_KEY);
        if (logId != null) {
            settings.setTraceId(logId);
        }
        if (queryTimeoutMillis > 0) {
            settings.setTimeout(queryTimeoutMillis, TimeUnit.MILLISECONDS);
        }
        return settings;
    }

}
