package ru.yandex.intranet.d.datasource.impl;

import java.time.Duration;
import java.util.Optional;
import java.util.concurrent.TimeUnit;

import com.yandex.ydb.table.query.DataQuery;
import com.yandex.ydb.table.query.DataQueryResult;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.settings.ExecuteDataQuerySettings;
import com.yandex.ydb.table.transaction.TransactionMode;
import com.yandex.ydb.table.transaction.TxControl;
import org.slf4j.MDC;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.datasource.model.YdbExecuteDataQuerySettings;
import ru.yandex.intranet.d.datasource.model.YdbTxDataQuery;
import ru.yandex.intranet.d.metrics.YdbMetrics;
import ru.yandex.intranet.d.util.AsyncMetrics;
import ru.yandex.intranet.d.util.MdcTaskDecorator;

/**
 * YDB transaction data query.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public class YdbTxDataQueryImpl implements YdbTxDataQuery {

    private final DataQuery dataQuery;
    private final TxControl txControl;
    private final long queryTimeoutMillis;
    private final long queryRetries;
    private final long transactionRetries;
    private final boolean retryCommit;
    private final YdbMetrics ydbMetrics;
    private final Duration queryClientTimeout;

    @SuppressWarnings("ParameterNumber")
    public YdbTxDataQueryImpl(DataQuery dataQuery, TxControl txControl,
                              long queryTimeoutMillis, long queryRetries, long transactionRetries,
                              boolean retryCommit, YdbMetrics ydbMetrics, Duration queryClientTimeout) {
        this.dataQuery = dataQuery;
        this.txControl = txControl;
        this.queryTimeoutMillis = queryTimeoutMillis;
        this.queryRetries = queryRetries;
        this.transactionRetries = transactionRetries;
        this.retryCommit = retryCommit;
        this.ydbMetrics = ydbMetrics;
        this.queryClientTimeout = queryClientTimeout;
    }

    @Override
    public String getId() {
        return dataQuery.getId();
    }

    @Override
    public Params newParams() {
        return dataQuery.newParams();
    }

    @Override
    public Optional<String> getText() {
        return dataQuery.getText();
    }

    @Override
    public Mono<DataQueryResult> execute(Params params, YdbExecuteDataQuerySettings settings) {
        return AsyncMetrics.metric(appendTxRetryImmediate(appendRetryImmediate(appendTimeout(Mono.fromFuture(() ->
                dataQuery.execute(txControl, params, prepareSettings(settings)))
                .map(r -> YdbStatusUtils.checkResult(r, "Failed to execute YDB data query"))))),
                ydbMetrics::afterExecuteDataQuery);
    }

    @Override
    public Mono<DataQueryResult> execute(Params params) {
        return AsyncMetrics.metric(appendTxRetryImmediate(appendRetryImmediate(appendTimeout(Mono.fromFuture(() ->
                dataQuery.execute(txControl, params, prepareSettings()))
                .map(r -> YdbStatusUtils.checkResult(r, "Failed to execute YDB data query"))))),
                ydbMetrics::afterExecuteDataQuery);
    }

    @Override
    public Mono<DataQueryResult> execute() {
        return AsyncMetrics.metric(appendTxRetryImmediate(appendRetryImmediate(appendTimeout(Mono.fromFuture(() ->
                dataQuery.execute(txControl, Params.empty(), prepareSettings()))
                .map(r -> YdbStatusUtils.checkResult(r, "Failed to execute YDB data query"))))),
                ydbMetrics::afterExecuteDataQuery);
    }

    @Override
    public Mono<DataQueryResult> executeRetryable(Params params, YdbExecuteDataQuerySettings settings) {
        return AsyncMetrics.metric(appendTxRetryImmediate(appendIdempotentRetry(appendTimeout(Mono.fromFuture(() ->
                                dataQuery.execute(txControl, params, prepareSettings(settings)))
                        .map(r -> YdbStatusUtils.checkResult(r, "Failed to execute YDB data query"))))),
                ydbMetrics::afterExecuteDataQuery);
    }

    @Override
    public Mono<DataQueryResult> executeRetryable(Params params) {
        return AsyncMetrics.metric(appendTxRetryImmediate(appendIdempotentRetry(appendTimeout(Mono.fromFuture(() ->
                                dataQuery.execute(txControl, params, prepareSettings()))
                        .map(r -> YdbStatusUtils.checkResult(r, "Failed to execute YDB data query"))))),
                ydbMetrics::afterExecuteDataQuery);
    }

    @Override
    public Mono<DataQueryResult> executeRetryable() {
        return AsyncMetrics.metric(appendTxRetryImmediate(appendIdempotentRetry(appendTimeout(Mono.fromFuture(() ->
                                dataQuery.execute(txControl, Params.empty(), prepareSettings()))
                        .map(r -> YdbStatusUtils.checkResult(r, "Failed to execute YDB data query"))))),
                ydbMetrics::afterExecuteDataQuery);
    }

    @Override
    public Mono<DataQueryResult> executeCommit(TransactionMode txMode, Params params,
                                               YdbExecuteDataQuerySettings settings) {
        return AsyncMetrics.metric(appendTxRetryImmediate(appendRetryImmediate(appendTimeout(Mono.fromFuture(() ->
                dataQuery.execute(txModeToTxControl(txMode), params, prepareSettings(settings)))
                .map(r -> YdbStatusUtils.checkResult(r, "Failed to execute YDB data query"))))),
                ydbMetrics::afterExecuteDataQuery);
    }

    @Override
    public Mono<DataQueryResult> executeCommit(TransactionMode txMode, Params params) {
        return AsyncMetrics.metric(appendTxRetryImmediate(appendRetryImmediate(appendTimeout(Mono.fromFuture(() ->
                dataQuery.execute(txModeToTxControl(txMode), params, prepareSettings()))
                .map(r -> YdbStatusUtils.checkResult(r, "Failed to execute YDB data query"))))),
                ydbMetrics::afterExecuteDataQuery);
    }

    @Override
    public Mono<DataQueryResult> executeCommit(TransactionMode txMode) {
        return AsyncMetrics.metric(appendTxRetryImmediate(appendRetryImmediate(appendTimeout(Mono.fromFuture(() ->
                dataQuery.execute(txModeToTxControl(txMode), Params.empty(), prepareSettings()))
                .map(r -> YdbStatusUtils.checkResult(r, "Failed to execute YDB data query"))))),
                ydbMetrics::afterExecuteDataQuery);
    }

    @Override
    public Mono<DataQueryResult> executeCommitRetryable(TransactionMode txMode, Params params,
                                                        YdbExecuteDataQuerySettings settings) {
        return AsyncMetrics.metric(appendIdempotentTxRetry(appendIdempotentRetry(appendTimeout(Mono.fromFuture(() ->
                dataQuery.execute(txModeToTxControl(txMode), params, prepareSettings(settings)))
                .map(r -> YdbStatusUtils.checkResult(r, "Failed to execute YDB data query"))))),
                ydbMetrics::afterExecuteDataQuery);
    }

    @Override
    public Mono<DataQueryResult> executeCommitRetryable(TransactionMode txMode, Params params) {
        return AsyncMetrics.metric(appendIdempotentTxRetry(appendIdempotentRetry(appendTimeout(Mono.fromFuture(() ->
                dataQuery.execute(txModeToTxControl(txMode), params, prepareSettings()))
                .map(r -> YdbStatusUtils.checkResult(r, "Failed to execute YDB data query"))))),
                ydbMetrics::afterExecuteDataQuery);
    }

    @Override
    public Mono<DataQueryResult> executeCommitRetryable(TransactionMode txMode) {
        return AsyncMetrics.metric(appendIdempotentTxRetry(appendIdempotentRetry(appendTimeout(Mono.fromFuture(() ->
                dataQuery.execute(txModeToTxControl(txMode), Params.empty(), prepareSettings()))
                .map(r -> YdbStatusUtils.checkResult(r, "Failed to execute YDB data query"))))),
                ydbMetrics::afterExecuteDataQuery);
    }

    @Override
    public TxControl getTxControl() {
        return txControl;
    }

    private TxControl txModeToTxControl(TransactionMode txMode) {
        switch (txMode) {
            case SERIALIZABLE_READ_WRITE:
                return TxControl.serializableRw().setCommitTx(true);
            case ONLINE_READ_ONLY:
                return TxControl.onlineRo().setCommitTx(true);
            case STALE_READ_ONLY:
                return TxControl.staleRo().setCommitTx(true);
            default:
                throw new IllegalArgumentException("Unexpected transaction mode: " + txMode);
        }
    }

    private <T> Mono<T> appendRetryImmediate(Mono<T> publisher) {
        if (queryRetries > 0) {
            if (retryCommit) {
                return publisher.retryWhen(YdbRetry.retryIdempotentRequest(queryRetries, ydbMetrics));
            } else {
                return publisher.retryWhen(YdbRetry.retryRequest(queryRetries, ydbMetrics));
            }
        } else {
            return publisher;
        }
    }

    private <T> Mono<T> appendIdempotentRetry(Mono<T> publisher) {
        if (queryRetries > 0) {
            return publisher.retryWhen(YdbRetry.retryIdempotentRequest(queryRetries, ydbMetrics));
        } else {
            return publisher;
        }
    }

    private <T> Mono<T> appendIdempotentTxRetry(Mono<T> publisher) {
        if (transactionRetries > 0 && txControl.isCommitTx() && txControl.toPb().hasBeginTx()) {
            return publisher.retryWhen(YdbRetry.retryIdempotentTransaction(transactionRetries, ydbMetrics));
        } else {
            return publisher;
        }
    }

    private <T> Mono<T> appendTxRetryImmediate(Mono<T> publisher) {
        if (transactionRetries > 0 && txControl.isCommitTx() && txControl.toPb().hasBeginTx()) {
            if (retryCommit) {
                return publisher.retryWhen(YdbRetry.retryIdempotentTransaction(transactionRetries, ydbMetrics));
            } else {
                return publisher.retryWhen(YdbRetry.retryTransaction(transactionRetries, ydbMetrics));
            }
        } else {
            return publisher;
        }
    }

    private <T> Mono<T> appendTimeout(Mono<T> publisher) {
        if (!queryClientTimeout.isNegative() && !queryClientTimeout.isZero()) {
            return publisher.timeout(queryClientTimeout);
        } else {
            return publisher;
        }
    }

    private ExecuteDataQuerySettings prepareSettings() {
        return prepareSettings(YdbExecuteDataQuerySettings.builder().build());
    }

    private ExecuteDataQuerySettings prepareSettings(YdbExecuteDataQuerySettings ydbExecuteDataQuerySettings) {
        ExecuteDataQuerySettings settings = new ExecuteDataQuerySettings();
        if (ydbExecuteDataQuerySettings.isKeepInQueryCache()) {
            settings.keepInQueryCache();
        }
        String logId = MDC.get(MdcTaskDecorator.LOG_ID_MDC_KEY);
        if (logId != null) {
            settings.setTraceId(logId);
        }
        if (queryTimeoutMillis > 0) {
            settings.setTimeout(queryTimeoutMillis, TimeUnit.MILLISECONDS);
        }
        return settings;
    }

}
