package ru.yandex.intranet.d.datasource.impl;

import java.time.Duration;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.function.BiFunction;
import java.util.function.Function;

import com.yandex.ydb.table.Session;
import com.yandex.ydb.table.SessionStatus;
import com.yandex.ydb.table.query.DataQueryResult;
import com.yandex.ydb.table.query.ExplainDataQueryResult;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.settings.CommitTxSettings;
import com.yandex.ydb.table.settings.ExecuteDataQuerySettings;
import com.yandex.ydb.table.settings.ExplainDataQuerySettings;
import com.yandex.ydb.table.settings.KeepAliveSessionSettings;
import com.yandex.ydb.table.settings.PrepareDataQuerySettings;
import com.yandex.ydb.table.settings.ReadTableSettings;
import com.yandex.ydb.table.settings.RollbackTxSettings;
import com.yandex.ydb.table.transaction.Transaction;
import com.yandex.ydb.table.transaction.TransactionMode;
import com.yandex.ydb.table.transaction.TxControl;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;

import ru.yandex.intranet.d.datasource.model.YdbExecuteDataQuerySettings;
import ru.yandex.intranet.d.datasource.model.YdbPrepareDataQuerySettings;
import ru.yandex.intranet.d.datasource.model.YdbReadTableSettings;
import ru.yandex.intranet.d.datasource.model.YdbTxDataQuery;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.metrics.YdbMetrics;
import ru.yandex.intranet.d.util.AsyncMetrics;
import ru.yandex.intranet.d.util.MdcTaskDecorator;

/**
 * YDB transaction session.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public class YdbTxSessionImpl implements YdbTxSession {

    private static final Logger LOG = LoggerFactory.getLogger(YdbSessionImpl.class);

    private final Session session;
    private final TxControl txControl;
    private final long queryTimeoutMillis;
    private final long queryRetries;
    private final long transactionRetries;
    private final boolean retryCommit;
    private final YdbMetrics ydbMetrics;
    private final Duration queryClientTimeout;

    @SuppressWarnings("ParameterNumber")
    public YdbTxSessionImpl(Session session, TxControl txControl, long queryTimeoutMillis,
                            long queryRetries, long transactionRetries, boolean retryCommit,
                            YdbMetrics ydbMetrics, Duration queryClientTimeout) {
        this.session = session;
        this.txControl = txControl;
        this.queryTimeoutMillis = queryTimeoutMillis;
        this.queryRetries = queryRetries;
        this.transactionRetries = transactionRetries;
        this.retryCommit = retryCommit;
        this.ydbMetrics = ydbMetrics;
        this.queryClientTimeout = queryClientTimeout;
    }

    @Override
    public String getId() {
        return session.getId();
    }

    @Override
    public Mono<DataQueryResult> executeDataQuery(String query, Params params,
                                                  YdbExecuteDataQuerySettings settings) {
        return AsyncMetrics.metric(appendTxRetryImmediate(appendRetryImmediate(appendTimeout(Mono.fromFuture(() ->
                session.executeDataQuery(query, txControl, params, prepareExecuteDataQuerySettings(settings)))
                .map(s -> YdbStatusUtils.checkResult(s, "Failed to execute YDB data query"))))),
                ydbMetrics::afterExecuteDataQuery);
    }

    @Override
    public Mono<DataQueryResult> executeDataQuery(String query, Params params) {
        return AsyncMetrics.metric(appendTxRetryImmediate(appendRetryImmediate(appendTimeout(Mono.fromFuture(() ->
                session.executeDataQuery(query, txControl, params, prepareExecuteDataQuerySettings()))
                .map(s -> YdbStatusUtils.checkResult(s, "Failed to execute YDB data query"))))),
                ydbMetrics::afterExecuteDataQuery);
    }

    @Override
    public Mono<DataQueryResult> executeDataQuery(String query) {
        return AsyncMetrics.metric(appendTxRetryImmediate(appendRetryImmediate(appendTimeout(Mono.fromFuture(() ->
                session.executeDataQuery(query, txControl, Params.empty(), prepareExecuteDataQuerySettings()))
                .map(s -> YdbStatusUtils.checkResult(s, "Failed to execute YDB data query"))))),
                ydbMetrics::afterExecuteDataQuery);
    }

    @Override
    public Mono<DataQueryResult> executeDataQueryRetryable(String query, Params params,
                                                           YdbExecuteDataQuerySettings settings) {
        return AsyncMetrics.metric(appendTxRetryImmediate(appendIdempotentRetry(appendTimeout(Mono.fromFuture(() ->
                        session.executeDataQuery(query, txControl, params, prepareExecuteDataQuerySettings(settings)))
                        .map(s -> YdbStatusUtils.checkResult(s, "Failed to execute YDB data query"))))),
                ydbMetrics::afterExecuteDataQuery);
    }

    @Override
    public Mono<DataQueryResult> executeDataQueryRetryable(String query, Params params) {
        return AsyncMetrics.metric(appendTxRetryImmediate(appendIdempotentRetry(appendTimeout(Mono.fromFuture(() ->
                        session.executeDataQuery(query, txControl, params, prepareExecuteDataQuerySettings()))
                        .map(s -> YdbStatusUtils.checkResult(s, "Failed to execute YDB data query"))))),
                ydbMetrics::afterExecuteDataQuery);
    }

    @Override
    public Mono<DataQueryResult> executeDataQueryRetryable(String query) {
        return AsyncMetrics.metric(appendTxRetryImmediate(appendIdempotentRetry(appendTimeout(Mono.fromFuture(() ->
                        session.executeDataQuery(query, txControl, Params.empty(), prepareExecuteDataQuerySettings()))
                        .map(s -> YdbStatusUtils.checkResult(s, "Failed to execute YDB data query"))))),
                ydbMetrics::afterExecuteDataQuery);
    }

    @Override
    public Mono<DataQueryResult> executeDataQueryCommit(String query, TransactionMode txMode, Params params,
                                                        YdbExecuteDataQuerySettings settings) {
        return AsyncMetrics.metric(appendTxRetryImmediate(appendRetryImmediate(appendTimeout(Mono.fromFuture(() ->
                session.executeDataQuery(query, txModeToTxControl(txMode, true), params,
                        prepareExecuteDataQuerySettings(settings)))
                .map(s -> YdbStatusUtils.checkResult(s, "Failed to execute YDB data query"))))),
                ydbMetrics::afterExecuteDataQuery);
    }

    @Override
    public Mono<DataQueryResult> executeDataQueryCommit(String query, TransactionMode txMode, Params params) {
        return AsyncMetrics.metric(appendTxRetryImmediate(appendRetryImmediate(appendTimeout(Mono.fromFuture(() ->
                session.executeDataQuery(query, txModeToTxControl(txMode, true), params,
                        prepareExecuteDataQuerySettings()))
                .map(s -> YdbStatusUtils.checkResult(s, "Failed to execute YDB data query"))))),
                ydbMetrics::afterExecuteDataQuery);
    }

    @Override
    public Mono<DataQueryResult> executeDataQueryCommit(String query, TransactionMode txMode) {
        return AsyncMetrics.metric(appendTxRetryImmediate(appendRetryImmediate(appendTimeout(Mono.fromFuture(() ->
                session.executeDataQuery(query, txModeToTxControl(txMode, true), Params.empty(),
                        prepareExecuteDataQuerySettings()))
                .map(s -> YdbStatusUtils.checkResult(s, "Failed to execute YDB data query"))))),
                ydbMetrics::afterExecuteDataQuery);
    }

    @Override
    public Mono<DataQueryResult> executeDataQueryCommitRetryable(String query, TransactionMode txMode, Params params,
                                                                 YdbExecuteDataQuerySettings settings) {
        return AsyncMetrics.metric(appendIdempotentTxRetry(appendIdempotentRetry(appendTimeout(Mono.fromFuture(() ->
                session.executeDataQuery(query, txModeToTxControl(txMode, true), params,
                        prepareExecuteDataQuerySettings(settings)))
                .map(s -> YdbStatusUtils.checkResult(s, "Failed to execute YDB data query"))))),
                ydbMetrics::afterExecuteDataQuery);
    }

    @Override
    public Mono<DataQueryResult> executeDataQueryCommitRetryable(String query, TransactionMode txMode,
                                                                 Params params) {
        return AsyncMetrics.metric(appendIdempotentTxRetry(appendIdempotentRetry(appendTimeout(Mono.fromFuture(() ->
                session.executeDataQuery(query, txModeToTxControl(txMode, true), params,
                        prepareExecuteDataQuerySettings()))
                .map(s -> YdbStatusUtils.checkResult(s, "Failed to execute YDB data query"))))),
                ydbMetrics::afterExecuteDataQuery);
    }

    @Override
    public Mono<DataQueryResult> executeDataQueryCommitRetryable(String query, TransactionMode txMode) {
        return AsyncMetrics.metric(appendIdempotentTxRetry(appendIdempotentRetry(appendTimeout(Mono.fromFuture(() ->
                session.executeDataQuery(query, txModeToTxControl(txMode, true), Params.empty(),
                        prepareExecuteDataQuerySettings()))
                .map(s -> YdbStatusUtils.checkResult(s, "Failed to execute YDB data query"))))),
                ydbMetrics::afterExecuteDataQuery);
    }

    @Override
    public Mono<YdbTxDataQuery> prepareDataQuery(String query, YdbPrepareDataQuerySettings settings) {
        return AsyncMetrics.metric(appendIdempotentRetry(appendTimeout(Mono.fromFuture(() -> session
                .prepareDataQuery(query, preparePrepareDataQuerySettings(settings)))
                .map(s -> YdbStatusUtils.checkAnyResult(s.map(dataQuery -> new YdbTxDataQueryImpl(dataQuery, txControl,
                        queryTimeoutMillis, queryRetries, transactionRetries, retryCommit, ydbMetrics,
                                queryClientTimeout)),
                        "Failed to prepare YDB data query")))), ydbMetrics::afterPrepareDataQuery);
    }

    @Override
    public Mono<YdbTxDataQuery> prepareDataQuery(String query) {
        return AsyncMetrics.metric(appendIdempotentRetry(appendTimeout(Mono.fromFuture(() -> session
                .prepareDataQuery(query, preparePrepareDataQuerySettings()))
                .map(s -> YdbStatusUtils.checkAnyResult(s.map(dataQuery -> new YdbTxDataQueryImpl(dataQuery, txControl,
                        queryTimeoutMillis, queryRetries, transactionRetries, retryCommit, ydbMetrics,
                                queryClientTimeout)),
                        "Failed to prepare YDB data query")))), ydbMetrics::afterPrepareDataQuery);
    }

    @Override
    public Mono<ExplainDataQueryResult> explainDataQuery(String query) {
        return AsyncMetrics.metric(appendIdempotentRetry(appendTimeout(Mono.fromFuture(() -> session
                .explainDataQuery(query, prepareExplainDataQuerySettings()))
                .map(r -> YdbStatusUtils.checkAnyResult(r, "Failed to explain YDB data query")))),
                ydbMetrics::afterExplainDataQuery);
    }

    @Override
    @SuppressWarnings("FutureReturnValueIgnored")
    public Flux<ResultSetReader> readTable(String tablePath, YdbReadTableSettings settings) {
        return AsyncMetrics.metric(appendIdempotentRetry(appendTimeout(settings, Flux.defer(() -> Flux.create(sink ->
                session.readTable(tablePath, prepareReadTableSettings(settings), sink::next)
                .whenComplete((status, ex) -> {
                    if (ex != null) {
                        sink.error(YdbStatusUtils.checkException(ex));
                    } else {
                        if (status.isSuccess()) {
                            sink.complete();
                        } else {
                            sink.error(YdbStatusUtils.checkAnyStatusReturn(status, "Failed to read YDB table"));
                        }
                    }
                }))))), ydbMetrics::afterReadTable);
    }

    @Override
    public Mono<SessionStatus> keepAlive() {
        return AsyncMetrics.metric(appendIdempotentRetry(appendTimeout(Mono
                .fromFuture(() -> session.keepAlive(prepareKeepAliveSessionSettings()))
                .map(r -> YdbStatusUtils.checkAnyResult(r, "Failed YDB session keep-alive")))),
                ydbMetrics::afterSessionKeepAlive);
    }

    @Override
    public void invalidateQueryCache() {
        session.invalidateQueryCache();
    }

    @Override
    public <T> Mono<T> usingTxMono(TransactionMode transactionMode,
                                   Function<YdbTxSession, Mono<T>> body) {
        Mono<T> result = Mono.usingWhen(beginTransactionImpl(transactionMode), t ->
                        body.apply(initTxSession(TxControl.id(t).setCommitTx(false), false)), this::commitImpl,
                (t, e) -> rollbackImpl(t), this::rollbackImpl);
        if (transactionRetries > 0) {
            return result
                    .retryWhen(YdbRetry.retryTransaction(transactionRetries, ydbMetrics));
        } else {
            return result;
        }
    }

    @Override
    public <T> Flux<T> usingTxFlux(TransactionMode transactionMode,
                                   Function<YdbTxSession, Flux<T>> body) {
        Flux<T> result = Flux.usingWhen(beginTransactionImpl(transactionMode), t ->
                        body.apply(initTxSession(TxControl.id(t).setCommitTx(false), false)), this::commitImpl,
                (t, e) -> rollbackImpl(t), this::rollbackImpl);
        if (transactionRetries > 0) {
            return result
                    .retryWhen(YdbRetry.retryTransaction(transactionRetries, ydbMetrics));
        } else {
            return result;
        }
    }

    @Override
    public <T> Mono<T> usingTxMonoRetryable(TransactionMode transactionMode,
                                            Function<YdbTxSession, Mono<T>> body) {
        Mono<T> result = Mono.usingWhen(beginTransactionImpl(transactionMode),
                t -> body.apply(initTxSession(TxControl.id(t).setCommitTx(false), false)), this::commitImpl,
                (t, e) -> rollbackImpl(t), this::rollbackImpl);
        if (transactionRetries > 0) {
            return result
                    .retryWhen(YdbRetry.retryIdempotentTransaction(transactionRetries, ydbMetrics));
        } else {
            return result;
        }
    }

    @Override
    public <T> Flux<T> usingTxFluxRetryable(TransactionMode transactionMode,
                                            Function<YdbTxSession, Flux<T>> body) {
        Flux<T> result = Flux.usingWhen(beginTransactionImpl(transactionMode),
                t -> body.apply(initTxSession(TxControl.id(t).setCommitTx(false), false)), this::commitImpl,
                (t, e) -> rollbackImpl(t), this::rollbackImpl);
        if (transactionRetries > 0) {
            return result
                    .retryWhen(YdbRetry.retryIdempotentTransaction(transactionRetries, ydbMetrics));
        } else {
            return result;
        }
    }

    @Override
    public <U, V, W> Mono<W> usingCompTxMono(TransactionMode txMode,
                                             Function<YdbTxSession, Mono<Tuple2<U, String>>> preamble,
                                             BiFunction<YdbTxSession, U, Mono<V>> body,
                                             BiFunction<YdbTxSession, V, Mono<W>> trailer) {
        Mono<W> result = Mono.usingWhen(Mono.defer(() -> preamble.apply(initTxSession(txMode, false, false))),
                t -> body.apply(initTxSessionBody(t.getT2(), false), t.getT1())
                        .flatMap(v -> trailer.apply(initTxSessionTrailer(t.getT2(), false), v)),
                t -> Mono.empty(), (t, e) -> rollbackImpl(t.getT2()), t -> rollbackImpl(t.getT2()));
        if (transactionRetries > 0) {
            return result
                    .retryWhen(YdbRetry.retryTransaction(transactionRetries, ydbMetrics));
        } else {
            return result;
        }
    }

    @Override
    public <U, V, W> Flux<W> usingCompTxFlux(TransactionMode txMode,
                                             Function<YdbTxSession, Flux<Tuple2<U, String>>> preamble,
                                             BiFunction<YdbTxSession, U, Flux<V>> body,
                                             BiFunction<YdbTxSession, V, Flux<W>> trailer) {
        Flux<W> result = Flux.usingWhen(Flux.defer(() -> preamble.apply(initTxSession(txMode, false, false))),
                t -> body.apply(initTxSessionBody(t.getT2(), false), t.getT1())
                        .flatMap(v -> trailer.apply(initTxSessionTrailer(t.getT2(), false), v)),
                t -> Mono.empty(), (t, e) -> rollbackImpl(t.getT2()), t -> rollbackImpl(t.getT2()));
        if (transactionRetries > 0) {
            return result
                    .retryWhen(YdbRetry.retryTransaction(transactionRetries, ydbMetrics));
        } else {
            return result;
        }
    }

    @Override
    public <U, V, W> Mono<W> usingCompTxMonoRetryable(TransactionMode txMode,
                                                      Function<YdbTxSession, Mono<Tuple2<U, String>>> preamble,
                                                      BiFunction<YdbTxSession, U, Mono<V>> body,
                                                      BiFunction<YdbTxSession, V, Mono<W>> trailer) {
        Mono<W> result = Mono.usingWhen(Mono.defer(() -> preamble.apply(initTxSession(txMode, false, false))),
                t -> body.apply(initTxSessionBody(t.getT2(), false), t.getT1())
                        .flatMap(v -> trailer.apply(initTxSessionTrailer(t.getT2(), false), v)),
                t -> Mono.empty(), (t, e) -> rollbackImpl(t.getT2()), t -> rollbackImpl(t.getT2()));
        if (transactionRetries > 0) {
            return result
                    .retryWhen(YdbRetry.retryIdempotentTransaction(transactionRetries, ydbMetrics));
        } else {
            return result;
        }
    }

    @Override
    public <U, V, W> Flux<W> usingCompTxFluxRetryable(TransactionMode txMode,
                                                      Function<YdbTxSession, Flux<Tuple2<U, String>>> preamble,
                                                      BiFunction<YdbTxSession, U, Flux<V>> body,
                                                      BiFunction<YdbTxSession, V, Flux<W>> trailer) {
        Flux<W> result = Flux.usingWhen(Flux.defer(() -> preamble.apply(initTxSession(txMode, false, false))),
                t -> body.apply(initTxSessionBody(t.getT2(), false), t.getT1())
                        .flatMap(v -> trailer.apply(initTxSessionTrailer(t.getT2(), false), v)),
                t -> Mono.empty(), (t, e) -> rollbackImpl(t.getT2()), t -> rollbackImpl(t.getT2()));
        if (transactionRetries > 0) {
            return result
                    .retryWhen(YdbRetry.retryIdempotentTransaction(transactionRetries, ydbMetrics));
        } else {
            return result;
        }
    }

    @Override
    public TxControl getTxControl() {
        return txControl;
    }

    @Override
    public Mono<Void> commitTransaction() {
        return AsyncMetrics.metric(appendRetry(appendTimeout(Mono.fromFuture(() -> session
                .commitTransaction(txControl.toPb().getTxId(), prepareCommitTxSettings()))
                .doOnSuccess(r -> YdbStatusUtils.checkStatus(r, "Failed to commit YDB transaction")).then())),
                ydbMetrics::afterCommitTransaction);
    }

    @Override
    public Mono<Void> rollbackTransaction() {
        return AsyncMetrics.metric(appendRetry(appendTimeout(Mono.fromFuture(() -> session
                .rollbackTransaction(txControl.toPb().getTxId(), prepareRollbackTxSettings()))
                .doOnSuccess(r -> YdbStatusUtils.checkAnyStatus(r, "Failed to rollback YDB transaction")).then())),
                ydbMetrics::afterRollbackTransaction);
    }

    @Override
    public String toString() {
        return session.toString();
    }

    private TxControl txModeToTxControl(TransactionMode txMode, boolean commit) {
        switch (txMode) {
            case SERIALIZABLE_READ_WRITE:
                return TxControl.serializableRw().setCommitTx(commit);
            case ONLINE_READ_ONLY:
                return TxControl.onlineRo().setCommitTx(commit);
            case STALE_READ_ONLY:
                return TxControl.staleRo().setCommitTx(commit);
            default:
                throw new IllegalArgumentException("Unexpected transaction mode: " + txMode);
        }
    }

    private YdbTxSession initTxSession(TransactionMode txMode, boolean commit, boolean retryCommit) {
        return initTxSession(txModeToTxControl(txMode, commit), retryCommit);
    }

    private YdbTxSession initTxSession(TxControl txControl, boolean retryCommit) {
        return new YdbTxSessionImpl(session, txControl, queryTimeoutMillis, queryRetries, transactionRetries,
                retryCommit, ydbMetrics, queryClientTimeout);
    }

    private YdbTxSession initTxSessionBody(String txId, boolean retryCommit) {
        return initTxSession(TxControl.id(txId).setCommitTx(false), retryCommit);
    }

    private YdbTxSession initTxSessionTrailer(String txId, boolean retryCommit) {
        return initTxSession(TxControl.id(txId).setCommitTx(true), retryCommit);
    }

    private Mono<Transaction> beginTransactionImpl(TransactionMode transactionMode) {
        return appendRetry(appendTimeout(Mono.fromFuture(() -> session.beginTransaction(transactionMode))
                .doOnDiscard(com.yandex.ydb.core.Result.class, this::onDiscardedTransaction)
                .map(r -> YdbStatusUtils.checkAnyResult(r, "Failed to begin YDB transaction"))));
    }

    private Mono<Void> commitImpl(Transaction transaction) {
        return appendRetry(appendTimeout(Mono.fromFuture(transaction::commit)
                .doOnSuccess(s -> YdbStatusUtils.checkStatus(s, "Failed to commit YDB transaction")).then()));
    }

    private Mono<Void> rollbackImpl(Transaction transaction) {
        return appendRetryTimeout(appendRetry(appendTimeout(Mono.fromFuture(transaction::rollback)
                .doOnSuccess(s -> YdbStatusUtils.checkAnyStatus(s, "Failed to rollback YDB transaction")))))
                .doOnError(e -> LOG.warn("Failed to rollback YDB transaction", e))
                .then()
                .onErrorResume(e -> Mono.empty());
    }

    private Mono<Void> rollbackImpl(String txId) {
        return appendRetryTimeout(appendRetry(appendTimeout(Mono.fromFuture(() -> session
                .rollbackTransaction(txId, prepareRollbackTxSettings()))
                .doOnSuccess(s -> YdbStatusUtils.checkAnyStatus(s, "Failed to rollback YDB transaction")))))
                .doOnError(e -> LOG.warn("Failed to rollback YDB transaction", e))
                .then()
                .onErrorResume(e -> Mono.empty());
    }

    private <T> Mono<T> appendRetry(Mono<T> publisher) {
        if (queryRetries > 0) {
            return publisher.retryWhen(YdbRetry.retryRequest(queryRetries, ydbMetrics));
        } else {
            return publisher;
        }
    }

    private <T> Mono<T> appendRetryImmediate(Mono<T> publisher) {
        if (queryRetries > 0) {
            if (retryCommit) {
                return publisher.retryWhen(YdbRetry.retryIdempotentRequest(queryRetries, ydbMetrics));
            } else {
                return publisher.retryWhen(YdbRetry.retryRequest(queryRetries, ydbMetrics));
            }
        } else {
            return publisher;
        }
    }

    private <T> Mono<T> appendIdempotentRetry(Mono<T> publisher) {
        if (queryRetries > 0) {
            return publisher.retryWhen(YdbRetry.retryRequest(queryRetries, ydbMetrics));
        } else {
            return publisher;
        }
    }

    private <T> Mono<T> appendIdempotentTxRetry(Mono<T> publisher) {
        if (transactionRetries > 0 && txControl.isCommitTx() && txControl.toPb().hasBeginTx()) {
            return publisher.retryWhen(YdbRetry.retryIdempotentTransaction(transactionRetries, ydbMetrics));
        } else {
            return publisher;
        }
    }

    private <T> Mono<T> appendTxRetryImmediate(Mono<T> publisher) {
        if (transactionRetries > 0 && txControl.isCommitTx() && txControl.toPb().hasBeginTx()) {
            if (retryCommit) {
                return publisher.retryWhen(YdbRetry.retryIdempotentTransaction(transactionRetries, ydbMetrics));
            } else {
                return publisher.retryWhen(YdbRetry.retryTransaction(transactionRetries, ydbMetrics));
            }
        } else {
            return publisher;
        }
    }

    private <T> Flux<T> appendRetry(Flux<T> publisher) {
        if (queryRetries > 0) {
            return publisher.retryWhen(YdbRetry.retryRequest(queryRetries, ydbMetrics));
        } else {
            return publisher;
        }
    }

    private <T> Flux<T> appendIdempotentRetry(Flux<T> publisher) {
        if (queryRetries > 0) {
            return publisher.retryWhen(YdbRetry.retryIdempotentRequest(queryRetries, ydbMetrics));
        } else {
            return publisher;
        }
    }

    private <T> Mono<T> appendRetryTimeout(Mono<T> publisher) {
        if (queryRetries > 0) {
            return publisher.retryWhen(YdbRetry.retryTimeout(queryRetries, ydbMetrics));
        } else {
            return publisher;
        }
    }

    private <T> Mono<T> appendTimeout(Mono<T> publisher) {
        if (!queryClientTimeout.isNegative() && !queryClientTimeout.isZero()) {
            return publisher.timeout(queryClientTimeout);
        } else {
            return publisher;
        }
    }

    private <T> Flux<T> appendTimeout(YdbReadTableSettings ydbReadTableSettings, Flux<T> publisher) {
        if (ydbReadTableSettings.getTimeout().isPresent()
                && !ydbReadTableSettings.getTimeout().get().isNegative()
                && !ydbReadTableSettings.getTimeout().get().isZero()) {
            return publisher.timeout(ydbReadTableSettings.getTimeout().get());
        } else if (!queryClientTimeout.isNegative() && !queryClientTimeout.isZero()) {
            return publisher.timeout(queryClientTimeout);
        } else {
            return publisher;
        }
    }

    private ExecuteDataQuerySettings prepareExecuteDataQuerySettings() {
        return prepareExecuteDataQuerySettings(YdbExecuteDataQuerySettings.builder().build());
    }

    private ExecuteDataQuerySettings prepareExecuteDataQuerySettings(
            YdbExecuteDataQuerySettings ydbExecuteDataQuerySettings) {
        ExecuteDataQuerySettings settings = new ExecuteDataQuerySettings();
        if (ydbExecuteDataQuerySettings.isKeepInQueryCache()) {
            settings.keepInQueryCache();
        }
        String logId = MDC.get(MdcTaskDecorator.LOG_ID_MDC_KEY);
        if (logId != null) {
            settings.setTraceId(logId);
        }
        if (queryTimeoutMillis > 0) {
            settings.setTimeout(queryTimeoutMillis, TimeUnit.MILLISECONDS);
        }
        return settings;
    }

    private PrepareDataQuerySettings preparePrepareDataQuerySettings() {
        return preparePrepareDataQuerySettings(YdbPrepareDataQuerySettings.builder().build());
    }

    private PrepareDataQuerySettings preparePrepareDataQuerySettings(
            YdbPrepareDataQuerySettings ydbPrepareDataQuerySettings) {
        PrepareDataQuerySettings settings = new PrepareDataQuerySettings();
        if (ydbPrepareDataQuerySettings.isKeepInQueryCache()) {
            settings.keepInQueryCache();
        }
        String logId = MDC.get(MdcTaskDecorator.LOG_ID_MDC_KEY);
        if (logId != null) {
            settings.setTraceId(logId);
        }
        if (queryTimeoutMillis > 0) {
            settings.setTimeout(queryTimeoutMillis, TimeUnit.MILLISECONDS);
        }
        return settings;
    }

    private ExplainDataQuerySettings prepareExplainDataQuerySettings() {
        ExplainDataQuerySettings settings = new ExplainDataQuerySettings();
        String logId = MDC.get(MdcTaskDecorator.LOG_ID_MDC_KEY);
        if (logId != null) {
            settings.setTraceId(logId);
        }
        if (queryTimeoutMillis > 0) {
            settings.setTimeout(queryTimeoutMillis, TimeUnit.MILLISECONDS);
        }
        return settings;
    }

    private CommitTxSettings prepareCommitTxSettings() {
        CommitTxSettings settings = new CommitTxSettings();
        String logId = MDC.get(MdcTaskDecorator.LOG_ID_MDC_KEY);
        if (logId != null) {
            settings.setTraceId(logId);
        }
        if (queryTimeoutMillis > 0) {
            settings.setTimeout(queryTimeoutMillis, TimeUnit.MILLISECONDS);
        }
        return settings;
    }

    private RollbackTxSettings prepareRollbackTxSettings() {
        RollbackTxSettings settings = new RollbackTxSettings();
        String logId = MDC.get(MdcTaskDecorator.LOG_ID_MDC_KEY);
        if (logId != null) {
            settings.setTraceId(logId);
        }
        if (queryTimeoutMillis > 0) {
            settings.setTimeout(queryTimeoutMillis, TimeUnit.MILLISECONDS);
        }
        return settings;
    }

    private ReadTableSettings prepareReadTableSettings(YdbReadTableSettings ydbReadTableSettings) {
        ReadTableSettings.Builder builder = ReadTableSettings.newBuilder();
        if (ydbReadTableSettings.getTimeout().isPresent()) {
            builder.timeout(ydbReadTableSettings.getTimeout().get());
        } else {
            if (queryTimeoutMillis > 0) {
                builder.timeout(queryTimeoutMillis, TimeUnit.MILLISECONDS);
            }
        }
        if (!ydbReadTableSettings.getColumns().isEmpty()) {
            builder.columns(ydbReadTableSettings.getColumns());
        }
        builder.orderedRead(ydbReadTableSettings.isOrdered());
        ydbReadTableSettings.getFromKey().ifPresent(k -> builder.fromKey(k, ydbReadTableSettings.isFromInclusive()));
        ydbReadTableSettings.getToKey().ifPresent(k -> builder.toKey(k, ydbReadTableSettings.isToInclusive()));
        builder.rowLimit(ydbReadTableSettings.getRowLimit());
        return builder.build();
    }

    private KeepAliveSessionSettings prepareKeepAliveSessionSettings() {
        KeepAliveSessionSettings settings = new KeepAliveSessionSettings();
        String logId = MDC.get(MdcTaskDecorator.LOG_ID_MDC_KEY);
        if (logId != null) {
            settings.setTraceId(logId);
        }
        if (queryTimeoutMillis > 0) {
            settings.setTimeout(queryTimeoutMillis, TimeUnit.MILLISECONDS);
        }
        return settings;
    }

    @SuppressWarnings("rawtypes")
    private void onDiscardedTransaction(com.yandex.ydb.core.Result r) {
        if (r == null) {
            return;
        }
        if (!r.isSuccess()) {
            return;
        }
        Optional valueO = r.ok();
        if (valueO.isEmpty()) {
            return;
        }
        Object value = valueO.get();
        if (!(value instanceof Transaction)) {
            return;
        }
        LOG.info("YDB transaction subscription was discarded");
        try {
            ((Transaction) value).rollback().whenComplete((s, e) -> {
                if (s != null && !s.isSuccess()) {
                    LOG.warn("Failed to rollback discarded YDB transaction: {}", s);
                }
                if (e != null) {
                    LOG.warn("Failed to rollback discarded YDB transaction", e);
                }
            });
        } catch (Exception e) {
            LOG.warn("Failed to rollback discarded YDB transaction", e);
        }
    }

}
