package ru.yandex.intranet.d.datasource.migrations.dao;

import java.time.Instant;
import java.util.Comparator;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import com.yandex.ydb.table.query.DataQueryResult;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.transaction.TransactionMode;
import com.yandex.ydb.table.values.PrimitiveValue;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;
import ru.yandex.intranet.d.datasource.Ydb;
import ru.yandex.intranet.d.datasource.impl.YdbQuerySource;
import ru.yandex.intranet.d.datasource.migrations.model.DbBootstrap;
import ru.yandex.intranet.d.datasource.model.WithTxId;
import ru.yandex.intranet.d.datasource.model.YdbSchemeClient;
import ru.yandex.intranet.d.datasource.model.YdbSession;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;

/**
 * Bootstrap progress DAO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class BootstrapProgressDao {

    private final YdbQuerySource ydbQuerySource;
    private final YdbSchemeClient ydbSchemeClient;
    private final String tablePathPrefix;

    public BootstrapProgressDao(YdbQuerySource ydbQuerySource,
                                YdbSchemeClient ydbSchemeClient,
                                @Value("${ydb.tablePathPrefix}") String tablePathPrefix) {
        this.ydbQuerySource = ydbQuerySource;
        this.ydbSchemeClient = ydbSchemeClient;
        this.tablePathPrefix = tablePathPrefix;
    }

    public Mono<Boolean> isBootstrapProgressInitialized() {
        return ydbSchemeClient.listDirectory(tablePathPrefix).map(r -> r.getChildren().stream()
                .anyMatch(e -> e.getName().equals("bootstrap_progress")));
    }

    public Mono<List<DbBootstrap>> skipAppliedBootstraps(YdbSession session, List<DbBootstrap> bootstraps) {
        String query = ydbQuerySource.getQuery("yql.queries.bootstrap_progress.findMax");
        return session.executeDataQueryCommitRetryable(query, TransactionMode.SERIALIZABLE_READ_WRITE, Params.empty())
                .map(this::toMax).map(max -> {
                    if (max.isEmpty()) {
                        return bootstraps;
                    }
                    return bootstraps.stream().sorted(Comparator.comparing(DbBootstrap::getOrder))
                            .filter(b -> b.getOrder() > max.get()).collect(Collectors.toList());
                });
    }

    public Mono<Void> saveAppliedBootstrapRetryable(YdbTxSession session, long order) {
        String query = ydbQuerySource.getQuery("yql.queries.bootstrap_progress.insertProgress");
        PrimitiveValue id = PrimitiveValue.int64(order);
        PrimitiveValue timestamp = PrimitiveValue.int64(Instant.now().toEpochMilli());
        return session.executeDataQueryRetryable(query, Params.of("$id", id, "$applied_timestamp", timestamp)).then();
    }

    public Mono<Void> saveAppliedBootstrapAndCommit(YdbSession session, long order) {
        String query = ydbQuerySource.getQuery("yql.queries.bootstrap_progress.insertProgress");
        PrimitiveValue id = PrimitiveValue.int64(order);
        PrimitiveValue timestamp = PrimitiveValue.int64(Instant.now().toEpochMilli());
        return session.executeDataQueryCommitRetryable(query, TransactionMode.SERIALIZABLE_READ_WRITE,
                Params.of("$id", id, "$applied_timestamp", timestamp)).then();
    }

    public Mono<WithTxId<Boolean>> existsByOrderStartTx(YdbTxSession session, long order) {
        String query = ydbQuerySource.getQuery("yql.queries.bootstrap_progress.existsById");
        PrimitiveValue id = PrimitiveValue.int64(order);
        return session.executeDataQueryRetryable(query, Params.of("$id", id))
                .map(result -> new WithTxId<>(toExists(result), result.getTxId()));
    }

    private Optional<Long> toMax(DataQueryResult result) {
        if (result.isEmpty() || result.getResultSetCount() > 1) {
            throw new IllegalStateException("Exactly one result set is required");
        }
        ResultSetReader reader = result.getResultSet(0);
        if (!reader.next() || reader.getRowCount() > 1) {
            throw new IllegalStateException("Exactly one result is required");
        }
        return Optional.ofNullable(Ydb.int64OrNull(reader.getColumn("max_id")));
    }

    private boolean toExists(DataQueryResult result) {
        if (result.isEmpty() || result.getResultSetCount() > 1) {
            throw new IllegalStateException("Exactly one result set is required");
        }
        ResultSetReader reader = result.getResultSet(0);
        if (!reader.next() || reader.getRowCount() > 1) {
            throw new IllegalStateException("Exactly one result is required");
        }
        return reader.getColumn("row_exists").getBool();
    }

}
