package ru.yandex.intranet.d.datasource.migrations.impl;

import com.yandex.ydb.table.transaction.TransactionMode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.core.publisher.Mono;
import ru.yandex.intranet.d.datasource.migrations.dao.BootstrapProgressDao;
import ru.yandex.intranet.d.datasource.migrations.model.DbBootstrap;
import ru.yandex.intranet.d.datasource.migrations.model.MigrationType;
import ru.yandex.intranet.d.datasource.model.WithTxId;
import ru.yandex.intranet.d.datasource.model.YdbSession;

/**
 * Database bootstrap resource.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public class DbBootstrapResource implements DbBootstrap {

    private static final Logger LOG = LoggerFactory.getLogger(DbBootstrapResource.class);

    private final String text;
    private final long order;
    private final MigrationType migrationType;

    public DbBootstrapResource(String text, long order, MigrationType migrationType) {
        this.text = text;
        this.order = order;
        this.migrationType = migrationType;
    }

    @Override
    public Mono<Void> run(YdbSession session, BootstrapProgressDao bootstrapProgressDao) {
        if (migrationType == MigrationType.DDL) {
            return session.executeSchemeQuery(text)
                    .then(bootstrapProgressDao.saveAppliedBootstrapAndCommit(session, order));
        } else {
            return session.usingCompTxMonoRetryable(TransactionMode.SERIALIZABLE_READ_WRITE,
                    ts -> bootstrapProgressDao.existsByOrderStartTx(ts, order).map(WithTxId::asTuple),
                    (ts, alreadyApplied) -> {
                        if (!alreadyApplied) {
                            return ts.executeDataQuery(text).thenReturn(false);
                        } else {
                            LOG.info("Bootstrap sequence {} was already applied", order);
                            return Mono.just(true);
                        }
                    },
                    (ts, alreadyApplied) -> {
                        if (alreadyApplied) {
                            return ts.commitTransaction();
                        } else {
                            return bootstrapProgressDao.saveAppliedBootstrapRetryable(ts, order);
                        }
                    });
        }
    }

    @Override
    public long getOrder() {
        return order;
    }

    @Override
    public MigrationType getType() {
        return migrationType;
    }

}
