package ru.yandex.intranet.d.datasource.migrations.impl;

import com.yandex.ydb.table.transaction.TransactionMode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.core.publisher.Mono;
import ru.yandex.intranet.d.datasource.migrations.dao.RecipeMigrationsProgressDao;
import ru.yandex.intranet.d.datasource.migrations.model.DbRecipeMigration;
import ru.yandex.intranet.d.datasource.migrations.model.MigrationType;
import ru.yandex.intranet.d.datasource.model.WithTxId;
import ru.yandex.intranet.d.datasource.model.YdbSession;

/**
 * Recipe database migration resource.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public class DbRecipeMigrationResource implements DbRecipeMigration {

    private static final Logger LOG = LoggerFactory.getLogger(DbRecipeMigrationResource.class);

    private final String text;
    private final long version;
    private final MigrationType migrationType;

    public DbRecipeMigrationResource(String text, long version, MigrationType migrationType) {
        this.text = text;
        this.version = version;
        this.migrationType = migrationType;
    }

    @Override
    public Mono<Void> run(YdbSession session, RecipeMigrationsProgressDao recipeMigrationsProgressDao) {
        if (migrationType == MigrationType.DDL) {
            return session.executeSchemeQuery(text)
                    .then(recipeMigrationsProgressDao.saveAppliedMigrationAndCommit(session, version));
        } else {
            return session.usingCompTxMonoRetryable(TransactionMode.SERIALIZABLE_READ_WRITE,
                    ts -> recipeMigrationsProgressDao.existsByVersionStartTx(ts, version).map(WithTxId::asTuple),
                    (ts, alreadyApplied) -> {
                        if (!alreadyApplied) {
                            return ts.executeDataQuery(text).thenReturn(false);
                        } else {
                            LOG.info("Recipe migration {} was already applied", version);
                            return Mono.just(true);
                        }
                    },
                    (ts, alreadyApplied) -> {
                        if (alreadyApplied) {
                            return ts.commitTransaction();
                        } else {
                            return recipeMigrationsProgressDao.saveAppliedMigrationRetryable(ts, version);
                        }
                    });
        }
    }

    @Override
    public long getVersion() {
        return version;
    }

    @Override
    public MigrationType getType() {
        return migrationType;
    }

}
