package ru.yandex.intranet.d.datasource.migrations.impl;

import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.nio.charset.StandardCharsets;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.base.Splitter;
import com.google.common.primitives.Longs;
import org.springframework.core.io.Resource;
import org.springframework.core.io.support.PathMatchingResourcePatternResolver;
import org.springframework.stereotype.Component;
import org.springframework.util.StreamUtils;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import reactor.core.scheduler.Schedulers;

import ru.yandex.intranet.d.datasource.impl.YdbQuerySource;
import ru.yandex.intranet.d.datasource.migrations.model.DbBootstrap;
import ru.yandex.intranet.d.datasource.migrations.model.DbMigration;
import ru.yandex.intranet.d.datasource.migrations.model.DbRecipeMigration;
import ru.yandex.intranet.d.datasource.migrations.model.MigrationType;

/**
 * Migrations provider.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class MigrationsProvider {

    private final YdbQuerySource ydbQuerySource;

    public MigrationsProvider(YdbQuerySource ydbQuerySource) {
        this.ydbQuerySource = ydbQuerySource;
    }

    public Mono<List<DbBootstrap>> getBootstrap() {
        PathMatchingResourcePatternResolver resourceResolver = new PathMatchingResourcePatternResolver();
        return Mono.just("classpath:db/bootstrap/*.yql")
                .publishOn(Schedulers.boundedElastic())
                .map(pattern -> getResources(resourceResolver, pattern))
                .<List<DbBootstrap>>flatMap(resources -> {
                    if (resources.isEmpty()) {
                        return Mono.just(List.of());
                    }
                    return Flux.fromIterable(resources)
                            .publishOn(Schedulers.boundedElastic())
                            .<DbBootstrap>map(this::getDbBootstrapResource)
                            .collectList();
                }).flatMap(l -> {
                    List<Long> orders = l.stream().map(DbBootstrap::getOrder).collect(Collectors.toList());
                    Set<Long> uniqueOrders = new HashSet<>(orders);
                    if (uniqueOrders.size() != orders.size()) {
                        return Mono.error(new IllegalStateException("Duplicate bootstraps found"));
                    }
                    return Mono.just(l);
                });
    }

    public Mono<List<DbMigration>> getMigrations() {
        PathMatchingResourcePatternResolver resourceResolver = new PathMatchingResourcePatternResolver();
        return Mono.just("classpath:db/migrations/*.yql")
                .publishOn(Schedulers.boundedElastic())
                .map(pattern -> getResources(resourceResolver, pattern))
                .<List<DbMigration>>flatMap(resources -> {
                    if (resources.isEmpty()) {
                        return Mono.just(List.of());
                    }
                    return Flux.fromIterable(resources)
                            .publishOn(Schedulers.boundedElastic())
                            .<DbMigration>map(this::getDbMigrationResource)
                            .collectList();
                }).flatMap(l -> {
                    List<Long> orders = l.stream().map(DbMigration::getVersion).collect(Collectors.toList());
                    Set<Long> uniqueOrders = new HashSet<>(orders);
                    if (uniqueOrders.size() != orders.size()) {
                        return Mono.error(new IllegalStateException("Duplicate migrations found"));
                    }
                    return Mono.just(l);
                });
    }

    public Mono<List<DbRecipeMigration>> getRecipeMigrations() {
        PathMatchingResourcePatternResolver resourceResolver = new PathMatchingResourcePatternResolver();
        return Mono.just("classpath:db/recipe/*.yql")
                .publishOn(Schedulers.boundedElastic())
                .map(pattern -> getResources(resourceResolver, pattern))
                .<List<DbRecipeMigration>>flatMap(resources -> {
                    if (resources.isEmpty()) {
                        return Mono.just(List.of());
                    }
                    return Flux.fromIterable(resources)
                            .publishOn(Schedulers.boundedElastic())
                            .<DbRecipeMigration>map(this::getRecipeMigrationResource)
                            .collectList();
                }).flatMap(l -> {
                    List<Long> orders = l.stream().map(DbRecipeMigration::getVersion).collect(Collectors.toList());
                    Set<Long> uniqueOrders = new HashSet<>(orders);
                    if (uniqueOrders.size() != orders.size()) {
                        return Mono.error(new IllegalStateException("Duplicate recipe migrations found"));
                    }
                    return Mono.just(l);
                });
    }

    private DbBootstrapResource getDbBootstrapResource(Resource resource) {
        String resourceName = getResourceName(resource, "bootstrap");
        List<String> parts = getNameParts(resourceName, "bootstrap");
        Long version = getResourceNumber(parts.get(0), resourceName, "bootstrap");
        MigrationType migrationType = getResourceType(parts.get(1), resourceName, "bootstrap");
        String text = resourceToString(resource);
        return new DbBootstrapResource(text, version, migrationType);
    }

    private DbMigrationResource getDbMigrationResource(Resource resource) {
        String resourceName = getResourceName(resource, "migration");
        List<String> parts = getNameParts(resourceName, "migration");
        Long version = getResourceNumber(parts.get(0), resourceName, "migration");
        MigrationType migrationType = getResourceType(parts.get(1), resourceName, "migration");
        String text = resourceToString(resource);
        return new DbMigrationResource(text, version, migrationType);
    }

    private DbRecipeMigrationResource getRecipeMigrationResource(Resource resource) {
        String resourceName = getResourceName(resource, "recipe migration");
        List<String> parts = getNameParts(resourceName, "recipe migration");
        Long version = getResourceNumber(parts.get(0), resourceName, "recipe migration");
        MigrationType migrationType = getResourceType(parts.get(1), resourceName, "recipe migration");
        String text = resourceToString(resource);
        return new DbRecipeMigrationResource(text, version, migrationType);
    }

    private String getResourceName(Resource resource, String type) {
        String resourceName = resource.getFilename();
        if (resourceName == null) {
            throw new IllegalStateException("Invalid " + type + " name: null");
        }
        return resourceName;
    }

    private List<String> getNameParts(String resourceName, String type) {
        List<String> parts = Splitter.on("-").splitToList(resourceName);
        if (parts.size() < 2 || parts.get(0).isEmpty() || parts.get(1).isEmpty()) {
            throw new IllegalStateException("Invalid " + type + " name: " + resourceName);
        }
        return parts;
    }

    private Long getResourceNumber(String numberPart, String resourceName, String type) {
        Long version = Longs.tryParse(numberPart);
        if (version == null) {
            throw new IllegalStateException("Invalid " + type + " name: " + resourceName);
        }
        return version;
    }

    private MigrationType getResourceType(String partType, String resourceName, String type) {
        if (!partType.equalsIgnoreCase("ddl") && !partType.equalsIgnoreCase("dml")) {
            throw new IllegalStateException("Invalid " + type + " name: " + resourceName);
        }
        return partType.equalsIgnoreCase("ddl")
                ? MigrationType.DDL : MigrationType.DML;
    }

    private List<Resource> getResources(PathMatchingResourcePatternResolver resourceResolver,
                                        String locationPattern) {
        try {
            return List.of(resourceResolver.getResources(locationPattern));
        } catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    private String resourceToString(Resource resource) {
        try (InputStream stream = resource.getInputStream()) {
            return ydbQuerySource.preprocessRawQuery(StreamUtils.copyToString(stream, StandardCharsets.UTF_8));
        } catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

}
