package ru.yandex.intranet.d.datasource.migrations.impl;

import java.util.Comparator;
import java.util.List;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.datasource.impl.YdbRetry;
import ru.yandex.intranet.d.datasource.migrations.dao.BootstrapProgressDao;
import ru.yandex.intranet.d.datasource.migrations.dao.RecipeMigrationsProgressDao;
import ru.yandex.intranet.d.datasource.migrations.dao.SchemaVersionsDao;
import ru.yandex.intranet.d.datasource.migrations.model.DbBootstrap;
import ru.yandex.intranet.d.datasource.migrations.model.DbMigration;
import ru.yandex.intranet.d.datasource.migrations.model.DbRecipeMigration;
import ru.yandex.intranet.d.datasource.model.YdbSession;
import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.metrics.YdbMetrics;

/**
 * Migrations executor.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class MigrationsRunner {

    private static final Logger LOG = LoggerFactory.getLogger(MigrationsRunner.class);

    private final SchemaVersionsDao schemaVersionsDao;
    private final BootstrapProgressDao bootstrapProgressDao;
    private final RecipeMigrationsProgressDao recipeMigrationsProgressDao;
    private final MigrationsProvider migrationsProvider;
    private final YdbTableClient tableClient;
    private final YdbMetrics ydbMetrics;

    public MigrationsRunner(SchemaVersionsDao schemaVersionsDao, BootstrapProgressDao bootstrapProgressDao,
                            RecipeMigrationsProgressDao recipeMigrationsProgressDao,
                            MigrationsProvider migrationsProvider, YdbTableClient tableClient, YdbMetrics ydbMetrics) {
        this.schemaVersionsDao = schemaVersionsDao;
        this.bootstrapProgressDao = bootstrapProgressDao;
        this.recipeMigrationsProgressDao = recipeMigrationsProgressDao;
        this.migrationsProvider = migrationsProvider;
        this.tableClient = tableClient;
        this.ydbMetrics = ydbMetrics;
    }

    public Mono<Void> doBootstrap() {
        return Mono.fromRunnable(() -> LOG.info("Preparing for YDB bootstrap..."))
                .then(migrationsProvider.getBootstrap()
                        .doOnSuccess(m -> LOG.info("Total {} YDB bootstraps found", m.size()))
                        .flatMap(this::executeBootstraps)
                        .doOnSuccess(v -> LOG.info("YDB bootstrap successfully finished")));
    }

    public Mono<Void> applyMigrations() {
        return Mono.fromRunnable(() -> LOG.info("Preparing for YDB migrations..."))
                .then(migrationsProvider.getMigrations()
                        .doOnSuccess(m -> LOG.info("Total {} YDB migrations found", m.size()))
                        .flatMap(this::executeMigrations)
                        .doOnSuccess(v -> LOG.info("YDB migrations successfully applied")));
    }

    public Mono<Void> applyRecipeMigrations() {
        return Mono.fromRunnable(() -> LOG.info("Preparing for YDB recipe migrations..."))
                .then(migrationsProvider.getRecipeMigrations()
                        .doOnSuccess(m -> LOG.info("Total {} YDB recipe migrations found", m.size()))
                        .flatMap(this::executeRecipeMigrations)
                        .doOnSuccess(v -> LOG.info("YDB recipe migrations successfully applied")));
    }

    public Mono<Integer> countPendingBootstraps() {
        return migrationsProvider.getBootstrap()
                .flatMap(bootstraps -> skipAppliedBootstraps(bootstraps).map(List::size));
    }

    public Mono<Integer> countPendingMigrations() {
        return migrationsProvider.getMigrations()
                .flatMap(migrations -> skipAppliedMigrations(migrations).map(List::size));
    }

    public Mono<List<DbBootstrap>> getPendingBootstraps() {
        return migrationsProvider.getBootstrap().flatMap(this::skipAppliedBootstraps);
    }

    public Mono<List<DbMigration>> getPendingMigrations() {
        return migrationsProvider.getMigrations().flatMap(this::skipAppliedMigrations);
    }

    public Mono<Void> applyBootstrap(DbBootstrap bootstrap) {
        return tableClient.usingSessionMonoRetryable(session -> bootstrap.run(session, bootstrapProgressDao));
    }

    public Mono<Void> applyMigration(DbMigration migration) {
        return tableClient.usingSessionMonoRetryable(session -> migration.run(session, schemaVersionsDao));
    }

    private Mono<List<DbBootstrap>> skipAppliedBootstraps(List<DbBootstrap> bootstraps) {
        return tableClient.usingSessionMonoRetryable(session ->
                skipAppliedBootstrap(session, bootstraps)
                .map(m -> m.stream().sorted(Comparator.comparing(DbBootstrap::getOrder))
                .collect(Collectors.toList())));
    }

    private Mono<List<DbMigration>> skipAppliedMigrations(List<DbMigration> migrations) {
        return tableClient.usingSessionMonoRetryable(session ->
                skipAppliedMigrations(session, migrations)
                .map(m -> m.stream().sorted(Comparator.comparing(DbMigration::getVersion))
                        .collect(Collectors.toList())));
    }

    private Mono<Void> executeBootstraps(List<DbBootstrap> bootstraps) {
        return tableClient.usingSessionMonoRetryable(session ->
                skipAppliedBootstrap(session, bootstraps)
                        .map(m -> m.stream().sorted(Comparator.comparing(DbBootstrap::getOrder))
                                .collect(Collectors.toList()))
                        .doOnSuccess(m -> LOG.info("Total {} YDB bootstraps to apply", m.size()))
                        .flatMapMany(Flux::fromIterable)
                        .concatMap(m -> runBootstrap(session, m))
                        .thenEmpty(Mono.empty())).retryWhen(YdbRetry.retryTimeout(1, ydbMetrics));
    }

    private Mono<List<DbBootstrap>> skipAppliedBootstrap(YdbSession session, List<DbBootstrap> bootstraps) {
        return bootstrapProgressDao.isBootstrapProgressInitialized().flatMap(initialized -> {
            if (initialized) {
                return bootstrapProgressDao.skipAppliedBootstraps(session, bootstraps);
            } else {
                return Mono.just(bootstraps);
            }
        });
    }

    private Mono<List<DbMigration>> skipAppliedMigrations(YdbSession session, List<DbMigration> migrations) {
        return schemaVersionsDao.isSchemaVersionsInitialized().flatMap(initialized -> {
            if (initialized) {
                return schemaVersionsDao.skipAppliedMigrations(session, migrations);
            } else {
                return Mono.just(migrations);
            }
        });
    }

    private Mono<List<DbRecipeMigration>> skipAppliedRecipeMigrations(YdbSession session,
                                                                List<DbRecipeMigration> migrations) {
        return recipeMigrationsProgressDao.isRecipeMigrationsProgressInitialized().flatMap(initialized -> {
            if (initialized) {
                return recipeMigrationsProgressDao.skipAppliedMigrations(session, migrations);
            } else {
                return Mono.just(migrations);
            }
        });
    }

    private Mono<Void> executeMigrations(List<DbMigration> migrations) {
        return tableClient.usingSessionMonoRetryable(session ->
                skipAppliedMigrations(session, migrations)
                        .map(m -> m.stream().sorted(Comparator.comparing(DbMigration::getVersion))
                                .collect(Collectors.toList()))
                        .doOnSuccess(m -> LOG.info("Total {} YDB migrations to apply", m.size()))
                        .flatMapMany(Flux::fromIterable)
                        .concatMap(m -> runMigration(session, m))
                        .thenEmpty(Mono.empty())).retryWhen(YdbRetry.retryTimeout(1, ydbMetrics));
    }

    private Mono<Void> runBootstrap(YdbSession session, DbBootstrap bootstrap) {
        return Mono.fromRunnable(() -> LOG.info("Running YDB bootstrap with order {}", bootstrap.getOrder()))
                .then(bootstrap.run(session, bootstrapProgressDao)
                        .doOnSuccess(v -> LOG.info("YDB bootstrap successfully finished for order {}",
                                bootstrap.getOrder())));
    }

    private Mono<Void> runMigration(YdbSession session, DbMigration migration) {
        return Mono.fromRunnable(() -> LOG.info("Running YDB migration for version {}", migration.getVersion()))
                .then(migration.run(session, schemaVersionsDao)
                        .doOnSuccess(v -> LOG.info("YDB migration successfully finished for version {}",
                                migration.getVersion())));
    }

    private Mono<Void> executeRecipeMigrations(List<DbRecipeMigration> migrations) {
        return tableClient.usingSessionMonoRetryable(session ->
                skipAppliedRecipeMigrations(session, migrations)
                        .map(m -> m.stream().sorted(Comparator.comparing(DbRecipeMigration::getVersion))
                                .collect(Collectors.toList()))
                        .doOnSuccess(m -> LOG.info("Total {} YDB recipe migrations to apply", m.size()))
                        .flatMapMany(Flux::fromIterable)
                        .concatMap(m -> runRecipeMigration(session, m))
                        .thenEmpty(Mono.empty())).retryWhen(YdbRetry.retryTimeout(1, ydbMetrics));
    }

    private Mono<Void> runRecipeMigration(YdbSession session, DbRecipeMigration migration) {
        return Mono.fromRunnable(() -> LOG.info("Running YDB recipe migration for version {}", migration.getVersion()))
                .then(migration.run(session, recipeMigrationsProgressDao)
                        .doOnSuccess(v -> LOG.info("YDB recipe migration successfully finished for version {}",
                                migration.getVersion())));
    }

}
