package ru.yandex.intranet.d.datasource.migrations.impl;

import java.time.Duration;
import java.util.List;
import java.util.Set;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

import com.google.common.util.concurrent.ThreadFactoryBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.annotation.Profile;
import org.springframework.stereotype.Component;
import reactor.core.Exceptions;

import ru.yandex.intranet.d.datasource.coordination.model.cluster.NodeInfo;
import ru.yandex.intranet.d.datasource.coordination.spring.AutoClusterManager;
import ru.yandex.intranet.d.datasource.migrations.model.DbBootstrap;
import ru.yandex.intranet.d.datasource.migrations.model.DbMigration;
import ru.yandex.intranet.d.datasource.migrations.model.MigrationsStatus;

/**
 * Migrations scheduler.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
@Profile({"dev", "testing", "production", "load-testing"})
public class MigrationsScheduler {

    private static final Logger LOG = LoggerFactory.getLogger(MigrationsScheduler.class);
    private static final Duration BLOCK_TIMEOUT = Duration.ofSeconds(20);

    private final AutoClusterManager clusterManager;
    private final MigrationsRunner migrationsRunner;
    private final ScheduledExecutorService scheduler;
    private final AtomicReference<MigrationsStatus> status = new AtomicReference<>(MigrationsStatus.UNDEFINED);
    private final AtomicBoolean hasError = new AtomicBoolean(false);
    private final AtomicBoolean wasStarted = new AtomicBoolean(false);
    private final AtomicLong pendingActions = new AtomicLong(0L);

    public MigrationsScheduler(AutoClusterManager clusterManager, MigrationsRunner migrationsRunner) {
        this.clusterManager = clusterManager;
        this.migrationsRunner = migrationsRunner;
        ThreadFactory threadFactory = new ThreadFactoryBuilder()
                .setDaemon(true)
                .setNameFormat("migrations-scheduler-pool-%d")
                .setUncaughtExceptionHandler((t, e) ->
                        LOG.error("Uncaught exception in migrations scheduler thread " + t, e))
                .build();
        ScheduledThreadPoolExecutor scheduledThreadPoolExecutor = new ScheduledThreadPoolExecutor(2,
                threadFactory);
        scheduledThreadPoolExecutor.setRemoveOnCancelPolicy(true);
        this.scheduler = scheduledThreadPoolExecutor;
    }

    @PostConstruct
    @SuppressWarnings("FutureReturnValueIgnored")
    public void postConstruct() {
        LOG.info("Scheduling migration task...");
        scheduler.schedule(this::doMigrations, 1, TimeUnit.MILLISECONDS);
        LOG.info("Migration task scheduled");
    }

    @PreDestroy
    public void preDestroy() {
        LOG.info("Stopping migrations scheduler...");
        scheduler.shutdown();
        try {
            scheduler.awaitTermination(1, TimeUnit.SECONDS);
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        scheduler.shutdownNow();
        LOG.info("Stopped migrations scheduler");
    }

    public boolean hasError() {
        return hasError.get();
    }

    public MigrationsStatus getMigrationsStatus() {
        return status.get();
    }

    public long getPendingActionsCount() {
        return pendingActions.get();
    }

    @SuppressWarnings("FutureReturnValueIgnored")
    private void doMigrations() {
        if (!clusterManager.isRunning()) {
            if (!wasStarted.get()) {
                LOG.info("Cluster manager is not yet started, retrying later");
                status.set(MigrationsStatus.INITIALIZING);
                scheduler.schedule(this::doMigrations, 5, TimeUnit.SECONDS);
            } else {
                status.set(MigrationsStatus.STOPPING);
                LOG.info("Cluster manager is stopped, no more migration retries");
            }
            return;
        }
        if (!wasStarted.get()) {
            wasStarted.set(true);
        }
        boolean retry = true;
        try {
            Integer pendingBootstraps = migrationsRunner.countPendingBootstraps().block(BLOCK_TIMEOUT);
            Integer pendingMigrations = migrationsRunner.countPendingMigrations().block(BLOCK_TIMEOUT);
            int pendingBootstrapsCount = pendingBootstraps != null ? pendingBootstraps : 0;
            int pendingMigrationsCount = pendingMigrations != null ? pendingMigrations : 0;
            boolean isPendingBootstraps = pendingBootstrapsCount > 0;
            boolean isPendingMigrations =  pendingMigrationsCount > 0;
            if (!isPendingBootstraps && !isPendingMigrations) {
                LOG.info("No more migrations pending.");
                hasError.set(false);
                status.set(MigrationsStatus.APPLIED);
                retry = false;
                pendingActions.set(0L);
                return;
            }
            pendingActions.set((long) pendingBootstrapsCount + (long) pendingMigrationsCount);
            status.set(MigrationsStatus.PENDING);
            LOG.info("Bootstrap pending: {}, migrations pending: {}", isPendingBootstraps, isPendingMigrations);
            boolean applied = applyMigrations();
            if (applied) {
                status.set(MigrationsStatus.APPLIED);
                hasError.set(false);
                retry = false;
                LOG.info("All migrations applied");
                pendingActions.set(0L);
            } else {
                LOG.info("Not all migrations was applied");
                hasError.set(false);
            }
        } catch (Throwable e) {
            LOG.info("Failed to do a migration", e);
            hasError.set(true);
            Exceptions.throwIfJvmFatal(e);
        } finally {
            if (retry) {
                scheduler.schedule(this::doMigrations, 1, TimeUnit.MINUTES);
                LOG.info("Migrations are not yet finished, retrying later");
            } else {
                LOG.info("Migrations are finished");
            }
        }
    }

    private boolean applyMigrations() {
        Set<NodeInfo> members = clusterManager.getMembers().block(BLOCK_TIMEOUT);
        if (members == null || members.isEmpty()) {
            LOG.info("No cluster members info is available, not applying migrations");
            return false;
        }
        String currentVersion = clusterManager.getCurrentVersion();
        boolean allCurrentVersion = members.stream()
                .allMatch(nodeInfo -> currentVersion.equals(nodeInfo.getVersion()));
        if (!allCurrentVersion) {
            LOG.info("Not all cluster members has the same version as this node, not applying migrations");
            return false;
        }
        Boolean isLeader = clusterManager.isLeader().block(BLOCK_TIMEOUT);
        if (isLeader == null || !isLeader) {
            LOG.info("Not a leader, not applying migrations");
            return false;
        }
        List<DbBootstrap> pendingBootstrapsList = migrationsRunner.getPendingBootstraps().block(BLOCK_TIMEOUT);
        List<DbMigration> pendingMigrationsList = migrationsRunner.getPendingMigrations().block(BLOCK_TIMEOUT);
        List<DbBootstrap> pendingBootstraps = pendingBootstrapsList != null ? pendingBootstrapsList : List.of();
        List<DbMigration> pendingMigrations = pendingMigrationsList != null ? pendingMigrationsList : List.of();
        if (pendingBootstraps.isEmpty() && pendingMigrations.isEmpty()) {
            LOG.info("No more migrations to apply");
            return true;
        }
        LOG.info("Total {} bootstraps and {} migrations are still pending", pendingBootstraps.size(),
                pendingMigrations.size());
        long currentPendingActions = (long) pendingBootstraps.size() + (long) pendingMigrations.size();
        pendingActions.set(currentPendingActions);
        for (DbBootstrap bootstrap : pendingBootstraps) {
            Boolean isLeaderNow = clusterManager.isLeader().block(BLOCK_TIMEOUT);
            if (isLeaderNow == null || !isLeaderNow) {
                LOG.info("Not a leader, not applying migrations");
                return false;
            }
            LOG.info("Applying bootstrap {}...", bootstrap.getOrder());
            migrationsRunner.applyBootstrap(bootstrap).block(BLOCK_TIMEOUT);
            LOG.info("Applied bootstrap {}", bootstrap.getOrder());
            currentPendingActions--;
            pendingActions.set(currentPendingActions);
        }
        for (DbMigration migration : pendingMigrations) {
            Boolean isLeaderNow = clusterManager.isLeader().block(BLOCK_TIMEOUT);
            if (isLeaderNow == null || !isLeaderNow) {
                LOG.info("Not a leader, not applying migrations");
                return false;
            }
            LOG.info("Applying migration {}...", migration.getVersion());
            migrationsRunner.applyMigration(migration).block(BLOCK_TIMEOUT);
            LOG.info("Applied migration {}", migration.getVersion());
            currentPendingActions--;
            pendingActions.set(currentPendingActions);
        }
        return true;
    }

}
