package ru.yandex.intranet.d.datasource.model;

import java.util.Objects;
import java.util.function.Function;

import reactor.util.function.Tuple2;
import reactor.util.function.Tuples;

/**
 * Data with Ydb transaction Id.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 * @since 17.09.2020
 */
public final class WithTxId<T> {
    private final T data;
    private final String transactionId;

    public WithTxId(T data, String transactionId) {
        this.data = data;
        this.transactionId = transactionId;
    }

    public <O> WithTxId<O> map(Function<T, O> mapper) {
        return new WithTxId<>(mapper.apply(data), transactionId);
    }

    public T get() {
        return data;
    }

    public String getTransactionId() {
        return transactionId;
    }

    public Tuple2<T, String> asTuple() {
        return Tuples.of(data, transactionId);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        WithTxId<?> that = (WithTxId<?>) o;
        return Objects.equals(data, that.data) &&
                Objects.equals(transactionId, that.transactionId);
    }

    @Override
    public int hashCode() {
        return Objects.hash(data, transactionId);
    }

    @Override
    public String toString() {
        return "DataWithYdbTransactionId{" +
                "data=" + data +
                ", transactionId='" + transactionId + '\'' +
                '}';
    }

    public WithTxId<T> toClosed() {
        return new WithTxId<>(data, null);
    }

    public T getClosed() {
        if (transactionId != null && !transactionId.isEmpty()) {
            throw new IllegalStateException("Transaction isn't closed!");
        }
        return data;
    }
}
