package ru.yandex.intranet.d.datasource.model;

import java.time.Duration;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.TimeUnit;

/**
 * YDB execute scan query settings.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class YdbExecuteScanQuerySettings {

    private final YdbScanQueryRequestMode requestMode;
    private final Duration timeout;
    private final YdbQueryStatsCollectionMode statsMode;

    private YdbExecuteScanQuerySettings(YdbScanQueryRequestMode requestMode,
                                        Duration timeout,
                                        YdbQueryStatsCollectionMode statsMode) {
        this.requestMode = requestMode;
        this.timeout = timeout;
        this.statsMode = statsMode;
    }

    public static Builder builder() {
        return new Builder();
    }

    public Optional<YdbScanQueryRequestMode> getRequestMode() {
        return Optional.ofNullable(requestMode);
    }

    public Optional<Duration> getTimeout() {
        return Optional.ofNullable(timeout);
    }

    public Optional<YdbQueryStatsCollectionMode> getStatsMode() {
        return Optional.ofNullable(statsMode);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        YdbExecuteScanQuerySettings that = (YdbExecuteScanQuerySettings) o;
        return requestMode == that.requestMode &&
                Objects.equals(timeout, that.timeout) &&
                statsMode == that.statsMode;
    }

    @Override
    public int hashCode() {
        return Objects.hash(requestMode, timeout, statsMode);
    }

    @Override
    public String toString() {
        return "YdbExecuteScanQuerySettings{" +
                "requestMode=" + requestMode +
                ", timeout=" + timeout +
                ", statsMode=" + statsMode +
                '}';
    }

    public static final class Builder {

        private YdbScanQueryRequestMode requestMode;
        private Duration timeout;
        private YdbQueryStatsCollectionMode statsMode;

        private Builder() {
        }

        public Builder requestMode(YdbScanQueryRequestMode requestMode) {
            this.requestMode = requestMode;
            return this;
        }

        public Builder timeout(Duration timeout) {
            this.timeout = timeout;
            return this;
        }

        public Builder timeout(long duration, TimeUnit unit) {
            this.timeout = Duration.ofNanos(unit.toNanos(duration));
            return this;
        }

        public Builder statsMode(YdbQueryStatsCollectionMode statsMode) {
            this.statsMode = statsMode;
            return this;
        }

        public YdbExecuteScanQuerySettings build() {
            return new YdbExecuteScanQuerySettings(requestMode, timeout, statsMode);
        }

    }

}
